import { Injectable, Logger, BadRequestException, NotFoundException } from '@nestjs/common';
import { PrismaService } from '../../common/prisma/prisma.service';
import { NotificationService } from '../notification/notification.service';

export enum EventType {
  CONCERT = 'concert',
  SPORT = 'sport',
  CONFERENCE = 'conference',
  WEDDING = 'wedding',
  CORPORATE = 'corporate',
  FESTIVAL = 'festival',
  OTHER = 'other',
}

export enum EventStatus {
  DRAFT = 'draft',
  PUBLISHED = 'published',
  ONGOING = 'ongoing',
  COMPLETED = 'completed',
  CANCELLED = 'cancelled',
}

interface EventData {
  name: string;
  description?: string;
  type: EventType;
  venueId?: number;
  venueName: string;
  venueAddress: string;
  latitude: number;
  longitude: number;
  startDate: Date;
  endDate: Date;
  expectedAttendees?: number;
  imageUrl?: string;
  surgeMultiplier?: number;
  preBookingEnabled?: boolean;
  preBookingStartDate?: Date;
}

interface EventBooking {
  id: number;
  eventId: number;
  userId: number;
  pickupType: 'to_event' | 'from_event' | 'both';
  pickupAddress?: string;
  pickupTime?: Date;
  dropAddress?: string;
  dropTime?: Date;
  vehicleTypeId: number;
  passengers: number;
  status: string;
  estimatedFare: number;
}

@Injectable()
export class EventsService {
  private readonly logger = new Logger(EventsService.name);

  constructor(
    private prisma: PrismaService,
    private notificationService: NotificationService,
  ) {}

  // ============================================================================
  // EVENT MANAGEMENT
  // ============================================================================

  /**
   * Create a new event
   */
  async createEvent(merchantId: number, data: EventData): Promise<any> {
    const event = await this.prisma.event.create({
      data: {
        merchant_id: merchantId,
        name: data.name,
        description: data.description,
        type: data.type,
        venue_id: data.venueId,
        venue_name: data.venueName,
        venue_address: data.venueAddress,
        latitude: data.latitude,
        longitude: data.longitude,
        start_date: data.startDate,
        end_date: data.endDate,
        expected_attendees: data.expectedAttendees,
        image_url: data.imageUrl,
        surge_multiplier: data.surgeMultiplier || 1.0,
        pre_booking_enabled: data.preBookingEnabled ? 1 : 0,
        pre_booking_start_date: data.preBookingStartDate,
        status: EventStatus.DRAFT,
        created_at: new Date(),
      },
    });

    this.logger.log(`Event created: ${data.name} (#${event.id})`);
    return event;
  }

  /**
   * Update event
   */
  async updateEvent(eventId: number, merchantId: number, data: Partial<EventData>): Promise<any> {
    const event = await this.prisma.event.findFirst({
      where: { id: eventId, merchant_id: merchantId },
    });

    if (!event) {
      throw new NotFoundException('Evenement non trouve');
    }

    return this.prisma.event.update({
      where: { id: eventId },
      data: {
        ...(data.name && { name: data.name }),
        ...(data.description !== undefined && { description: data.description }),
        ...(data.type && { type: data.type }),
        ...(data.venueName && { venue_name: data.venueName }),
        ...(data.venueAddress && { venue_address: data.venueAddress }),
        ...(data.latitude && { latitude: data.latitude }),
        ...(data.longitude && { longitude: data.longitude }),
        ...(data.startDate && { start_date: data.startDate }),
        ...(data.endDate && { end_date: data.endDate }),
        ...(data.expectedAttendees !== undefined && { expected_attendees: data.expectedAttendees }),
        ...(data.imageUrl !== undefined && { image_url: data.imageUrl }),
        ...(data.surgeMultiplier !== undefined && { surge_multiplier: data.surgeMultiplier }),
        ...(data.preBookingEnabled !== undefined && { pre_booking_enabled: data.preBookingEnabled ? 1 : 0 }),
        ...(data.preBookingStartDate && { pre_booking_start_date: data.preBookingStartDate }),
        updated_at: new Date(),
      },
    });
  }

  /**
   * Publish event
   */
  async publishEvent(eventId: number, merchantId: number): Promise<any> {
    const event = await this.prisma.event.findFirst({
      where: { id: eventId, merchant_id: merchantId },
    });

    if (!event) {
      throw new NotFoundException('Evenement non trouve');
    }

    if (event.status !== EventStatus.DRAFT) {
      throw new BadRequestException('Seuls les evenements en brouillon peuvent etre publies');
    }

    return this.prisma.event.update({
      where: { id: eventId },
      data: {
        status: EventStatus.PUBLISHED,
        published_at: new Date(),
      },
    });
  }

  /**
   * Cancel event
   */
  async cancelEvent(eventId: number, merchantId: number, reason?: string): Promise<any> {
    const event = await this.prisma.event.findFirst({
      where: { id: eventId, merchant_id: merchantId },
    });

    if (!event) {
      throw new NotFoundException('Evenement non trouve');
    }

    // Cancel all pending bookings
    const pendingBookings = await this.prisma.eventBooking.findMany({
      where: { event_id: eventId, status: 'pending' },
    });

    for (const booking of pendingBookings) {
      await this.prisma.eventBooking.update({
        where: { id: booking.id },
        data: { status: 'cancelled', cancellation_reason: 'Evenement annule' },
      });

      // Notify user
      await this.notificationService.sendToUser(booking.user_id, 'user', {
        title: 'Reservation annulee',
        body: `L'evenement "${event.name}" a ete annule`,
        data: { type: 'event_cancelled', event_id: String(eventId) },
      });
    }

    return this.prisma.event.update({
      where: { id: eventId },
      data: {
        status: EventStatus.CANCELLED,
        cancellation_reason: reason,
        cancelled_at: new Date(),
      },
    });
  }

  /**
   * Get upcoming events
   */
  async getUpcomingEvents(
    merchantId: number,
    options?: { type?: EventType; page?: number; limit?: number },
  ): Promise<{ data: any[]; total: number }> {
    const page = options?.page || 1;
    const limit = options?.limit || 20;
    const skip = (page - 1) * limit;

    const where: any = {
      merchant_id: merchantId,
      status: EventStatus.PUBLISHED,
      start_date: { gte: new Date() },
    };

    if (options?.type) {
      where.type = options.type;
    }

    const [events, total] = await Promise.all([
      this.prisma.event.findMany({
        where,
        skip,
        take: limit,
        orderBy: { start_date: 'asc' },
      }),
      this.prisma.event.count({ where }),
    ]);

    return { data: events, total };
  }

  /**
   * Get event details
   */
  async getEventById(eventId: number): Promise<any> {
    const event = await this.prisma.event.findUnique({
      where: { id: eventId },
      include: {
        _count: {
          select: { bookings: true },
        },
      },
    });

    if (!event) {
      throw new NotFoundException('Evenement non trouve');
    }

    return {
      ...event,
      bookingsCount: event._count.bookings,
    };
  }

  // ============================================================================
  // EVENT BOOKING
  // ============================================================================

  /**
   * Create event booking (pre-book ride to/from event)
   */
  async createEventBooking(
    eventId: number,
    userId: number,
    data: {
      pickupType: 'to_event' | 'from_event' | 'both';
      pickupAddress?: string;
      pickupLatitude?: number;
      pickupLongitude?: number;
      pickupTime?: Date;
      dropAddress?: string;
      dropLatitude?: number;
      dropLongitude?: number;
      dropTime?: Date;
      vehicleTypeId: number;
      passengers: number;
    },
  ): Promise<any> {
    const event = await this.prisma.event.findUnique({
      where: { id: eventId },
    });

    if (!event) {
      throw new NotFoundException('Evenement non trouve');
    }

    if (event.status !== EventStatus.PUBLISHED) {
      throw new BadRequestException('Cet evenement n\'accepte pas les reservations');
    }

    if (event.pre_booking_enabled !== 1) {
      throw new BadRequestException('La pre-reservation n\'est pas activee pour cet evenement');
    }

    // Validate booking times
    if (data.pickupType === 'to_event' || data.pickupType === 'both') {
      if (!data.pickupAddress || !data.pickupTime) {
        throw new BadRequestException('Adresse et heure de depart requises');
      }
    }

    if (data.pickupType === 'from_event' || data.pickupType === 'both') {
      if (!data.dropAddress || !data.dropTime) {
        throw new BadRequestException('Adresse et heure de retour requises');
      }
    }

    // Calculate estimated fare
    const estimatedFare = await this.calculateEventBookingFare(
      event,
      data.vehicleTypeId,
      data.pickupType,
      data.pickupLatitude,
      data.pickupLongitude,
      data.dropLatitude,
      data.dropLongitude,
    );

    const booking = await this.prisma.eventBooking.create({
      data: {
        event_id: eventId,
        user_id: userId,
        merchant_id: event.merchant_id,
        pickup_type: data.pickupType,
        pickup_address: data.pickupAddress,
        pickup_latitude: data.pickupLatitude,
        pickup_longitude: data.pickupLongitude,
        pickup_time: data.pickupTime,
        drop_address: data.dropAddress || event.venue_address,
        drop_latitude: data.dropLatitude || event.latitude,
        drop_longitude: data.dropLongitude || event.longitude,
        drop_time: data.dropTime,
        vehicle_type_id: data.vehicleTypeId,
        passengers: data.passengers,
        estimated_fare: estimatedFare,
        status: 'pending',
        created_at: new Date(),
      },
    });

    this.logger.log(`Event booking created for event #${eventId} by user #${userId}`);
    return booking;
  }

  /**
   * Get user's event bookings
   */
  async getUserEventBookings(userId: number): Promise<any[]> {
    return this.prisma.eventBooking.findMany({
      where: { user_id: userId },
      include: {
        event: {
          select: {
            id: true,
            name: true,
            venue_name: true,
            start_date: true,
            image_url: true,
          },
        },
      },
      orderBy: { created_at: 'desc' },
    });
  }

  /**
   * Cancel event booking
   */
  async cancelEventBooking(bookingId: number, userId: number): Promise<void> {
    const booking = await this.prisma.eventBooking.findFirst({
      where: { id: bookingId, user_id: userId },
      include: { event: true },
    });

    if (!booking) {
      throw new NotFoundException('Reservation non trouvee');
    }

    if (booking.status !== 'pending') {
      throw new BadRequestException('Cette reservation ne peut pas etre annulee');
    }

    await this.prisma.eventBooking.update({
      where: { id: bookingId },
      data: {
        status: 'cancelled',
        cancelled_at: new Date(),
      },
    });
  }

  // ============================================================================
  // VENUE MANAGEMENT
  // ============================================================================

  /**
   * Create venue
   */
  async createVenue(
    merchantId: number,
    data: {
      name: string;
      address: string;
      latitude: number;
      longitude: number;
      capacity?: number;
      description?: string;
      imageUrl?: string;
    },
  ): Promise<any> {
    return this.prisma.venue.create({
      data: {
        merchant_id: merchantId,
        name: data.name,
        address: data.address,
        latitude: data.latitude,
        longitude: data.longitude,
        capacity: data.capacity,
        description: data.description,
        image_url: data.imageUrl,
        is_active: 1,
        created_at: new Date(),
      },
    });
  }

  /**
   * Get venues
   */
  async getVenues(merchantId: number): Promise<any[]> {
    return this.prisma.venue.findMany({
      where: { merchant_id: merchantId, is_active: 1 },
      orderBy: { name: 'asc' },
    });
  }

  // ============================================================================
  // DRIVER ALLOCATION
  // ============================================================================

  /**
   * Get event bookings for driver assignment
   */
  async getPendingEventBookings(
    merchantId: number,
    eventId?: number,
  ): Promise<any[]> {
    const where: any = {
      merchant_id: merchantId,
      status: 'pending',
    };

    if (eventId) {
      where.event_id = eventId;
    }

    return this.prisma.eventBooking.findMany({
      where,
      include: {
        event: {
          select: { name: true, venue_name: true, start_date: true },
        },
        user: {
          select: { first_name: true, last_name: true, mobile: true },
        },
      },
      orderBy: { pickup_time: 'asc' },
    });
  }

  /**
   * Assign driver to event booking
   */
  async assignDriverToEventBooking(
    bookingId: number,
    driverId: number,
  ): Promise<any> {
    const booking = await this.prisma.eventBooking.findUnique({
      where: { id: bookingId },
    });

    if (!booking) {
      throw new NotFoundException('Reservation non trouvee');
    }

    const updated = await this.prisma.eventBooking.update({
      where: { id: bookingId },
      data: {
        driver_id: driverId,
        status: 'confirmed',
        assigned_at: new Date(),
      },
    });

    // Notify user
    const driver = await this.prisma.driver.findUnique({ where: { id: driverId } });
    await this.notificationService.sendToUser(booking.user_id, 'user', {
      title: 'Chauffeur assigne',
      body: `${driver.first_name} vous accompagnera pour l'evenement`,
      data: { type: 'event_driver_assigned', booking_id: String(bookingId) },
    });

    return updated;
  }

  // ============================================================================
  // HELPERS
  // ============================================================================

  /**
   * Calculate event booking fare
   */
  private async calculateEventBookingFare(
    event: any,
    vehicleTypeId: number,
    pickupType: string,
    pickupLat?: number,
    pickupLng?: number,
    dropLat?: number,
    dropLng?: number,
  ): Promise<number> {
    const vehicleType = await this.prisma.vehicleType.findUnique({
      where: { id: vehicleTypeId },
    });

    if (!vehicleType) {
      return 0;
    }

    let totalFare = 0;
    const baseFare = Number(vehicleType.base_fare) || 500;
    const perKmRate = Number(vehicleType.per_km_rate) || 100;
    const surgeMultiplier = Number(event.surge_multiplier) || 1.0;

    // Calculate distance to event
    if ((pickupType === 'to_event' || pickupType === 'both') && pickupLat && pickupLng) {
      const distanceToEvent = this.calculateDistance(
        pickupLat,
        pickupLng,
        Number(event.latitude),
        Number(event.longitude),
      );
      totalFare += baseFare + distanceToEvent * perKmRate;
    }

    // Calculate distance from event
    if ((pickupType === 'from_event' || pickupType === 'both') && dropLat && dropLng) {
      const distanceFromEvent = this.calculateDistance(
        Number(event.latitude),
        Number(event.longitude),
        dropLat,
        dropLng,
      );
      totalFare += baseFare + distanceFromEvent * perKmRate;
    }

    // Apply surge
    totalFare = Math.round(totalFare * surgeMultiplier);

    return totalFare;
  }

  private calculateDistance(lat1: number, lng1: number, lat2: number, lng2: number): number {
    const R = 6371;
    const dLat = (lat2 - lat1) * (Math.PI / 180);
    const dLng = (lng2 - lng1) * (Math.PI / 180);
    const a =
      Math.sin(dLat / 2) * Math.sin(dLat / 2) +
      Math.cos(lat1 * (Math.PI / 180)) *
        Math.cos(lat2 * (Math.PI / 180)) *
        Math.sin(dLng / 2) *
        Math.sin(dLng / 2);
    const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
    return R * c;
  }
}
