import {
  Controller,
  Get,
  Post,
  Put,
  Body,
  Param,
  Query,
  UseGuards,
  ParseIntPipe,
} from '@nestjs/common';
import { ApiTags, ApiOperation, ApiBearerAuth } from '@nestjs/swagger';
import { EventsService, EventType } from './events.service';
import { JwtAuthGuard } from '../../common/guards/jwt-auth.guard';
import { MerchantGuard } from '../../common/guards/merchant.guard';
import { MerchantId, CurrentUser } from '../../common/decorators/merchant.decorator';

@ApiTags('Events')
@Controller('api/events')
@UseGuards(JwtAuthGuard, MerchantGuard)
@ApiBearerAuth()
export class EventsController {
  constructor(private eventsService: EventsService) {}

  // ============================================================================
  // PUBLIC ENDPOINTS (Users)
  // ============================================================================

  @Get()
  @ApiOperation({ summary: 'Get upcoming events' })
  async getUpcomingEvents(
    @MerchantId() merchantId: number,
    @Query('type') type?: EventType,
    @Query('page') page?: string,
    @Query('limit') limit?: string,
  ) {
    return this.eventsService.getUpcomingEvents(merchantId, {
      type,
      page: page ? parseInt(page) : 1,
      limit: limit ? parseInt(limit) : 20,
    });
  }

  @Get(':id')
  @ApiOperation({ summary: 'Get event details' })
  async getEventById(@Param('id', ParseIntPipe) eventId: number) {
    return this.eventsService.getEventById(eventId);
  }

  @Post(':id/book')
  @ApiOperation({ summary: 'Book ride for event' })
  async createEventBooking(
    @Param('id', ParseIntPipe) eventId: number,
    @CurrentUser() user: any,
    @Body()
    body: {
      pickupType: 'to_event' | 'from_event' | 'both';
      pickupAddress?: string;
      pickupLatitude?: number;
      pickupLongitude?: number;
      pickupTime?: string;
      dropAddress?: string;
      dropLatitude?: number;
      dropLongitude?: number;
      dropTime?: string;
      vehicleTypeId: number;
      passengers: number;
    },
  ) {
    return this.eventsService.createEventBooking(eventId, user.id, {
      pickupType: body.pickupType,
      pickupAddress: body.pickupAddress,
      pickupLatitude: body.pickupLatitude,
      pickupLongitude: body.pickupLongitude,
      pickupTime: body.pickupTime ? new Date(body.pickupTime) : undefined,
      dropAddress: body.dropAddress,
      dropLatitude: body.dropLatitude,
      dropLongitude: body.dropLongitude,
      dropTime: body.dropTime ? new Date(body.dropTime) : undefined,
      vehicleTypeId: body.vehicleTypeId,
      passengers: body.passengers,
    });
  }

  @Get('bookings/my')
  @ApiOperation({ summary: 'Get my event bookings' })
  async getMyEventBookings(@CurrentUser() user: any) {
    return this.eventsService.getUserEventBookings(user.id);
  }

  @Post('bookings/:id/cancel')
  @ApiOperation({ summary: 'Cancel event booking' })
  async cancelEventBooking(
    @Param('id', ParseIntPipe) bookingId: number,
    @CurrentUser() user: any,
  ) {
    await this.eventsService.cancelEventBooking(bookingId, user.id);
    return { message: 'Reservation annulee' };
  }

  // ============================================================================
  // MERCHANT ENDPOINTS
  // ============================================================================

  @Post('merchant/create')
  @ApiOperation({ summary: 'Create event (merchant)' })
  async createEvent(
    @MerchantId() merchantId: number,
    @Body()
    body: {
      name: string;
      description?: string;
      type: EventType;
      venueId?: number;
      venueName: string;
      venueAddress: string;
      latitude: number;
      longitude: number;
      startDate: string;
      endDate: string;
      expectedAttendees?: number;
      imageUrl?: string;
      surgeMultiplier?: number;
      preBookingEnabled?: boolean;
      preBookingStartDate?: string;
    },
  ) {
    return this.eventsService.createEvent(merchantId, {
      name: body.name,
      description: body.description,
      type: body.type,
      venueId: body.venueId,
      venueName: body.venueName,
      venueAddress: body.venueAddress,
      latitude: body.latitude,
      longitude: body.longitude,
      startDate: new Date(body.startDate),
      endDate: new Date(body.endDate),
      expectedAttendees: body.expectedAttendees,
      imageUrl: body.imageUrl,
      surgeMultiplier: body.surgeMultiplier,
      preBookingEnabled: body.preBookingEnabled,
      preBookingStartDate: body.preBookingStartDate
        ? new Date(body.preBookingStartDate)
        : undefined,
    });
  }

  @Put('merchant/:id')
  @ApiOperation({ summary: 'Update event (merchant)' })
  async updateEvent(
    @Param('id', ParseIntPipe) eventId: number,
    @MerchantId() merchantId: number,
    @Body()
    body: {
      name?: string;
      description?: string;
      type?: EventType;
      venueName?: string;
      venueAddress?: string;
      latitude?: number;
      longitude?: number;
      startDate?: string;
      endDate?: string;
      expectedAttendees?: number;
      imageUrl?: string;
      surgeMultiplier?: number;
      preBookingEnabled?: boolean;
      preBookingStartDate?: string;
    },
  ) {
    return this.eventsService.updateEvent(eventId, merchantId, {
      ...body,
      startDate: body.startDate ? new Date(body.startDate) : undefined,
      endDate: body.endDate ? new Date(body.endDate) : undefined,
      preBookingStartDate: body.preBookingStartDate
        ? new Date(body.preBookingStartDate)
        : undefined,
    });
  }

  @Post('merchant/:id/publish')
  @ApiOperation({ summary: 'Publish event (merchant)' })
  async publishEvent(
    @Param('id', ParseIntPipe) eventId: number,
    @MerchantId() merchantId: number,
  ) {
    return this.eventsService.publishEvent(eventId, merchantId);
  }

  @Post('merchant/:id/cancel')
  @ApiOperation({ summary: 'Cancel event (merchant)' })
  async cancelEvent(
    @Param('id', ParseIntPipe) eventId: number,
    @MerchantId() merchantId: number,
    @Body() body?: { reason?: string },
  ) {
    return this.eventsService.cancelEvent(eventId, merchantId, body?.reason);
  }

  // ============================================================================
  // VENUE MANAGEMENT
  // ============================================================================

  @Get('venues/list')
  @ApiOperation({ summary: 'Get venues' })
  async getVenues(@MerchantId() merchantId: number) {
    return this.eventsService.getVenues(merchantId);
  }

  @Post('venues/create')
  @ApiOperation({ summary: 'Create venue' })
  async createVenue(
    @MerchantId() merchantId: number,
    @Body()
    body: {
      name: string;
      address: string;
      latitude: number;
      longitude: number;
      capacity?: number;
      description?: string;
      imageUrl?: string;
    },
  ) {
    return this.eventsService.createVenue(merchantId, body);
  }

  // ============================================================================
  // ADMIN/DISPATCHER ENDPOINTS
  // ============================================================================

  @Get('admin/bookings/pending')
  @ApiOperation({ summary: 'Get pending event bookings (admin)' })
  async getPendingBookings(
    @MerchantId() merchantId: number,
    @Query('eventId') eventId?: string,
  ) {
    return this.eventsService.getPendingEventBookings(
      merchantId,
      eventId ? parseInt(eventId) : undefined,
    );
  }

  @Post('admin/bookings/:id/assign')
  @ApiOperation({ summary: 'Assign driver to event booking (admin)' })
  async assignDriver(
    @Param('id', ParseIntPipe) bookingId: number,
    @Body() body: { driverId: number },
  ) {
    return this.eventsService.assignDriverToEventBooking(bookingId, body.driverId);
  }
}
