import { Injectable, Logger } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';

@Injectable()
export class EmailTemplateService {
  private readonly logger = new Logger(EmailTemplateService.name);
  private appName: string;
  private appUrl: string;
  private supportEmail: string;
  private primaryColor: string;

  constructor(private configService: ConfigService) {
    this.appName = this.configService.get<string>('APP_NAME') || 'MonkAPI';
    this.appUrl = this.configService.get<string>('APP_URL') || 'https://monkapi.com';
    this.supportEmail = this.configService.get<string>('SUPPORT_EMAIL') || 'support@monkapi.com';
    this.primaryColor = this.configService.get<string>('EMAIL_PRIMARY_COLOR') || '#4F46E5';
  }

  /**
   * Render HTML template
   */
  async render(templateName: string, data: Record<string, any>): Promise<string> {
    const template = this.getTemplate(templateName);
    return this.compileTemplate(template, data);
  }

  /**
   * Render plain text version
   */
  async renderText(templateName: string, data: Record<string, any>): Promise<string> {
    const template = this.getTextTemplate(templateName);
    return this.compileTemplate(template, data);
  }

  /**
   * Compile template with data
   */
  private compileTemplate(template: string, data: Record<string, any>): string {
    let compiled = template;

    // Add default data
    const fullData = {
      ...data,
      appName: this.appName,
      appUrl: this.appUrl,
      supportEmail: this.supportEmail,
      primaryColor: this.primaryColor,
      currentYear: new Date().getFullYear(),
    };

    // Replace variables {{variable}}
    Object.keys(fullData).forEach(key => {
      const regex = new RegExp(`{{\\s*${key}\\s*}}`, 'g');
      const value = fullData[key];
      compiled = compiled.replace(regex, value !== undefined ? String(value) : '');
    });

    // Handle date formatting {{date:fieldName}}
    compiled = compiled.replace(/{{date:(\w+)}}/g, (match, fieldName) => {
      const dateValue = fullData[fieldName];
      if (dateValue instanceof Date) {
        return dateValue.toLocaleDateString('fr-FR', {
          day: '2-digit',
          month: 'long',
          year: 'numeric',
        });
      }
      return dateValue || '';
    });

    // Handle currency formatting {{currency:amount:currencyCode}}
    compiled = compiled.replace(/{{currency:(\w+):(\w+)}}/g, (match, amountField, currencyField) => {
      const amount = fullData[amountField];
      const currency = fullData[currencyField] || 'XOF';
      if (amount !== undefined) {
        return new Intl.NumberFormat('fr-FR', {
          style: 'currency',
          currency,
        }).format(amount);
      }
      return '';
    });

    return compiled;
  }

  /**
   * Get HTML template
   */
  private getTemplate(name: string): string {
    const templates: Record<string, string> = {
      'welcome': this.getWelcomeTemplate(),
      'otp': this.getOtpTemplate(),
      'password-reset': this.getPasswordResetTemplate(),
      'booking-confirmation': this.getBookingConfirmationTemplate(),
      'booking-completed': this.getBookingCompletedTemplate(),
      'driver-approved': this.getDriverApprovedTemplate(),
      'driver-rejected': this.getDriverRejectedTemplate(),
      'payment-confirmation': this.getPaymentConfirmationTemplate(),
      'withdrawal-processed': this.getWithdrawalProcessedTemplate(),
      'weekly-earnings': this.getWeeklyEarningsTemplate(),
      'referral-bonus': this.getReferralBonusTemplate(),
      'document-expiry': this.getDocumentExpiryTemplate(),
      'corporate-invoice': this.getCorporateInvoiceTemplate(),
    };

    return templates[name] || this.getDefaultTemplate();
  }

  /**
   * Get plain text template
   */
  private getTextTemplate(name: string): string {
    const templates: Record<string, string> = {
      'welcome': `Bienvenue {{name}}!\n\nMerci de rejoindre {{appName}}.`,
      'otp': `Votre code de verification: {{otp}}\n\nCe code expire dans {{expiresIn}} minutes.`,
      'password-reset': `Bonjour {{name}},\n\nCliquez sur ce lien pour reinitialiser votre mot de passe: {{resetLink}}\n\nCe lien expire dans {{expiresIn}} minutes.`,
      'booking-confirmation': `Votre course #{{bookingNumber}} est confirmee.\n\nDepart: {{pickupAddress}}\nArrivee: {{dropoffAddress}}\nTarif estime: {{currency:estimatedFare:currency}}`,
      'booking-completed': `Course #{{bookingNumber}} terminee.\n\nMontant: {{currency:fare:currency}}\nMerci d'avoir choisi {{appName}}!`,
    };

    return templates[name] || 'Message de {{appName}}';
  }

  // ============================================================================
  // BASE LAYOUT
  // ============================================================================

  private wrapInLayout(content: string): string {
    return `
<!DOCTYPE html>
<html lang="fr">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>{{subject}}</title>
  <style>
    body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Helvetica, Arial, sans-serif; line-height: 1.6; color: #333; margin: 0; padding: 0; background-color: #f5f5f5; }
    .container { max-width: 600px; margin: 0 auto; background-color: #ffffff; }
    .header { background-color: {{primaryColor}}; padding: 30px 20px; text-align: center; }
    .header h1 { color: #ffffff; margin: 0; font-size: 28px; }
    .content { padding: 40px 30px; }
    .footer { background-color: #f9fafb; padding: 20px 30px; text-align: center; font-size: 12px; color: #6b7280; }
    .button { display: inline-block; padding: 14px 32px; background-color: {{primaryColor}}; color: #ffffff !important; text-decoration: none; border-radius: 8px; font-weight: 600; margin: 20px 0; }
    .button:hover { opacity: 0.9; }
    .info-box { background-color: #f3f4f6; border-radius: 8px; padding: 20px; margin: 20px 0; }
    .highlight { color: {{primaryColor}}; font-weight: 600; }
    .divider { border-top: 1px solid #e5e7eb; margin: 30px 0; }
    h2 { color: #111827; margin-top: 0; }
    p { margin: 16px 0; color: #4b5563; }
    .code { font-size: 32px; font-weight: bold; letter-spacing: 8px; color: {{primaryColor}}; background-color: #f3f4f6; padding: 20px; border-radius: 8px; text-align: center; }
    table { width: 100%; border-collapse: collapse; }
    table td { padding: 12px 0; border-bottom: 1px solid #e5e7eb; }
    table td:last-child { text-align: right; }
  </style>
</head>
<body>
  <div class="container">
    <div class="header">
      <h1>{{appName}}</h1>
    </div>
    <div class="content">
      ${content}
    </div>
    <div class="footer">
      <p>&copy; {{currentYear}} {{appName}}. Tous droits reserves.</p>
      <p>Si vous avez des questions, contactez-nous a <a href="mailto:{{supportEmail}}">{{supportEmail}}</a></p>
    </div>
  </div>
</body>
</html>`;
  }

  // ============================================================================
  // TEMPLATES
  // ============================================================================

  private getDefaultTemplate(): string {
    return this.wrapInLayout(`
      <h2>{{subject}}</h2>
      <p>{{message}}</p>
    `);
  }

  private getWelcomeTemplate(): string {
    return this.wrapInLayout(`
      <h2>Bienvenue {{name}}! 🎉</h2>
      <p>Nous sommes ravis de vous accueillir sur <span class="highlight">{{appName}}</span>.</p>
      <p>Votre compte a ete cree avec succes. Vous pouvez maintenant profiter de tous nos services.</p>
      <div class="info-box">
        <p><strong>Que pouvez-vous faire maintenant?</strong></p>
        <ul>
          <li>Commander une course en quelques clics</li>
          <li>Suivre votre chauffeur en temps reel</li>
          <li>Payer facilement avec plusieurs options</li>
        </ul>
      </div>
      <a href="{{appUrl}}" class="button">Commencer</a>
      <p>A bientot sur la route!</p>
    `);
  }

  private getOtpTemplate(): string {
    return this.wrapInLayout(`
      <h2>Votre code de verification</h2>
      <p>Bonjour {{name}},</p>
      <p>Utilisez le code ci-dessous pour verifier votre compte:</p>
      <div class="code">{{otp}}</div>
      <p style="text-align: center; color: #6b7280; font-size: 14px;">Ce code expire dans <strong>{{expiresIn}} minutes</strong></p>
      <div class="divider"></div>
      <p style="font-size: 14px; color: #6b7280;">Si vous n'avez pas demande ce code, vous pouvez ignorer cet email.</p>
    `);
  }

  private getPasswordResetTemplate(): string {
    return this.wrapInLayout(`
      <h2>Reinitialisation de mot de passe</h2>
      <p>Bonjour {{name}},</p>
      <p>Vous avez demande la reinitialisation de votre mot de passe. Cliquez sur le bouton ci-dessous pour creer un nouveau mot de passe:</p>
      <div style="text-align: center;">
        <a href="{{resetLink}}" class="button">Reinitialiser mon mot de passe</a>
      </div>
      <p style="text-align: center; color: #6b7280; font-size: 14px;">Ce lien expire dans <strong>{{expiresIn}} minutes</strong></p>
      <div class="divider"></div>
      <p style="font-size: 14px; color: #6b7280;">Si vous n'avez pas demande cette reinitialisation, ignorez cet email. Votre mot de passe restera inchange.</p>
    `);
  }

  private getBookingConfirmationTemplate(): string {
    return this.wrapInLayout(`
      <h2>Course confirmee! ✅</h2>
      <p>Bonjour {{name}},</p>
      <p>Votre course <span class="highlight">#{{bookingNumber}}</span> a ete confirmee.</p>
      <div class="info-box">
        <table>
          <tr>
            <td><strong>Depart</strong></td>
            <td>{{pickupAddress}}</td>
          </tr>
          <tr>
            <td><strong>Arrivee</strong></td>
            <td>{{dropoffAddress}}</td>
          </tr>
          <tr>
            <td><strong>Tarif estime</strong></td>
            <td class="highlight">{{currency:estimatedFare:currency}}</td>
          </tr>
        </table>
      </div>
      <p>Vous recevrez une notification lorsqu'un chauffeur acceptera votre course.</p>
    `);
  }

  private getBookingCompletedTemplate(): string {
    return this.wrapInLayout(`
      <h2>Merci pour votre course! 🚗</h2>
      <p>Bonjour {{name}},</p>
      <p>Votre course <span class="highlight">#{{bookingNumber}}</span> est terminee.</p>
      <div class="info-box">
        <h3 style="margin-top: 0;">Details de la course</h3>
        <table>
          <tr>
            <td>Depart</td>
            <td>{{pickupAddress}}</td>
          </tr>
          <tr>
            <td>Arrivee</td>
            <td>{{dropoffAddress}}</td>
          </tr>
          <tr>
            <td>Distance</td>
            <td>{{distance}} km</td>
          </tr>
          <tr>
            <td>Duree</td>
            <td>{{duration}} min</td>
          </tr>
          <tr>
            <td>Chauffeur</td>
            <td>{{driverName}}</td>
          </tr>
          <tr>
            <td>Vehicule</td>
            <td>{{vehicleInfo}}</td>
          </tr>
          <tr>
            <td>Paiement</td>
            <td>{{paymentMethod}}</td>
          </tr>
          <tr>
            <td><strong>Total</strong></td>
            <td class="highlight" style="font-size: 18px;">{{currency:fare:currency}}</td>
          </tr>
        </table>
      </div>
      <p>N'hesitez pas a noter votre chauffeur dans l'application.</p>
      <p>Merci d'avoir choisi {{appName}}!</p>
    `);
  }

  private getDriverApprovedTemplate(): string {
    return this.wrapInLayout(`
      <h2>Felicitations {{name}}! 🎊</h2>
      <p>Votre compte chauffeur a ete <span class="highlight">approuve</span>!</p>
      <p>Vous pouvez maintenant vous connecter a l'application chauffeur et commencer a accepter des courses.</p>
      <div class="info-box">
        <p><strong>Prochaines etapes:</strong></p>
        <ol>
          <li>Connectez-vous a l'application chauffeur</li>
          <li>Activez votre statut "En ligne"</li>
          <li>Attendez les demandes de course</li>
          <li>Gagnez de l'argent!</li>
        </ol>
      </div>
      <a href="{{appUrl}}/driver" class="button">Ouvrir l'application</a>
      <p>Bonne route!</p>
    `);
  }

  private getDriverRejectedTemplate(): string {
    return this.wrapInLayout(`
      <h2>Mise a jour de votre demande</h2>
      <p>Bonjour {{name}},</p>
      <p>Nous avons examine votre demande d'inscription en tant que chauffeur, mais malheureusement nous ne pouvons pas l'approuver pour le moment.</p>
      <div class="info-box">
        <p><strong>Raison:</strong></p>
        <p>{{reason}}</p>
      </div>
      <p>Vous pouvez soumettre une nouvelle demande apres avoir resolu ce probleme.</p>
      <p>Si vous pensez qu'il s'agit d'une erreur, n'hesitez pas a nous contacter.</p>
    `);
  }

  private getPaymentConfirmationTemplate(): string {
    return this.wrapInLayout(`
      <h2>Paiement confirme! ✓</h2>
      <p>Bonjour {{name}},</p>
      <p>Votre paiement a ete traite avec succes.</p>
      <div class="info-box">
        <table>
          <tr>
            <td>Reference</td>
            <td>{{reference}}</td>
          </tr>
          <tr>
            <td>Description</td>
            <td>{{description}}</td>
          </tr>
          <tr>
            <td>Methode</td>
            <td>{{paymentMethod}}</td>
          </tr>
          <tr>
            <td><strong>Montant</strong></td>
            <td class="highlight" style="font-size: 18px;">{{currency:amount:currency}}</td>
          </tr>
        </table>
      </div>
      <p>Merci pour votre confiance!</p>
    `);
  }

  private getWithdrawalProcessedTemplate(): string {
    return this.wrapInLayout(`
      <h2>Demande de retrait {{status}}</h2>
      <p>Bonjour {{name}},</p>
      <p>Votre demande de retrait a ete <span class="highlight">{{status}}</span>.</p>
      <div class="info-box">
        <table>
          <tr>
            <td>Reference</td>
            <td>{{reference}}</td>
          </tr>
          <tr>
            <td>Methode</td>
            <td>{{method}}</td>
          </tr>
          <tr>
            <td><strong>Montant</strong></td>
            <td class="highlight">{{currency:amount:currency}}</td>
          </tr>
        </table>
      </div>
      {{#if failureReason}}
      <div class="info-box" style="background-color: #fef2f2;">
        <p><strong>Raison de l'echec:</strong></p>
        <p>{{failureReason}}</p>
      </div>
      {{/if}}
    `);
  }

  private getWeeklyEarningsTemplate(): string {
    return this.wrapInLayout(`
      <h2>Votre resume hebdomadaire 📊</h2>
      <p>Bonjour {{name}},</p>
      <p>Voici vos resultats pour la semaine du {{date:weekStart}} au {{date:weekEnd}}.</p>
      <div class="info-box">
        <table>
          <tr>
            <td>Courses effectuees</td>
            <td class="highlight">{{totalTrips}}</td>
          </tr>
          <tr>
            <td>Gains</td>
            <td class="highlight">{{currency:totalEarnings:currency}}</td>
          </tr>
          <tr>
            <td>Pourboires</td>
            <td>{{currency:totalTips:currency}}</td>
          </tr>
          <tr>
            <td>Meilleur jour</td>
            <td>{{topDay}}</td>
          </tr>
          <tr>
            <td>Note moyenne</td>
            <td>⭐ {{averageRating}}/5</td>
          </tr>
        </table>
      </div>
      <p>Continuez comme ca! 💪</p>
    `);
  }

  private getReferralBonusTemplate(): string {
    return this.wrapInLayout(`
      <h2>Vous avez gagne un bonus! 🎁</h2>
      <p>Bonjour {{name}},</p>
      <p>Grace a votre parrainage de <span class="highlight">{{referredName}}</span>, vous avez gagne:</p>
      <div style="text-align: center; padding: 30px;">
        <span style="font-size: 48px; font-weight: bold; color: {{primaryColor}};">{{currency:bonusAmount:currency}}</span>
      </div>
      <p>Ce montant a ete credite sur votre portefeuille.</p>
      <p>Continuez a parrainer vos amis pour gagner encore plus!</p>
    `);
  }

  private getDocumentExpiryTemplate(): string {
    return this.wrapInLayout(`
      <h2>Rappel: Document expirant bientot ⚠️</h2>
      <p>Bonjour {{name}},</p>
      <p>Votre <span class="highlight">{{documentType}}</span> expire le <strong>{{date:expiryDate}}</strong> (dans {{daysRemaining}} jours).</p>
      <div class="info-box">
        <p>Pour continuer a utiliser l'application chauffeur, veuillez mettre a jour ce document avant son expiration.</p>
      </div>
      <a href="{{appUrl}}/driver/documents" class="button">Mettre a jour mes documents</a>
      <p>Merci de votre cooperation!</p>
    `);
  }

  private getCorporateInvoiceTemplate(): string {
    return this.wrapInLayout(`
      <h2>Facture {{invoiceNumber}}</h2>
      <p>Bonjour,</p>
      <p>Veuillez trouver ci-joint la facture pour la periode: <strong>{{period}}</strong></p>
      <div class="info-box">
        <table>
          <tr>
            <td>Entreprise</td>
            <td>{{companyName}}</td>
          </tr>
          <tr>
            <td>Numero de facture</td>
            <td>{{invoiceNumber}}</td>
          </tr>
          <tr>
            <td>Nombre de courses</td>
            <td>{{totalTrips}}</td>
          </tr>
          <tr>
            <td>Date d'echeance</td>
            <td>{{date:dueDate}}</td>
          </tr>
          <tr>
            <td><strong>Montant total</strong></td>
            <td class="highlight" style="font-size: 18px;">{{currency:totalAmount:currency}}</td>
          </tr>
        </table>
      </div>
      <p>Merci pour votre confiance.</p>
    `);
  }
}
