import {
  Controller,
  Get,
  Put,
  Post,
  Body,
  UseGuards,
  Query,
} from '@nestjs/common';
import { ApiTags, ApiOperation, ApiBearerAuth } from '@nestjs/swagger';

import { DriverService } from './driver.service';
import { JwtAuthGuard } from '../../common/guards/jwt-auth.guard';
import { MerchantGuard } from '../../common/guards/merchant.guard';
import { CurrentUser, MerchantId } from '../../common/decorators/merchant.decorator';

import { UpdateDriverProfileDto } from './dto/update-profile.dto';
import { UpdateLocationDto } from './dto/update-location.dto';
import { UpdateStatusDto } from './dto/update-status.dto';

@ApiTags('drivers')
@Controller('driver')
@UseGuards(JwtAuthGuard, MerchantGuard)
@ApiBearerAuth()
export class DriverController {
  constructor(private readonly driverService: DriverService) {}

  @Get('profile')
  @ApiOperation({ summary: 'Get driver profile' })
  async getProfile(@CurrentUser() user: any) {
    return this.driverService.getProfile(user.id);
  }

  @Put('profile')
  @ApiOperation({ summary: 'Update driver profile' })
  async updateProfile(
    @CurrentUser() user: any,
    @Body() dto: UpdateDriverProfileDto,
  ) {
    return this.driverService.updateProfile(user.id, dto);
  }

  @Post('location')
  @ApiOperation({ summary: 'Update driver location' })
  async updateLocation(
    @CurrentUser() user: any,
    @Body() dto: UpdateLocationDto,
  ) {
    return this.driverService.updateLocation(user.id, dto);
  }

  @Post('status')
  @ApiOperation({ summary: 'Update driver online/offline status' })
  async updateStatus(
    @CurrentUser() user: any,
    @Body() dto: UpdateStatusDto,
  ) {
    return this.driverService.updateStatus(user.id, dto);
  }

  @Get('bookings')
  @ApiOperation({ summary: 'Get driver booking history' })
  async getBookings(
    @CurrentUser() user: any,
    @Query('page') page: number = 1,
    @Query('limit') limit: number = 20,
  ) {
    return this.driverService.getBookings(user.id, page, limit);
  }

  @Get('earnings')
  @ApiOperation({ summary: 'Get driver earnings' })
  async getEarnings(
    @CurrentUser() user: any,
    @Query('start_date') startDate?: string,
    @Query('end_date') endDate?: string,
  ) {
    return this.driverService.getEarnings(user.id, startDate, endDate);
  }

  @Get('wallet')
  @ApiOperation({ summary: 'Get driver wallet balance and transactions' })
  async getWallet(
    @CurrentUser() user: any,
    @Query('page') page: number = 1,
    @Query('limit') limit: number = 20,
  ) {
    return this.driverService.getWallet(user.id, page, limit);
  }

  @Get('vehicles')
  @ApiOperation({ summary: 'Get driver vehicles' })
  async getVehicles(@CurrentUser() user: any) {
    return this.driverService.getVehicles(user.id);
  }
}
