import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../common/prisma/prisma.service';
import { WebsocketService } from '../websocket/websocket.service';
import { QueueService } from '../queue/queue.service';

interface DriverCandidate {
  id: number;
  first_name: string;
  last_name: string;
  latitude: number;
  longitude: number;
  rating: number;
  total_rides: number;
  vehicle_type_id: number;
  distance: number; // km
  score: number;
}

interface AssignmentResult {
  success: boolean;
  driverId?: number;
  reason?: string;
}

@Injectable()
export class DriverAssignmentService {
  private readonly logger = new Logger(DriverAssignmentService.name);

  // Configuration
  private readonly MAX_SEARCH_RADIUS_KM = 10; // Max search radius
  private readonly INITIAL_SEARCH_RADIUS_KM = 3; // Initial search radius
  private readonly RADIUS_INCREMENT_KM = 2; // Increment per attempt
  private readonly MAX_DRIVERS_TO_NOTIFY = 5; // Max drivers to notify at once
  private readonly DRIVER_RESPONSE_TIMEOUT_MS = 30000; // 30 seconds

  // Scoring weights
  private readonly WEIGHT_DISTANCE = 0.4; // 40%
  private readonly WEIGHT_RATING = 0.3; // 30%
  private readonly WEIGHT_EXPERIENCE = 0.2; // 20%
  private readonly WEIGHT_ACCEPTANCE_RATE = 0.1; // 10%

  constructor(
    private prisma: PrismaService,
    private websocketService: WebsocketService,
    private queueService: QueueService,
  ) {}

  /**
   * Main assignment method - finds and assigns best driver for booking
   */
  async assignDriverToBooking(
    bookingId: number,
    merchantId: number,
    pickupLat: number,
    pickupLng: number,
    vehicleTypeId?: number,
  ): Promise<AssignmentResult> {
    this.logger.log(`Starting driver assignment for booking #${bookingId}`);

    // Get booking details
    const booking = await this.prisma.booking.findUnique({
      where: { id: bookingId },
    });

    if (!booking || booking.booking_status !== 'pending') {
      return { success: false, reason: 'Booking not available for assignment' };
    }

    // Find nearby drivers with expanding radius
    let radius = this.INITIAL_SEARCH_RADIUS_KM;
    let candidates: DriverCandidate[] = [];

    while (radius <= this.MAX_SEARCH_RADIUS_KM && candidates.length === 0) {
      candidates = await this.findNearbyDrivers(
        merchantId,
        pickupLat,
        pickupLng,
        radius,
        vehicleTypeId,
      );

      if (candidates.length === 0) {
        radius += this.RADIUS_INCREMENT_KM;
        this.logger.log(`No drivers found, expanding radius to ${radius}km`);
      }
    }

    if (candidates.length === 0) {
      this.logger.warn(`No available drivers for booking #${bookingId}`);
      return { success: false, reason: 'No drivers available in your area' };
    }

    // Score and rank drivers
    const rankedDrivers = this.rankDrivers(candidates);
    this.logger.log(`Found ${rankedDrivers.length} candidate drivers`);

    // Try to assign to top drivers
    const driversToNotify = rankedDrivers.slice(0, this.MAX_DRIVERS_TO_NOTIFY);

    // Broadcast booking request to selected drivers
    for (const driver of driversToNotify) {
      await this.notifyDriverOfBookingRequest(driver.id, booking);
    }

    // Update booking with assignment attempt
    await this.prisma.booking.update({
      where: { id: bookingId },
      data: {
        driver_assignment_attempts: { increment: 1 },
      },
    });

    // Schedule timeout check
    await this.queueService.addBookingJob('check-timeout', {
      bookingId,
      timeoutAt: Date.now() + this.DRIVER_RESPONSE_TIMEOUT_MS,
    }, {
      delay: this.DRIVER_RESPONSE_TIMEOUT_MS,
    });

    return { success: true };
  }

  /**
   * Find nearby available drivers using Haversine formula
   */
  async findNearbyDrivers(
    merchantId: number,
    latitude: number,
    longitude: number,
    radiusKm: number,
    vehicleTypeId?: number,
  ): Promise<DriverCandidate[]> {
    // Using raw SQL with Haversine formula for accurate distance calculation
    const drivers = await this.prisma.$queryRaw<DriverCandidate[]>`
      SELECT
        d.id,
        d.first_name,
        d.last_name,
        d.latitude,
        d.longitude,
        d.rating,
        d.total_rides,
        d.vehicle_type_id,
        d.acceptance_rate,
        (
          6371 * acos(
            cos(radians(${latitude}))
            * cos(radians(d.latitude))
            * cos(radians(d.longitude) - radians(${longitude}))
            + sin(radians(${latitude}))
            * sin(radians(d.latitude))
          )
        ) AS distance
      FROM drivers d
      WHERE d.merchant_id = ${merchantId}
        AND d.driver_status = 1
        AND d.is_online = 1
        AND d.free_busy = 2
        AND d.latitude IS NOT NULL
        AND d.longitude IS NOT NULL
        ${vehicleTypeId ? this.prisma.$queryRaw`AND d.vehicle_type_id = ${vehicleTypeId}` : this.prisma.$queryRaw``}
      HAVING distance <= ${radiusKm}
      ORDER BY distance ASC
      LIMIT 20
    `;

    return drivers;
  }

  /**
   * Calculate distance between two points using Haversine formula
   */
  calculateDistance(
    lat1: number,
    lng1: number,
    lat2: number,
    lng2: number,
  ): number {
    const R = 6371; // Earth's radius in km
    const dLat = this.toRad(lat2 - lat1);
    const dLng = this.toRad(lng2 - lng1);

    const a =
      Math.sin(dLat / 2) * Math.sin(dLat / 2) +
      Math.cos(this.toRad(lat1)) *
        Math.cos(this.toRad(lat2)) *
        Math.sin(dLng / 2) *
        Math.sin(dLng / 2);

    const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
    return R * c;
  }

  private toRad(deg: number): number {
    return deg * (Math.PI / 180);
  }

  /**
   * Rank drivers based on multiple factors
   */
  rankDrivers(drivers: DriverCandidate[]): DriverCandidate[] {
    if (drivers.length === 0) return [];

    // Get max values for normalization
    const maxDistance = Math.max(...drivers.map((d) => d.distance));
    const maxRating = 5;
    const maxRides = Math.max(...drivers.map((d) => d.total_rides || 1));

    // Calculate scores
    const scoredDrivers = drivers.map((driver) => {
      // Distance score (inverse - closer is better)
      const distanceScore = maxDistance > 0
        ? 1 - (driver.distance / maxDistance)
        : 1;

      // Rating score (higher is better)
      const ratingScore = (driver.rating || 4) / maxRating;

      // Experience score (more rides is better)
      const experienceScore = (driver.total_rides || 0) / maxRides;

      // Acceptance rate score
      const acceptanceScore = (driver['acceptance_rate'] || 80) / 100;

      // Weighted total score
      const score =
        distanceScore * this.WEIGHT_DISTANCE +
        ratingScore * this.WEIGHT_RATING +
        experienceScore * this.WEIGHT_EXPERIENCE +
        acceptanceScore * this.WEIGHT_ACCEPTANCE_RATE;

      return { ...driver, score };
    });

    // Sort by score descending
    return scoredDrivers.sort((a, b) => b.score - a.score);
  }

  /**
   * Notify driver of new booking request via WebSocket
   */
  async notifyDriverOfBookingRequest(driverId: number, booking: any): Promise<void> {
    this.logger.log(`Notifying driver #${driverId} of booking #${booking.id}`);

    // Get user info
    const user = await this.prisma.user.findUnique({
      where: { id: booking.user_id },
      select: { first_name: true, last_name: true, profile_picture: true, rating: true },
    });

    // Send via WebSocket
    this.websocketService.notifyNewBookingRequest(driverId, {
      booking_id: booking.id,
      merchant_booking_id: booking.merchant_booking_id,
      pickup_address: booking.pickup_address,
      pickup_latitude: booking.pickup_latitude,
      pickup_longitude: booking.pickup_longitude,
      drop_address: booking.drop_address,
      drop_latitude: booking.drop_latitude,
      drop_longitude: booking.drop_longitude,
      estimated_fare: booking.estimated_fare,
      estimated_distance: booking.travel_distance,
      estimated_time: booking.travel_time,
      payment_type: booking.payment_type,
      user: {
        name: `${user?.first_name || ''} ${user?.last_name || ''}`.trim(),
        profile_picture: user?.profile_picture,
        rating: user?.rating,
      },
      timeout_seconds: this.DRIVER_RESPONSE_TIMEOUT_MS / 1000,
    });

    // Also send push notification
    await this.queueService.addNotificationJob('send-push', {
      userId: driverId,
      userType: 'driver',
      title: 'Nouvelle course!',
      body: `Course vers ${booking.drop_address}`,
      data: {
        type: 'new_booking_request',
        booking_id: String(booking.id),
      },
    });
  }

  /**
   * Handle driver acceptance of booking
   */
  async handleDriverAcceptance(
    bookingId: number,
    driverId: number,
  ): Promise<{ success: boolean; message: string }> {
    // Check if booking is still available
    const booking = await this.prisma.booking.findUnique({
      where: { id: bookingId },
    });

    if (!booking) {
      return { success: false, message: 'Booking not found' };
    }

    if (booking.booking_status !== 'pending') {
      return { success: false, message: 'Booking already assigned' };
    }

    // Get driver info
    const driver = await this.prisma.driver.findUnique({
      where: { id: driverId },
    });

    if (!driver || driver.is_online !== 1 || driver.free_busy !== 2) {
      return { success: false, message: 'Driver not available' };
    }

    // Assign driver to booking (atomic transaction)
    try {
      await this.prisma.$transaction([
        // Update booking
        this.prisma.booking.update({
          where: { id: bookingId, booking_status: 'pending' },
          data: {
            driver_id: driverId,
            booking_status: 'accepted',
            accepted_time: new Date(),
          },
        }),
        // Mark driver as busy
        this.prisma.driver.update({
          where: { id: driverId },
          data: { free_busy: 1 }, // Busy
        }),
        // Update driver acceptance stats
        this.prisma.driver.update({
          where: { id: driverId },
          data: {
            total_accepted: { increment: 1 },
          },
        }),
      ]);

      // Notify user that driver accepted
      this.websocketService.notifyBookingAccepted(booking.user_id, {
        booking_id: bookingId,
        driver: {
          id: driver.id,
          name: `${driver.first_name} ${driver.last_name}`,
          phone: driver.mobile,
          profile_picture: driver.profile_picture,
          rating: driver.rating,
          vehicle_number: driver.vehicle_number,
          vehicle_model: driver.vehicle_model,
          latitude: driver.latitude,
          longitude: driver.longitude,
        },
      });

      this.logger.log(`Driver #${driverId} accepted booking #${bookingId}`);
      return { success: true, message: 'Booking accepted' };
    } catch (error) {
      this.logger.error(`Failed to assign driver: ${error.message}`);
      return { success: false, message: 'Assignment failed, booking may have been taken' };
    }
  }

  /**
   * Handle driver rejection of booking
   */
  async handleDriverRejection(
    bookingId: number,
    driverId: number,
    reason?: string,
  ): Promise<void> {
    this.logger.log(`Driver #${driverId} rejected booking #${bookingId}: ${reason}`);

    // Update driver rejection stats
    await this.prisma.driver.update({
      where: { id: driverId },
      data: {
        total_rejected: { increment: 1 },
      },
    });

    // Log rejection
    await this.prisma.bookingRejection.create({
      data: {
        booking_id: bookingId,
        driver_id: driverId,
        reason: reason || 'No reason provided',
        rejected_at: new Date(),
      },
    });

    // Check if we need to find more drivers
    const booking = await this.prisma.booking.findUnique({
      where: { id: bookingId },
    });

    if (booking && booking.booking_status === 'pending') {
      // Re-run assignment to find more drivers
      await this.assignDriverToBooking(
        bookingId,
        booking.merchant_id,
        Number(booking.pickup_latitude),
        Number(booking.pickup_longitude),
        booking.vehicle_type_id,
      );
    }
  }
}
