import {
  Controller,
  Get,
  Post,
  Body,
  Param,
  Query,
  UseGuards,
  ParseIntPipe,
} from '@nestjs/common';
import { ApiTags, ApiOperation, ApiBearerAuth } from '@nestjs/swagger';
import { DocumentService, DocumentType } from './document.service';
import { JwtAuthGuard } from '../../common/guards/jwt-auth.guard';
import { MerchantGuard } from '../../common/guards/merchant.guard';
import { MerchantId, CurrentUser } from '../../common/decorators/merchant.decorator';

@ApiTags('Documents')
@Controller('api/documents')
@UseGuards(JwtAuthGuard, MerchantGuard)
@ApiBearerAuth()
export class DocumentController {
  constructor(private documentService: DocumentService) {}

  // ============================================================================
  // DRIVER ENDPOINTS
  // ============================================================================

  @Get('requirements')
  @ApiOperation({ summary: 'Get document requirements' })
  async getRequirements(@MerchantId() merchantId: number) {
    return this.documentService.getDocumentRequirements(merchantId);
  }

  @Get('my-documents')
  @ApiOperation({ summary: 'Get my documents (driver)' })
  async getMyDocuments(@CurrentUser() driver: any) {
    return this.documentService.getDriverDocuments(driver.id);
  }

  @Post('upload')
  @ApiOperation({ summary: 'Upload a document (driver)' })
  async uploadDocument(
    @CurrentUser() driver: any,
    @MerchantId() merchantId: number,
    @Body()
    body: {
      type: DocumentType;
      fileUrl: string;
      expiryDate?: string;
    },
  ) {
    return this.documentService.uploadDocument(
      driver.id,
      merchantId,
      body.type,
      body.fileUrl,
      body.expiryDate ? new Date(body.expiryDate) : undefined,
    );
  }

  // ============================================================================
  // ADMIN ENDPOINTS
  // ============================================================================

  @Get('admin/pending')
  @ApiOperation({ summary: 'Get pending documents (admin)' })
  async getPendingDocuments(
    @MerchantId() merchantId: number,
    @Query('page') page?: string,
    @Query('limit') limit?: string,
    @Query('type') type?: DocumentType,
  ) {
    return this.documentService.getPendingDocuments(merchantId, {
      page: page ? parseInt(page) : 1,
      limit: limit ? parseInt(limit) : 20,
      type,
    });
  }

  @Get('admin/driver/:driverId')
  @ApiOperation({ summary: 'Get driver documents (admin)' })
  async getDriverDocuments(@Param('driverId', ParseIntPipe) driverId: number) {
    return this.documentService.getDriverDocuments(driverId);
  }

  @Post('admin/:id/approve')
  @ApiOperation({ summary: 'Approve document (admin)' })
  async approveDocument(
    @Param('id', ParseIntPipe) documentId: number,
    @CurrentUser() admin: any,
  ) {
    return this.documentService.approveDocument(documentId, admin.id);
  }

  @Post('admin/:id/reject')
  @ApiOperation({ summary: 'Reject document (admin)' })
  async rejectDocument(
    @Param('id', ParseIntPipe) documentId: number,
    @CurrentUser() admin: any,
    @Body() body: { reason: string },
  ) {
    return this.documentService.rejectDocument(documentId, admin.id, body.reason);
  }

  @Post('admin/check-expiring')
  @ApiOperation({ summary: 'Check and notify expiring documents (admin/cron)' })
  async checkExpiringDocuments(@Body() body?: { daysAhead?: number }) {
    await this.documentService.checkExpiringDocuments(body?.daysAhead || 30);
    return { message: 'Verification terminee' };
  }

  @Post('admin/mark-expired')
  @ApiOperation({ summary: 'Mark expired documents (admin/cron)' })
  async markExpiredDocuments() {
    const count = await this.documentService.markExpiredDocuments();
    return { markedCount: count };
  }
}
