import { Injectable, BadRequestException, NotFoundException } from '@nestjs/common';
import { PrismaService } from '../../common/prisma/prisma.service';

// ============================================================================
// TYPES
// ============================================================================

export type DisputeStatus = 'open' | 'investigating' | 'pending_response' | 'resolved' | 'closed' | 'escalated';
export type DisputeType = 'fare' | 'service' | 'safety' | 'payment' | 'driver_behavior' | 'user_behavior' | 'damage' | 'lost_item' | 'other';
export type DisputeResolution = 'refund_full' | 'refund_partial' | 'credit' | 'no_action' | 'warning_driver' | 'warning_user' | 'ban_driver' | 'ban_user';

export interface Dispute {
  id: number;
  disputeNumber: string;
  bookingId: number;
  type: DisputeType;
  status: DisputeStatus;
  initiatorType: 'user' | 'driver';
  initiatorId: number;
  respondentType: 'user' | 'driver';
  respondentId: number;
  reason: string;
  description: string;
  evidence: DisputeEvidence[];
  amount?: number; // Montant contesté
  resolution?: DisputeResolution;
  resolutionAmount?: number;
  resolutionNotes?: string;
  assignedTo?: number;
  priority: 'low' | 'medium' | 'high' | 'critical';
  respondentResponse?: string;
  respondedAt?: Date;
  resolvedAt?: Date;
  closedAt?: Date;
  escalatedAt?: Date;
  merchantId: number;
  createdAt: Date;
  updatedAt: Date;
}

export interface DisputeEvidence {
  id: number;
  disputeId: number;
  uploaderId: number;
  uploaderType: 'user' | 'driver' | 'admin';
  type: 'image' | 'video' | 'audio' | 'document';
  fileUrl: string;
  description?: string;
  createdAt: Date;
}

export interface DisputeComment {
  id: number;
  disputeId: number;
  authorId: number;
  authorType: 'user' | 'driver' | 'admin' | 'system';
  authorName: string;
  message: string;
  isInternal: boolean;
  createdAt: Date;
}

export interface CreateDisputeDto {
  bookingId: number;
  type: DisputeType;
  reason: string;
  description: string;
  amount?: number;
  evidence?: { type: string; fileUrl: string; description?: string }[];
}

// ============================================================================
// SERVICE
// ============================================================================

@Injectable()
export class DisputeService {
  constructor(private readonly prisma: PrismaService) {}

  // ==========================================================================
  // DISPUTE CRUD
  // ==========================================================================

  /**
   * Créer un litige
   */
  async createDispute(
    merchantId: number,
    initiatorType: 'user' | 'driver',
    initiatorId: number,
    data: CreateDisputeDto,
  ): Promise<Dispute> {
    // Vérifier la réservation
    const booking = await this.prisma.booking.findFirst({
      where: { id: data.bookingId, merchantId },
    });

    if (!booking) {
      throw new NotFoundException('Réservation non trouvée');
    }

    // Vérifier que l'initiateur est lié à la réservation
    if (initiatorType === 'user' && booking.userId !== initiatorId) {
      throw new BadRequestException('Vous n\'êtes pas autorisé à contester cette réservation');
    }

    if (initiatorType === 'driver' && booking.driverId !== initiatorId) {
      throw new BadRequestException('Vous n\'êtes pas autorisé à contester cette réservation');
    }

    // Vérifier qu'il n'y a pas déjà un litige actif
    const existingDispute = await this.prisma.dispute.findFirst({
      where: {
        bookingId: data.bookingId,
        status: { notIn: ['resolved', 'closed'] },
      },
    });

    if (existingDispute) {
      throw new BadRequestException('Un litige est déjà en cours pour cette réservation');
    }

    // Déterminer le respondent
    const respondentType: 'user' | 'driver' = initiatorType === 'user' ? 'driver' : 'user';
    const respondentId = initiatorType === 'user' ? booking.driverId : booking.userId;

    // Générer le numéro de litige
    const disputeNumber = await this.generateDisputeNumber(merchantId);

    // Déterminer la priorité
    const priority = this.determinePriority(data.type, data.amount);

    const dispute = await this.prisma.dispute.create({
      data: {
        merchantId,
        disputeNumber,
        bookingId: data.bookingId,
        type: data.type,
        status: 'open',
        initiatorType,
        initiatorId,
        respondentType,
        respondentId: respondentId!,
        reason: data.reason,
        description: data.description,
        amount: data.amount,
        priority,
      },
    });

    // Ajouter les preuves
    if (data.evidence && data.evidence.length > 0) {
      await this.prisma.disputeEvidence.createMany({
        data: data.evidence.map((e) => ({
          disputeId: dispute.id,
          uploaderId: initiatorId,
          uploaderType: initiatorType,
          type: e.type,
          fileUrl: e.fileUrl,
          description: e.description,
        })),
      });
    }

    // Ajouter un commentaire système
    await this.addSystemComment(dispute.id, `Litige créé par ${initiatorType}`);

    return this.getDispute(dispute.id, merchantId) as Promise<Dispute>;
  }

  /**
   * Obtenir un litige
   */
  async getDispute(id: number, merchantId: number): Promise<Dispute | null> {
    const dispute = await this.prisma.dispute.findFirst({
      where: { id, merchantId },
      include: {
        evidence: true,
        booking: {
          select: {
            bookingNumber: true,
            totalFare: true,
            pickupAddress: true,
            dropoffAddress: true,
          },
        },
      },
    });

    return dispute ? this.mapDispute(dispute) : null;
  }

  /**
   * Lister les litiges
   */
  async listDisputes(
    merchantId: number,
    options: {
      page?: number;
      limit?: number;
      status?: DisputeStatus;
      type?: DisputeType;
      priority?: string;
      assignedTo?: number;
      userId?: number;
      driverId?: number;
      search?: string;
    } = {},
  ): Promise<{ disputes: Dispute[]; total: number; page: number; totalPages: number }> {
    const page = options.page || 1;
    const limit = options.limit || 20;
    const skip = (page - 1) * limit;

    const where: any = { merchantId };

    if (options.status) where.status = options.status;
    if (options.type) where.type = options.type;
    if (options.priority) where.priority = options.priority;
    if (options.assignedTo) where.assignedTo = options.assignedTo;

    if (options.userId) {
      where.OR = [
        { initiatorType: 'user', initiatorId: options.userId },
        { respondentType: 'user', respondentId: options.userId },
      ];
    }

    if (options.driverId) {
      where.OR = [
        { initiatorType: 'driver', initiatorId: options.driverId },
        { respondentType: 'driver', respondentId: options.driverId },
      ];
    }

    if (options.search) {
      where.OR = [
        { disputeNumber: { contains: options.search } },
        { reason: { contains: options.search } },
        { description: { contains: options.search } },
      ];
    }

    const [disputes, total] = await Promise.all([
      this.prisma.dispute.findMany({
        where,
        include: {
          evidence: true,
          booking: {
            select: { bookingNumber: true },
          },
        },
        skip,
        take: limit,
        orderBy: [
          { priority: 'desc' },
          { createdAt: 'desc' },
        ],
      }),
      this.prisma.dispute.count({ where }),
    ]);

    return {
      disputes: disputes.map(this.mapDispute),
      total,
      page,
      totalPages: Math.ceil(total / limit),
    };
  }

  /**
   * Litiges d'un utilisateur
   */
  async getUserDisputes(userId: number, merchantId: number): Promise<Dispute[]> {
    const disputes = await this.prisma.dispute.findMany({
      where: {
        merchantId,
        OR: [
          { initiatorType: 'user', initiatorId: userId },
          { respondentType: 'user', respondentId: userId },
        ],
      },
      include: {
        evidence: true,
        booking: { select: { bookingNumber: true } },
      },
      orderBy: { createdAt: 'desc' },
    });

    return disputes.map(this.mapDispute);
  }

  /**
   * Litiges d'un chauffeur
   */
  async getDriverDisputes(driverId: number, merchantId: number): Promise<Dispute[]> {
    const disputes = await this.prisma.dispute.findMany({
      where: {
        merchantId,
        OR: [
          { initiatorType: 'driver', initiatorId: driverId },
          { respondentType: 'driver', respondentId: driverId },
        ],
      },
      include: {
        evidence: true,
        booking: { select: { bookingNumber: true } },
      },
      orderBy: { createdAt: 'desc' },
    });

    return disputes.map(this.mapDispute);
  }

  // ==========================================================================
  // DISPUTE ACTIONS
  // ==========================================================================

  /**
   * Répondre à un litige (respondent)
   */
  async respondToDispute(
    id: number,
    merchantId: number,
    respondentId: number,
    response: string,
    evidence?: { type: string; fileUrl: string; description?: string }[],
  ): Promise<Dispute | null> {
    const dispute = await this.prisma.dispute.findFirst({
      where: { id, merchantId, respondentId },
    });

    if (!dispute) return null;

    if (dispute.respondentResponse) {
      throw new BadRequestException('Vous avez déjà répondu à ce litige');
    }

    const updated = await this.prisma.dispute.update({
      where: { id },
      data: {
        respondentResponse: response,
        respondedAt: new Date(),
        status: 'investigating',
      },
    });

    // Ajouter les preuves du respondent
    if (evidence && evidence.length > 0) {
      await this.prisma.disputeEvidence.createMany({
        data: evidence.map((e) => ({
          disputeId: id,
          uploaderId: respondentId,
          uploaderType: dispute.respondentType,
          type: e.type,
          fileUrl: e.fileUrl,
          description: e.description,
        })),
      });
    }

    await this.addSystemComment(id, 'Réponse reçue du défendeur');

    return this.getDispute(id, merchantId);
  }

  /**
   * Assigner un litige à un agent
   */
  async assignDispute(
    id: number,
    merchantId: number,
    agentId: number,
  ): Promise<Dispute | null> {
    const dispute = await this.prisma.dispute.findFirst({
      where: { id, merchantId },
    });

    if (!dispute) return null;

    const updated = await this.prisma.dispute.update({
      where: { id },
      data: {
        assignedTo: agentId,
        status: dispute.status === 'open' ? 'investigating' : dispute.status,
      },
    });

    await this.addSystemComment(id, `Litige assigné à l'agent #${agentId}`);

    return this.getDispute(id, merchantId);
  }

  /**
   * Résoudre un litige
   */
  async resolveDispute(
    id: number,
    merchantId: number,
    resolution: DisputeResolution,
    notes: string,
    amount?: number,
  ): Promise<Dispute | null> {
    const dispute = await this.prisma.dispute.findFirst({
      where: { id, merchantId },
    });

    if (!dispute) return null;

    if (['resolved', 'closed'].includes(dispute.status)) {
      throw new BadRequestException('Ce litige est déjà résolu');
    }

    const updated = await this.prisma.dispute.update({
      where: { id },
      data: {
        status: 'resolved',
        resolution,
        resolutionAmount: amount,
        resolutionNotes: notes,
        resolvedAt: new Date(),
      },
    });

    // Appliquer la résolution
    await this.applyResolution(dispute, resolution, amount);

    await this.addSystemComment(id, `Litige résolu: ${resolution}`);

    return this.getDispute(id, merchantId);
  }

  /**
   * Escalader un litige
   */
  async escalateDispute(
    id: number,
    merchantId: number,
    reason: string,
  ): Promise<Dispute | null> {
    const dispute = await this.prisma.dispute.findFirst({
      where: { id, merchantId },
    });

    if (!dispute) return null;

    const updated = await this.prisma.dispute.update({
      where: { id },
      data: {
        status: 'escalated',
        priority: 'critical',
        escalatedAt: new Date(),
      },
    });

    await this.addSystemComment(id, `Litige escaladé: ${reason}`);

    return this.getDispute(id, merchantId);
  }

  /**
   * Fermer un litige
   */
  async closeDispute(id: number, merchantId: number): Promise<Dispute | null> {
    const dispute = await this.prisma.dispute.findFirst({
      where: { id, merchantId },
    });

    if (!dispute) return null;

    const updated = await this.prisma.dispute.update({
      where: { id },
      data: {
        status: 'closed',
        closedAt: new Date(),
      },
    });

    await this.addSystemComment(id, 'Litige fermé');

    return this.getDispute(id, merchantId);
  }

  // ==========================================================================
  // EVIDENCE
  // ==========================================================================

  /**
   * Ajouter une preuve
   */
  async addEvidence(
    disputeId: number,
    uploaderId: number,
    uploaderType: 'user' | 'driver' | 'admin',
    data: { type: string; fileUrl: string; description?: string },
  ): Promise<DisputeEvidence> {
    const evidence = await this.prisma.disputeEvidence.create({
      data: {
        disputeId,
        uploaderId,
        uploaderType,
        type: data.type,
        fileUrl: data.fileUrl,
        description: data.description,
      },
    });

    return this.mapEvidence(evidence);
  }

  /**
   * Obtenir les preuves d'un litige
   */
  async getEvidence(disputeId: number): Promise<DisputeEvidence[]> {
    const evidence = await this.prisma.disputeEvidence.findMany({
      where: { disputeId },
      orderBy: { createdAt: 'asc' },
    });

    return evidence.map(this.mapEvidence);
  }

  // ==========================================================================
  // COMMENTS
  // ==========================================================================

  /**
   * Ajouter un commentaire
   */
  async addComment(
    disputeId: number,
    authorId: number,
    authorType: 'user' | 'driver' | 'admin',
    authorName: string,
    message: string,
    isInternal = false,
  ): Promise<DisputeComment> {
    const comment = await this.prisma.disputeComment.create({
      data: {
        disputeId,
        authorId,
        authorType,
        authorName,
        message,
        isInternal,
      },
    });

    return this.mapComment(comment);
  }

  /**
   * Ajouter un commentaire système
   */
  private async addSystemComment(disputeId: number, message: string): Promise<void> {
    await this.prisma.disputeComment.create({
      data: {
        disputeId,
        authorId: 0,
        authorType: 'system',
        authorName: 'Système',
        message,
        isInternal: true,
      },
    });
  }

  /**
   * Obtenir les commentaires d'un litige
   */
  async getComments(disputeId: number, includeInternal = false): Promise<DisputeComment[]> {
    const where: any = { disputeId };
    if (!includeInternal) {
      where.isInternal = false;
    }

    const comments = await this.prisma.disputeComment.findMany({
      where,
      orderBy: { createdAt: 'asc' },
    });

    return comments.map(this.mapComment);
  }

  // ==========================================================================
  // STATISTICS
  // ==========================================================================

  /**
   * Statistiques des litiges
   */
  async getStatistics(merchantId: number): Promise<{
    total: number;
    open: number;
    resolved: number;
    byType: Record<DisputeType, number>;
    byResolution: Record<string, number>;
    avgResolutionTime: number;
    totalRefunded: number;
  }> {
    const [total, open, resolved, byType, byResolution, resolutionTimes, refunds] = await Promise.all([
      this.prisma.dispute.count({ where: { merchantId } }),
      this.prisma.dispute.count({
        where: { merchantId, status: { in: ['open', 'investigating', 'pending_response', 'escalated'] } },
      }),
      this.prisma.dispute.count({ where: { merchantId, status: 'resolved' } }),
      this.prisma.dispute.groupBy({
        by: ['type'],
        where: { merchantId },
        _count: { type: true },
      }),
      this.prisma.dispute.groupBy({
        by: ['resolution'],
        where: { merchantId, resolution: { not: null } },
        _count: { resolution: true },
      }),
      this.prisma.dispute.findMany({
        where: { merchantId, resolvedAt: { not: null } },
        select: { createdAt: true, resolvedAt: true },
      }),
      this.prisma.dispute.aggregate({
        where: {
          merchantId,
          resolution: { in: ['refund_full', 'refund_partial'] },
        },
        _sum: { resolutionAmount: true },
      }),
    ]);

    // Calculer le temps moyen de résolution
    let avgResolutionTime = 0;
    if (resolutionTimes.length > 0) {
      const totalHours = resolutionTimes.reduce((sum, d) => {
        const diff = new Date(d.resolvedAt!).getTime() - new Date(d.createdAt).getTime();
        return sum + diff / (1000 * 60 * 60);
      }, 0);
      avgResolutionTime = Math.round(totalHours / resolutionTimes.length);
    }

    const typeMap: Record<string, number> = {};
    byType.forEach((t) => {
      typeMap[t.type] = t._count.type;
    });

    const resolutionMap: Record<string, number> = {};
    byResolution.forEach((r) => {
      if (r.resolution) {
        resolutionMap[r.resolution] = r._count.resolution;
      }
    });

    return {
      total,
      open,
      resolved,
      byType: typeMap as Record<DisputeType, number>,
      byResolution: resolutionMap,
      avgResolutionTime,
      totalRefunded: parseFloat(refunds._sum.resolutionAmount || '0'),
    };
  }

  // ==========================================================================
  // HELPERS
  // ==========================================================================

  /**
   * Générer le numéro de litige
   */
  private async generateDisputeNumber(merchantId: number): Promise<string> {
    const today = new Date();
    const prefix = `DSP-${today.getFullYear()}${(today.getMonth() + 1).toString().padStart(2, '0')}`;

    const lastDispute = await this.prisma.dispute.findFirst({
      where: { merchantId, disputeNumber: { startsWith: prefix } },
      orderBy: { disputeNumber: 'desc' },
    });

    let sequence = 1;
    if (lastDispute) {
      const lastSequence = parseInt(lastDispute.disputeNumber.split('-').pop() || '0');
      sequence = lastSequence + 1;
    }

    return `${prefix}-${sequence.toString().padStart(5, '0')}`;
  }

  /**
   * Déterminer la priorité
   */
  private determinePriority(type: DisputeType, amount?: number): 'low' | 'medium' | 'high' | 'critical' {
    // Priorité critique pour sécurité
    if (type === 'safety') return 'critical';

    // Priorité haute pour comportements
    if (['driver_behavior', 'user_behavior', 'damage'].includes(type)) return 'high';

    // Priorité basée sur le montant
    if (amount) {
      if (amount > 50000) return 'high';
      if (amount > 10000) return 'medium';
    }

    return 'low';
  }

  /**
   * Appliquer la résolution
   */
  private async applyResolution(
    dispute: any,
    resolution: DisputeResolution,
    amount?: number,
  ): Promise<void> {
    switch (resolution) {
      case 'refund_full':
        // Créer un remboursement complet
        // await this.paymentService.refund(dispute.bookingId, dispute.amount);
        break;

      case 'refund_partial':
        // Créer un remboursement partiel
        // await this.paymentService.refund(dispute.bookingId, amount);
        break;

      case 'credit':
        // Ajouter un crédit au wallet
        // await this.walletService.credit(dispute.initiatorId, amount, 'Crédit litige');
        break;

      case 'warning_driver':
        // Envoyer un avertissement au chauffeur
        break;

      case 'warning_user':
        // Envoyer un avertissement à l'utilisateur
        break;

      case 'ban_driver':
        // Suspendre le chauffeur
        // await this.driverService.suspend(dispute.respondentType === 'driver' ? dispute.respondentId : dispute.initiatorId);
        break;

      case 'ban_user':
        // Suspendre l'utilisateur
        // await this.userService.suspend(dispute.respondentType === 'user' ? dispute.respondentId : dispute.initiatorId);
        break;
    }
  }

  private mapDispute(dispute: any): Dispute {
    return {
      id: dispute.id,
      disputeNumber: dispute.disputeNumber,
      bookingId: dispute.bookingId,
      type: dispute.type,
      status: dispute.status,
      initiatorType: dispute.initiatorType,
      initiatorId: dispute.initiatorId,
      respondentType: dispute.respondentType,
      respondentId: dispute.respondentId,
      reason: dispute.reason,
      description: dispute.description,
      evidence: dispute.evidence?.map(this.mapEvidence) || [],
      amount: dispute.amount ? parseFloat(dispute.amount) : undefined,
      resolution: dispute.resolution,
      resolutionAmount: dispute.resolutionAmount ? parseFloat(dispute.resolutionAmount) : undefined,
      resolutionNotes: dispute.resolutionNotes,
      assignedTo: dispute.assignedTo,
      priority: dispute.priority,
      respondentResponse: dispute.respondentResponse,
      respondedAt: dispute.respondedAt,
      resolvedAt: dispute.resolvedAt,
      closedAt: dispute.closedAt,
      escalatedAt: dispute.escalatedAt,
      merchantId: dispute.merchantId,
      createdAt: dispute.createdAt,
      updatedAt: dispute.updatedAt,
      ...(dispute.booking && { booking: dispute.booking }),
    };
  }

  private mapEvidence(evidence: any): DisputeEvidence {
    return {
      id: evidence.id,
      disputeId: evidence.disputeId,
      uploaderId: evidence.uploaderId,
      uploaderType: evidence.uploaderType,
      type: evidence.type,
      fileUrl: evidence.fileUrl,
      description: evidence.description,
      createdAt: evidence.createdAt,
    };
  }

  private mapComment(comment: any): DisputeComment {
    return {
      id: comment.id,
      disputeId: comment.disputeId,
      authorId: comment.authorId,
      authorType: comment.authorType,
      authorName: comment.authorName,
      message: comment.message,
      isInternal: comment.isInternal,
      createdAt: comment.createdAt,
    };
  }
}
