import {
  Controller,
  Get,
  Post,
  Put,
  Body,
  Param,
  Query,
  UseGuards,
  ParseIntPipe,
  DefaultValuePipe,
} from '@nestjs/common';
import {
  DisputeService,
  DisputeStatus,
  DisputeType,
  DisputeResolution,
  CreateDisputeDto,
} from './dispute.service';
import { JwtAuthGuard } from '../../common/guards/jwt-auth.guard';
import { MerchantId } from '../../common/decorators/merchant.decorator';
import { CurrentUser } from '../../common/decorators/current-user.decorator';

// ============================================================================
// DTOs
// ============================================================================

class RespondDisputeDto {
  response: string;
  evidence?: { type: string; fileUrl: string; description?: string }[];
}

class ResolveDisputeDto {
  resolution: DisputeResolution;
  notes: string;
  amount?: number;
}

class AddCommentDto {
  message: string;
  isInternal?: boolean;
}

class AddEvidenceDto {
  type: string;
  fileUrl: string;
  description?: string;
}

// ============================================================================
// CONTROLLER
// ============================================================================

@Controller('disputes')
@UseGuards(JwtAuthGuard)
export class DisputeController {
  constructor(private readonly disputeService: DisputeService) {}

  // ==========================================================================
  // USER/DRIVER ENDPOINTS
  // ==========================================================================

  /**
   * Créer un litige (utilisateur)
   */
  @Post()
  async createDispute(
    @CurrentUser() user: any,
    @MerchantId() merchantId: number,
    @Body() dto: CreateDisputeDto,
  ) {
    const initiatorType = user.driverId ? 'driver' : 'user';
    const initiatorId = user.driverId || user.id;

    const dispute = await this.disputeService.createDispute(
      merchantId,
      initiatorType,
      initiatorId,
      dto,
    );

    return {
      success: true,
      dispute,
      message: `Litige #${dispute.disputeNumber} créé. Nous examinerons votre réclamation.`,
    };
  }

  /**
   * Mes litiges
   */
  @Get('my')
  async getMyDisputes(
    @CurrentUser() user: any,
    @MerchantId() merchantId: number,
  ) {
    if (user.driverId) {
      return this.disputeService.getDriverDisputes(user.driverId, merchantId);
    }
    return this.disputeService.getUserDisputes(user.id, merchantId);
  }

  /**
   * Détails d'un de mes litiges
   */
  @Get('my/:id')
  async getMyDispute(
    @CurrentUser() user: any,
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
  ) {
    const dispute = await this.disputeService.getDispute(id, merchantId);

    if (!dispute) {
      return { error: 'Litige non trouvé' };
    }

    // Vérifier l'accès
    const userId = user.driverId || user.id;
    const userType = user.driverId ? 'driver' : 'user';

    if (
      (dispute.initiatorType !== userType || dispute.initiatorId !== userId) &&
      (dispute.respondentType !== userType || dispute.respondentId !== userId)
    ) {
      return { error: 'Litige non trouvé' };
    }

    const comments = await this.disputeService.getComments(id, false);

    return {
      ...dispute,
      comments,
    };
  }

  /**
   * Répondre à un litige (défendeur)
   */
  @Post('my/:id/respond')
  async respondToDispute(
    @CurrentUser() user: any,
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
    @Body() dto: RespondDisputeDto,
  ) {
    const respondentId = user.driverId || user.id;

    const dispute = await this.disputeService.respondToDispute(
      id,
      merchantId,
      respondentId,
      dto.response,
      dto.evidence,
    );

    if (!dispute) {
      return { error: 'Litige non trouvé ou vous n\'êtes pas le défendeur' };
    }

    return {
      success: true,
      dispute,
      message: 'Réponse enregistrée',
    };
  }

  /**
   * Ajouter une preuve à mon litige
   */
  @Post('my/:id/evidence')
  async addMyEvidence(
    @CurrentUser() user: any,
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
    @Body() dto: AddEvidenceDto,
  ) {
    const dispute = await this.disputeService.getDispute(id, merchantId);
    if (!dispute) {
      return { error: 'Litige non trouvé' };
    }

    const userId = user.driverId || user.id;
    const userType = user.driverId ? 'driver' : 'user';

    // Vérifier l'accès
    if (
      (dispute.initiatorType !== userType || dispute.initiatorId !== userId) &&
      (dispute.respondentType !== userType || dispute.respondentId !== userId)
    ) {
      return { error: 'Litige non trouvé' };
    }

    const evidence = await this.disputeService.addEvidence(id, userId, userType, dto);

    return {
      success: true,
      evidence,
    };
  }

  /**
   * Ajouter un commentaire à mon litige
   */
  @Post('my/:id/comments')
  async addMyComment(
    @CurrentUser() user: any,
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
    @Body() dto: AddCommentDto,
  ) {
    const dispute = await this.disputeService.getDispute(id, merchantId);
    if (!dispute) {
      return { error: 'Litige non trouvé' };
    }

    const userId = user.driverId || user.id;
    const userType = user.driverId ? 'driver' : 'user';

    // Vérifier l'accès
    if (
      (dispute.initiatorType !== userType || dispute.initiatorId !== userId) &&
      (dispute.respondentType !== userType || dispute.respondentId !== userId)
    ) {
      return { error: 'Litige non trouvé' };
    }

    const comment = await this.disputeService.addComment(
      id,
      userId,
      userType,
      `${user.firstName} ${user.lastName}`,
      dto.message,
      false,
    );

    return {
      success: true,
      comment,
    };
  }

  // ==========================================================================
  // ADMIN ENDPOINTS
  // ==========================================================================

  /**
   * Lister tous les litiges (admin)
   */
  @Get('admin')
  async listDisputes(
    @MerchantId() merchantId: number,
    @Query('page', new DefaultValuePipe(1), ParseIntPipe) page: number,
    @Query('limit', new DefaultValuePipe(20), ParseIntPipe) limit: number,
    @Query('status') status?: DisputeStatus,
    @Query('type') type?: DisputeType,
    @Query('priority') priority?: string,
    @Query('assignedTo') assignedTo?: string,
    @Query('search') search?: string,
  ) {
    return this.disputeService.listDisputes(merchantId, {
      page,
      limit,
      status,
      type,
      priority,
      assignedTo: assignedTo ? parseInt(assignedTo) : undefined,
      search,
    });
  }

  /**
   * Détails d'un litige (admin)
   */
  @Get('admin/:id')
  async getDispute(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
  ) {
    const dispute = await this.disputeService.getDispute(id, merchantId);

    if (!dispute) {
      return { error: 'Litige non trouvé' };
    }

    const comments = await this.disputeService.getComments(id, true);

    return {
      ...dispute,
      comments,
    };
  }

  /**
   * Assigner un litige
   */
  @Post('admin/:id/assign')
  async assignDispute(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
    @Body() body: { agentId: number },
  ) {
    const dispute = await this.disputeService.assignDispute(id, merchantId, body.agentId);

    if (!dispute) {
      return { error: 'Litige non trouvé' };
    }

    return {
      success: true,
      dispute,
      message: 'Litige assigné',
    };
  }

  /**
   * Résoudre un litige
   */
  @Post('admin/:id/resolve')
  async resolveDispute(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
    @Body() dto: ResolveDisputeDto,
  ) {
    const dispute = await this.disputeService.resolveDispute(
      id,
      merchantId,
      dto.resolution,
      dto.notes,
      dto.amount,
    );

    if (!dispute) {
      return { error: 'Litige non trouvé' };
    }

    return {
      success: true,
      dispute,
      message: 'Litige résolu',
    };
  }

  /**
   * Escalader un litige
   */
  @Post('admin/:id/escalate')
  async escalateDispute(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
    @Body() body: { reason: string },
  ) {
    const dispute = await this.disputeService.escalateDispute(id, merchantId, body.reason);

    if (!dispute) {
      return { error: 'Litige non trouvé' };
    }

    return {
      success: true,
      dispute,
      message: 'Litige escaladé',
    };
  }

  /**
   * Fermer un litige
   */
  @Post('admin/:id/close')
  async closeDispute(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
  ) {
    const dispute = await this.disputeService.closeDispute(id, merchantId);

    if (!dispute) {
      return { error: 'Litige non trouvé' };
    }

    return {
      success: true,
      dispute,
      message: 'Litige fermé',
    };
  }

  /**
   * Ajouter un commentaire (admin)
   */
  @Post('admin/:id/comments')
  async addAdminComment(
    @CurrentUser() admin: any,
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
    @Body() dto: AddCommentDto,
  ) {
    const dispute = await this.disputeService.getDispute(id, merchantId);
    if (!dispute) {
      return { error: 'Litige non trouvé' };
    }

    const comment = await this.disputeService.addComment(
      id,
      admin.id,
      'admin',
      `${admin.firstName} ${admin.lastName}`,
      dto.message,
      dto.isInternal || false,
    );

    return {
      success: true,
      comment,
    };
  }

  /**
   * Ajouter une preuve (admin)
   */
  @Post('admin/:id/evidence')
  async addAdminEvidence(
    @CurrentUser() admin: any,
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
    @Body() dto: AddEvidenceDto,
  ) {
    const dispute = await this.disputeService.getDispute(id, merchantId);
    if (!dispute) {
      return { error: 'Litige non trouvé' };
    }

    const evidence = await this.disputeService.addEvidence(id, admin.id, 'admin', dto);

    return {
      success: true,
      evidence,
    };
  }

  // ==========================================================================
  // STATISTICS & METADATA
  // ==========================================================================

  /**
   * Statistiques des litiges
   */
  @Get('admin/stats/overview')
  async getStatistics(@MerchantId() merchantId: number) {
    return this.disputeService.getStatistics(merchantId);
  }

  /**
   * Types de litiges
   */
  @Get('meta/types')
  getDisputeTypes() {
    return {
      types: [
        { type: 'fare', name: 'Tarif', description: 'Contestation du tarif facturé' },
        { type: 'service', name: 'Service', description: 'Qualité du service' },
        { type: 'safety', name: 'Sécurité', description: 'Problème de sécurité' },
        { type: 'payment', name: 'Paiement', description: 'Problème de paiement' },
        { type: 'driver_behavior', name: 'Comportement chauffeur', description: 'Comportement inapproprié du chauffeur' },
        { type: 'user_behavior', name: 'Comportement client', description: 'Comportement inapproprié du client' },
        { type: 'damage', name: 'Dommage', description: 'Dommage matériel' },
        { type: 'lost_item', name: 'Objet perdu', description: 'Objet perdu dans le véhicule' },
        { type: 'other', name: 'Autre', description: 'Autre problème' },
      ],
    };
  }

  /**
   * Résolutions possibles
   */
  @Get('meta/resolutions')
  getResolutions() {
    return {
      resolutions: [
        { resolution: 'refund_full', name: 'Remboursement total', description: 'Remboursement intégral' },
        { resolution: 'refund_partial', name: 'Remboursement partiel', description: 'Remboursement partiel' },
        { resolution: 'credit', name: 'Crédit', description: 'Crédit sur le compte' },
        { resolution: 'no_action', name: 'Aucune action', description: 'Litige rejeté' },
        { resolution: 'warning_driver', name: 'Avertissement chauffeur', description: 'Avertissement au chauffeur' },
        { resolution: 'warning_user', name: 'Avertissement utilisateur', description: 'Avertissement au client' },
        { resolution: 'ban_driver', name: 'Suspension chauffeur', description: 'Suspension du chauffeur' },
        { resolution: 'ban_user', name: 'Suspension utilisateur', description: 'Suspension du client' },
      ],
    };
  }
}
