import { Injectable, Logger, BadRequestException, NotFoundException } from '@nestjs/common';
import { PrismaService } from '../../common/prisma/prisma.service';
import { PricingService } from '../pricing/pricing.service';
import { WebsocketService } from '../websocket/websocket.service';
import { QueueService } from '../queue/queue.service';

// Delivery status flow
const DELIVERY_STATUS = {
  PENDING: 'pending',
  CONFIRMED: 'confirmed',
  DRIVER_ASSIGNED: 'driver_assigned',
  PICKUP_IN_PROGRESS: 'pickup_in_progress',
  PICKED_UP: 'picked_up',
  IN_TRANSIT: 'in_transit',
  ARRIVED_AT_DROP: 'arrived_at_drop',
  DELIVERED: 'delivered',
  CANCELLED: 'cancelled',
  FAILED: 'failed',
};

interface CreateDeliveryDto {
  // Pickup details
  pickupAddress: string;
  pickupLatitude: number;
  pickupLongitude: number;
  pickupContactName: string;
  pickupContactPhone: string;
  pickupInstructions?: string;

  // Drop details
  dropAddress: string;
  dropLatitude: number;
  dropLongitude: number;
  dropContactName: string;
  dropContactPhone: string;
  dropInstructions?: string;

  // Package details
  packageType: string;
  packageWeight?: number;
  packageSize?: 'small' | 'medium' | 'large' | 'extra_large';
  packageDescription?: string;
  declaredValue?: number;
  isFragile?: boolean;
  requiresSignature?: boolean;

  // Options
  vehicleTypeId?: number;
  scheduledPickupTime?: Date;
  paymentType: 'cash' | 'card' | 'wallet';
  promoCode?: string;
}

interface DeliveryEstimate {
  baseFare: number;
  distanceFare: number;
  weightSurcharge: number;
  sizeSurcharge: number;
  fragileHandling: number;
  insuranceFee: number;
  surgeMultiplier: number;
  discount: number;
  totalFare: number;
  estimatedDistance: number;
  estimatedDuration: number;
  currency: string;
}

@Injectable()
export class DeliveryService {
  private readonly logger = new Logger(DeliveryService.name);

  // Package size multipliers
  private readonly SIZE_MULTIPLIERS = {
    small: 1.0,
    medium: 1.2,
    large: 1.5,
    extra_large: 2.0,
  };

  // Weight surcharge per kg above 5kg
  private readonly WEIGHT_SURCHARGE_PER_KG = 50;
  private readonly FREE_WEIGHT_LIMIT_KG = 5;

  // Fragile handling fee
  private readonly FRAGILE_FEE = 200;

  // Insurance rate (percentage of declared value)
  private readonly INSURANCE_RATE = 0.02;

  constructor(
    private prisma: PrismaService,
    private pricingService: PricingService,
    private websocketService: WebsocketService,
    private queueService: QueueService,
  ) {}

  /**
   * Calculate delivery estimate
   */
  async calculateEstimate(
    merchantId: number,
    dto: {
      pickupLatitude: number;
      pickupLongitude: number;
      dropLatitude: number;
      dropLongitude: number;
      packageWeight?: number;
      packageSize?: 'small' | 'medium' | 'large' | 'extra_large';
      isFragile?: boolean;
      declaredValue?: number;
      vehicleTypeId?: number;
      promoCode?: string;
    },
  ): Promise<DeliveryEstimate> {
    // Get base pricing from ride estimation
    const baseEstimate = await this.pricingService.calculateEstimate(
      merchantId,
      dto.vehicleTypeId || 1, // Default vehicle type
      dto.pickupLatitude,
      dto.pickupLongitude,
      dto.dropLatitude,
      dto.dropLongitude,
      dto.promoCode,
    );

    // Calculate weight surcharge
    const weight = dto.packageWeight || 0;
    const weightSurcharge = weight > this.FREE_WEIGHT_LIMIT_KG
      ? (weight - this.FREE_WEIGHT_LIMIT_KG) * this.WEIGHT_SURCHARGE_PER_KG
      : 0;

    // Calculate size surcharge
    const sizeMultiplier = this.SIZE_MULTIPLIERS[dto.packageSize || 'small'];
    const sizeSurcharge = Math.round(baseEstimate.baseFare * (sizeMultiplier - 1));

    // Fragile handling fee
    const fragileHandling = dto.isFragile ? this.FRAGILE_FEE : 0;

    // Insurance fee (optional)
    const insuranceFee = dto.declaredValue
      ? Math.round(dto.declaredValue * this.INSURANCE_RATE)
      : 0;

    // Calculate total
    const subtotal =
      baseEstimate.baseFare +
      baseEstimate.distanceFare +
      weightSurcharge +
      sizeSurcharge +
      fragileHandling +
      insuranceFee;

    // Apply surge
    const afterSurge = Math.round(subtotal * baseEstimate.surgeMultiplier);

    // Apply discount
    const totalFare = afterSurge - baseEstimate.discount;

    return {
      baseFare: baseEstimate.baseFare,
      distanceFare: baseEstimate.distanceFare,
      weightSurcharge,
      sizeSurcharge,
      fragileHandling,
      insuranceFee,
      surgeMultiplier: baseEstimate.surgeMultiplier,
      discount: baseEstimate.discount,
      totalFare,
      estimatedDistance: baseEstimate.distanceKm,
      estimatedDuration: baseEstimate.durationMinutes,
      currency: baseEstimate.currency,
    };
  }

  /**
   * Create a new delivery order
   */
  async createDelivery(
    merchantId: number,
    userId: number,
    dto: CreateDeliveryDto,
  ): Promise<any> {
    // Calculate fare
    const estimate = await this.calculateEstimate(merchantId, {
      pickupLatitude: dto.pickupLatitude,
      pickupLongitude: dto.pickupLongitude,
      dropLatitude: dto.dropLatitude,
      dropLongitude: dto.dropLongitude,
      packageWeight: dto.packageWeight,
      packageSize: dto.packageSize,
      isFragile: dto.isFragile,
      declaredValue: dto.declaredValue,
      vehicleTypeId: dto.vehicleTypeId,
      promoCode: dto.promoCode,
    });

    // Generate unique delivery ID
    const deliveryNumber = await this.generateDeliveryNumber(merchantId);

    // Create delivery record
    const delivery = await this.prisma.delivery.create({
      data: {
        merchant_id: merchantId,
        user_id: userId,
        delivery_number: deliveryNumber,

        // Pickup
        pickup_address: dto.pickupAddress,
        pickup_latitude: dto.pickupLatitude,
        pickup_longitude: dto.pickupLongitude,
        pickup_contact_name: dto.pickupContactName,
        pickup_contact_phone: dto.pickupContactPhone,
        pickup_instructions: dto.pickupInstructions,

        // Drop
        drop_address: dto.dropAddress,
        drop_latitude: dto.dropLatitude,
        drop_longitude: dto.dropLongitude,
        drop_contact_name: dto.dropContactName,
        drop_contact_phone: dto.dropContactPhone,
        drop_instructions: dto.dropInstructions,

        // Package
        package_type: dto.packageType,
        package_weight: dto.packageWeight,
        package_size: dto.packageSize || 'small',
        package_description: dto.packageDescription,
        declared_value: dto.declaredValue,
        is_fragile: dto.isFragile ? 1 : 0,
        requires_signature: dto.requiresSignature ? 1 : 0,

        // Pricing
        base_fare: estimate.baseFare,
        distance_fare: estimate.distanceFare,
        weight_surcharge: estimate.weightSurcharge,
        size_surcharge: estimate.sizeSurcharge,
        fragile_fee: estimate.fragileHandling,
        insurance_fee: estimate.insuranceFee,
        surge_multiplier: estimate.surgeMultiplier,
        discount_amount: estimate.discount,
        total_fare: estimate.totalFare,

        // Status
        delivery_status: dto.scheduledPickupTime
          ? DELIVERY_STATUS.CONFIRMED
          : DELIVERY_STATUS.PENDING,
        scheduled_pickup_time: dto.scheduledPickupTime,
        payment_type: dto.paymentType,
        payment_status: 'pending',

        // Distance/Time
        estimated_distance: estimate.estimatedDistance,
        estimated_duration: estimate.estimatedDuration,

        vehicle_type_id: dto.vehicleTypeId,
        created_at: new Date(),
      },
    });

    // If not scheduled, start driver assignment
    if (!dto.scheduledPickupTime) {
      await this.queueService.addBookingJob('assign-driver', {
        bookingId: delivery.id,
        merchantId,
        type: 'delivery',
      });
    }

    this.logger.log(`Delivery #${delivery.id} created by user #${userId}`);

    return delivery;
  }

  /**
   * Get delivery by ID
   */
  async getDeliveryById(deliveryId: number, userId?: number): Promise<any> {
    const where: any = { id: deliveryId };
    if (userId) {
      where.user_id = userId;
    }

    const delivery = await this.prisma.delivery.findFirst({
      where,
      include: {
        user: {
          select: {
            id: true,
            first_name: true,
            last_name: true,
            mobile: true,
          },
        },
        driver: {
          select: {
            id: true,
            first_name: true,
            last_name: true,
            mobile: true,
            profile_picture: true,
            rating: true,
            vehicle_number: true,
            vehicle_model: true,
            latitude: true,
            longitude: true,
          },
        },
      },
    });

    if (!delivery) {
      throw new NotFoundException('Livraison non trouvee');
    }

    return delivery;
  }

  /**
   * Get user's delivery history
   */
  async getUserDeliveries(
    userId: number,
    options?: { status?: string; page?: number; limit?: number },
  ): Promise<{ data: any[]; total: number }> {
    const page = options?.page || 1;
    const limit = options?.limit || 20;
    const skip = (page - 1) * limit;

    const where: any = { user_id: userId };
    if (options?.status) {
      where.delivery_status = options.status;
    }

    const [data, total] = await Promise.all([
      this.prisma.delivery.findMany({
        where,
        skip,
        take: limit,
        orderBy: { created_at: 'desc' },
        include: {
          driver: {
            select: {
              first_name: true,
              last_name: true,
              rating: true,
            },
          },
        },
      }),
      this.prisma.delivery.count({ where }),
    ]);

    return { data, total };
  }

  /**
   * Cancel a delivery
   */
  async cancelDelivery(
    deliveryId: number,
    userId: number,
    reason?: string,
  ): Promise<any> {
    const delivery = await this.prisma.delivery.findFirst({
      where: { id: deliveryId, user_id: userId },
    });

    if (!delivery) {
      throw new NotFoundException('Livraison non trouvee');
    }

    // Check if cancellation is allowed
    const cancellableStatuses = [
      DELIVERY_STATUS.PENDING,
      DELIVERY_STATUS.CONFIRMED,
      DELIVERY_STATUS.DRIVER_ASSIGNED,
    ];

    if (!cancellableStatuses.includes(delivery.delivery_status)) {
      throw new BadRequestException('Cette livraison ne peut plus etre annulee');
    }

    // Calculate cancellation fee
    let cancellationFee = 0;
    if (delivery.delivery_status === DELIVERY_STATUS.DRIVER_ASSIGNED) {
      cancellationFee = Math.round(Number(delivery.total_fare) * 0.1); // 10% fee
    }

    // Update delivery
    const updated = await this.prisma.delivery.update({
      where: { id: deliveryId },
      data: {
        delivery_status: DELIVERY_STATUS.CANCELLED,
        cancelled_time: new Date(),
        cancellation_reason: reason,
        cancellation_fee: cancellationFee,
      },
    });

    // If driver was assigned, notify and free them
    if (delivery.driver_id) {
      await this.prisma.driver.update({
        where: { id: delivery.driver_id },
        data: { free_busy: 2 }, // Free
      });

      // Notify driver
      this.websocketService.notifyDeliveryCancelled(delivery.driver_id, {
        delivery_id: deliveryId,
        reason,
      });
    }

    this.logger.log(`Delivery #${deliveryId} cancelled by user #${userId}`);

    return updated;
  }

  /**
   * Driver: Accept delivery
   */
  async acceptDelivery(deliveryId: number, driverId: number): Promise<any> {
    const delivery = await this.prisma.delivery.findUnique({
      where: { id: deliveryId },
    });

    if (!delivery) {
      throw new NotFoundException('Livraison non trouvee');
    }

    if (delivery.delivery_status !== DELIVERY_STATUS.PENDING) {
      throw new BadRequestException('Cette livraison n\'est plus disponible');
    }

    // Assign driver
    const updated = await this.prisma.delivery.update({
      where: { id: deliveryId },
      data: {
        driver_id: driverId,
        delivery_status: DELIVERY_STATUS.DRIVER_ASSIGNED,
        accepted_time: new Date(),
      },
    });

    // Mark driver as busy
    await this.prisma.driver.update({
      where: { id: driverId },
      data: { free_busy: 1 },
    });

    // Notify user
    const driver = await this.prisma.driver.findUnique({ where: { id: driverId } });
    this.websocketService.notifyDeliveryAccepted(delivery.user_id, {
      delivery_id: deliveryId,
      driver: {
        id: driver.id,
        name: `${driver.first_name} ${driver.last_name}`,
        phone: driver.mobile,
        profile_picture: driver.profile_picture,
        vehicle_number: driver.vehicle_number,
      },
    });

    return updated;
  }

  /**
   * Driver: Start pickup
   */
  async startPickup(deliveryId: number, driverId: number): Promise<any> {
    const delivery = await this.prisma.delivery.findFirst({
      where: { id: deliveryId, driver_id: driverId },
    });

    if (!delivery) {
      throw new NotFoundException('Livraison non trouvee');
    }

    if (delivery.delivery_status !== DELIVERY_STATUS.DRIVER_ASSIGNED) {
      throw new BadRequestException('Statut invalide');
    }

    const updated = await this.prisma.delivery.update({
      where: { id: deliveryId },
      data: {
        delivery_status: DELIVERY_STATUS.PICKUP_IN_PROGRESS,
        pickup_started_time: new Date(),
      },
    });

    // Notify user
    this.websocketService.notifyDeliveryStatusChanged(delivery.user_id, {
      delivery_id: deliveryId,
      status: DELIVERY_STATUS.PICKUP_IN_PROGRESS,
      message: 'Le chauffeur se dirige vers le point de collecte',
    });

    return updated;
  }

  /**
   * Driver: Confirm pickup
   */
  async confirmPickup(
    deliveryId: number,
    driverId: number,
    pickupProofImage?: string,
  ): Promise<any> {
    const delivery = await this.prisma.delivery.findFirst({
      where: { id: deliveryId, driver_id: driverId },
    });

    if (!delivery) {
      throw new NotFoundException('Livraison non trouvee');
    }

    if (delivery.delivery_status !== DELIVERY_STATUS.PICKUP_IN_PROGRESS) {
      throw new BadRequestException('Statut invalide');
    }

    const updated = await this.prisma.delivery.update({
      where: { id: deliveryId },
      data: {
        delivery_status: DELIVERY_STATUS.PICKED_UP,
        picked_up_time: new Date(),
        pickup_proof_image: pickupProofImage,
      },
    });

    // Notify user
    this.websocketService.notifyDeliveryStatusChanged(delivery.user_id, {
      delivery_id: deliveryId,
      status: DELIVERY_STATUS.PICKED_UP,
      message: 'Colis recupere, livraison en cours',
    });

    // Also notify drop contact via SMS
    await this.queueService.addNotificationJob('send-sms', {
      phone: delivery.drop_contact_phone,
      message: `Votre colis est en route! Numero de suivi: ${delivery.delivery_number}`,
    });

    return updated;
  }

  /**
   * Driver: Start transit
   */
  async startTransit(deliveryId: number, driverId: number): Promise<any> {
    const delivery = await this.prisma.delivery.findFirst({
      where: { id: deliveryId, driver_id: driverId },
    });

    if (!delivery) {
      throw new NotFoundException('Livraison non trouvee');
    }

    const updated = await this.prisma.delivery.update({
      where: { id: deliveryId },
      data: {
        delivery_status: DELIVERY_STATUS.IN_TRANSIT,
        transit_started_time: new Date(),
      },
    });

    this.websocketService.notifyDeliveryStatusChanged(delivery.user_id, {
      delivery_id: deliveryId,
      status: DELIVERY_STATUS.IN_TRANSIT,
      message: 'Colis en transit vers la destination',
    });

    return updated;
  }

  /**
   * Driver: Arrived at drop
   */
  async arrivedAtDrop(deliveryId: number, driverId: number): Promise<any> {
    const delivery = await this.prisma.delivery.findFirst({
      where: { id: deliveryId, driver_id: driverId },
    });

    if (!delivery) {
      throw new NotFoundException('Livraison non trouvee');
    }

    const updated = await this.prisma.delivery.update({
      where: { id: deliveryId },
      data: {
        delivery_status: DELIVERY_STATUS.ARRIVED_AT_DROP,
        arrived_at_drop_time: new Date(),
      },
    });

    // Notify user and drop contact
    this.websocketService.notifyDeliveryStatusChanged(delivery.user_id, {
      delivery_id: deliveryId,
      status: DELIVERY_STATUS.ARRIVED_AT_DROP,
      message: 'Le chauffeur est arrive a destination',
    });

    // SMS to drop contact
    await this.queueService.addNotificationJob('send-sms', {
      phone: delivery.drop_contact_phone,
      message: `Votre colis est arrive! Le livreur vous attend.`,
    });

    return updated;
  }

  /**
   * Driver: Complete delivery
   */
  async completeDelivery(
    deliveryId: number,
    driverId: number,
    data: {
      deliveryProofImage?: string;
      recipientSignature?: string;
      recipientName?: string;
    },
  ): Promise<any> {
    const delivery = await this.prisma.delivery.findFirst({
      where: { id: deliveryId, driver_id: driverId },
    });

    if (!delivery) {
      throw new NotFoundException('Livraison non trouvee');
    }

    if (delivery.delivery_status !== DELIVERY_STATUS.ARRIVED_AT_DROP) {
      throw new BadRequestException('Statut invalide');
    }

    // Check signature requirement
    if (delivery.requires_signature && !data.recipientSignature) {
      throw new BadRequestException('Signature du destinataire requise');
    }

    const updated = await this.prisma.delivery.update({
      where: { id: deliveryId },
      data: {
        delivery_status: DELIVERY_STATUS.DELIVERED,
        delivered_time: new Date(),
        delivery_proof_image: data.deliveryProofImage,
        recipient_signature: data.recipientSignature,
        recipient_name: data.recipientName,
        payment_status: delivery.payment_type === 'cash' ? 'collected' : 'completed',
      },
    });

    // Free driver
    await this.prisma.driver.update({
      where: { id: driverId },
      data: { free_busy: 2 },
    });

    // Notify user
    this.websocketService.notifyDeliveryStatusChanged(delivery.user_id, {
      delivery_id: deliveryId,
      status: DELIVERY_STATUS.DELIVERED,
      message: 'Livraison effectuee avec succes!',
    });

    // Queue post-delivery tasks
    await this.queueService.addBookingJob('complete-booking', {
      bookingId: deliveryId,
      type: 'delivery',
    });

    this.logger.log(`Delivery #${deliveryId} completed by driver #${driverId}`);

    return updated;
  }

  /**
   * Track delivery (public endpoint)
   */
  async trackDelivery(deliveryNumber: string): Promise<any> {
    const delivery = await this.prisma.delivery.findFirst({
      where: { delivery_number: deliveryNumber },
      select: {
        delivery_number: true,
        delivery_status: true,
        pickup_address: true,
        drop_address: true,
        estimated_duration: true,
        created_at: true,
        accepted_time: true,
        picked_up_time: true,
        delivered_time: true,
        driver: {
          select: {
            first_name: true,
            last_name: true,
            latitude: true,
            longitude: true,
          },
        },
      },
    });

    if (!delivery) {
      throw new NotFoundException('Livraison non trouvee');
    }

    return {
      ...delivery,
      timeline: this.buildDeliveryTimeline(delivery),
    };
  }

  /**
   * Build delivery timeline
   */
  private buildDeliveryTimeline(delivery: any): Array<{ status: string; time?: Date; completed: boolean }> {
    const statuses = [
      { status: 'Commande creee', time: delivery.created_at, completed: true },
      { status: 'Chauffeur assigne', time: delivery.accepted_time, completed: !!delivery.accepted_time },
      { status: 'Colis recupere', time: delivery.picked_up_time, completed: !!delivery.picked_up_time },
      { status: 'Livre', time: delivery.delivered_time, completed: !!delivery.delivered_time },
    ];

    return statuses;
  }

  /**
   * Generate unique delivery number
   */
  private async generateDeliveryNumber(merchantId: number): Promise<string> {
    const date = new Date();
    const prefix = 'DLV';
    const dateStr = date.toISOString().slice(0, 10).replace(/-/g, '');

    // Get today's count
    const startOfDay = new Date(date.setHours(0, 0, 0, 0));
    const endOfDay = new Date(date.setHours(23, 59, 59, 999));

    const count = await this.prisma.delivery.count({
      where: {
        merchant_id: merchantId,
        created_at: {
          gte: startOfDay,
          lte: endOfDay,
        },
      },
    });

    const sequence = String(count + 1).padStart(4, '0');
    return `${prefix}${dateStr}${sequence}`;
  }
}
