import {
  Controller,
  Get,
  Post,
  Body,
  Param,
  Query,
  UseGuards,
  ParseIntPipe,
} from '@nestjs/common';
import { ApiTags, ApiOperation, ApiBearerAuth } from '@nestjs/swagger';
import { DeliveryService } from './delivery.service';
import { JwtAuthGuard } from '../../common/guards/jwt-auth.guard';
import { MerchantGuard } from '../../common/guards/merchant.guard';
import { MerchantId, CurrentUser } from '../../common/decorators/merchant.decorator';
import { Public } from '../../common/decorators/public.decorator';
import {
  EstimateDeliveryDto,
  CreateDeliveryDto,
  ConfirmPickupDto,
  CompleteDeliveryDto,
  CancelDeliveryDto,
} from './dto';

@ApiTags('Delivery')
@Controller('api/delivery')
@UseGuards(JwtAuthGuard, MerchantGuard)
@ApiBearerAuth()
export class DeliveryController {
  constructor(private deliveryService: DeliveryService) {}

  // ============================================================================
  // USER ENDPOINTS
  // ============================================================================

  @Post('estimate')
  @ApiOperation({ summary: 'Get delivery fare estimate' })
  async estimateDelivery(
    @Body() dto: EstimateDeliveryDto,
    @MerchantId() merchantId: number,
  ) {
    return this.deliveryService.calculateEstimate(merchantId, {
      pickupLatitude: dto.pickupLatitude,
      pickupLongitude: dto.pickupLongitude,
      dropLatitude: dto.dropLatitude,
      dropLongitude: dto.dropLongitude,
      packageWeight: dto.packageWeight,
      packageSize: dto.packageSize,
      isFragile: dto.isFragile,
      declaredValue: dto.declaredValue,
      vehicleTypeId: dto.vehicleTypeId,
      promoCode: dto.promoCode,
    });
  }

  @Post('create')
  @ApiOperation({ summary: 'Create a new delivery order' })
  async createDelivery(
    @Body() dto: CreateDeliveryDto,
    @MerchantId() merchantId: number,
    @CurrentUser() user: any,
  ) {
    return this.deliveryService.createDelivery(merchantId, user.id, {
      pickupAddress: dto.pickupAddress,
      pickupLatitude: dto.pickupLatitude,
      pickupLongitude: dto.pickupLongitude,
      pickupContactName: dto.pickupContactName,
      pickupContactPhone: dto.pickupContactPhone,
      pickupInstructions: dto.pickupInstructions,
      dropAddress: dto.dropAddress,
      dropLatitude: dto.dropLatitude,
      dropLongitude: dto.dropLongitude,
      dropContactName: dto.dropContactName,
      dropContactPhone: dto.dropContactPhone,
      dropInstructions: dto.dropInstructions,
      packageType: dto.packageType,
      packageWeight: dto.packageWeight,
      packageSize: dto.packageSize,
      packageDescription: dto.packageDescription,
      declaredValue: dto.declaredValue,
      isFragile: dto.isFragile,
      requiresSignature: dto.requiresSignature,
      vehicleTypeId: dto.vehicleTypeId,
      scheduledPickupTime: dto.scheduledPickupTime ? new Date(dto.scheduledPickupTime) : undefined,
      paymentType: dto.paymentType,
      promoCode: dto.promoCode,
    });
  }

  @Get('list')
  @ApiOperation({ summary: 'Get user delivery history' })
  async getUserDeliveries(
    @CurrentUser() user: any,
    @Query('status') status?: string,
    @Query('page') page?: string,
    @Query('limit') limit?: string,
  ) {
    return this.deliveryService.getUserDeliveries(user.id, {
      status,
      page: page ? parseInt(page) : 1,
      limit: limit ? parseInt(limit) : 20,
    });
  }

  @Get(':id')
  @ApiOperation({ summary: 'Get delivery details' })
  async getDelivery(
    @Param('id', ParseIntPipe) id: number,
    @CurrentUser() user: any,
  ) {
    return this.deliveryService.getDeliveryById(id, user.id);
  }

  @Post(':id/cancel')
  @ApiOperation({ summary: 'Cancel a delivery' })
  async cancelDelivery(
    @Param('id', ParseIntPipe) id: number,
    @Body() dto: CancelDeliveryDto,
    @CurrentUser() user: any,
  ) {
    return this.deliveryService.cancelDelivery(id, user.id, dto.reason);
  }

  // ============================================================================
  // DRIVER ENDPOINTS
  // ============================================================================

  @Post(':id/accept')
  @ApiOperation({ summary: 'Driver: Accept delivery' })
  async acceptDelivery(
    @Param('id', ParseIntPipe) id: number,
    @CurrentUser() driver: any,
  ) {
    return this.deliveryService.acceptDelivery(id, driver.id);
  }

  @Post(':id/start-pickup')
  @ApiOperation({ summary: 'Driver: Start going to pickup point' })
  async startPickup(
    @Param('id', ParseIntPipe) id: number,
    @CurrentUser() driver: any,
  ) {
    return this.deliveryService.startPickup(id, driver.id);
  }

  @Post(':id/confirm-pickup')
  @ApiOperation({ summary: 'Driver: Confirm package pickup' })
  async confirmPickup(
    @Param('id', ParseIntPipe) id: number,
    @Body() dto: ConfirmPickupDto,
    @CurrentUser() driver: any,
  ) {
    return this.deliveryService.confirmPickup(id, driver.id, dto.pickupProofImage);
  }

  @Post(':id/start-transit')
  @ApiOperation({ summary: 'Driver: Start transit to drop point' })
  async startTransit(
    @Param('id', ParseIntPipe) id: number,
    @CurrentUser() driver: any,
  ) {
    return this.deliveryService.startTransit(id, driver.id);
  }

  @Post(':id/arrived')
  @ApiOperation({ summary: 'Driver: Arrived at drop point' })
  async arrivedAtDrop(
    @Param('id', ParseIntPipe) id: number,
    @CurrentUser() driver: any,
  ) {
    return this.deliveryService.arrivedAtDrop(id, driver.id);
  }

  @Post(':id/complete')
  @ApiOperation({ summary: 'Driver: Complete delivery' })
  async completeDelivery(
    @Param('id', ParseIntPipe) id: number,
    @Body() dto: CompleteDeliveryDto,
    @CurrentUser() driver: any,
  ) {
    return this.deliveryService.completeDelivery(id, driver.id, {
      deliveryProofImage: dto.deliveryProofImage,
      recipientSignature: dto.recipientSignature,
      recipientName: dto.recipientName,
    });
  }

  // ============================================================================
  // PUBLIC ENDPOINTS
  // ============================================================================

  @Get('track/:deliveryNumber')
  @Public()
  @ApiOperation({ summary: 'Track delivery by number (public)' })
  async trackDelivery(@Param('deliveryNumber') deliveryNumber: string) {
    return this.deliveryService.trackDelivery(deliveryNumber);
  }
}
