import { Injectable, Logger, OnModuleInit } from '@nestjs/common';
import { Cron, CronExpression } from '@nestjs/schedule';
import { PrismaService } from '../../../common/prisma/prisma.service';

@Injectable()
export class PaymentTasksService implements OnModuleInit {
  private readonly logger = new Logger(PaymentTasksService.name);

  constructor(private prisma: PrismaService) {}

  onModuleInit() {
    this.logger.log('Payment Tasks initialized');
  }

  /**
   * Process pending payouts to drivers
   * Runs daily at 6 AM
   */
  @Cron('0 6 * * *')
  async processDriverPayouts() {
    this.logger.debug('Running: processDriverPayouts');

    try {
      // Get drivers with pending payouts
      const drivers = await this.prisma.driver.findMany({
        where: {
          wallet_balance: { gte: 1000 }, // Minimum payout threshold
          auto_payout_enabled: true,
          payout_account_verified: true,
        },
        select: {
          id: true,
          name: true,
          wallet_balance: true,
          payout_method: true,
          payout_account: true,
        },
      });

      for (const driver of drivers) {
        try {
          // Create payout record
          await this.prisma.payout.create({
            data: {
              driver_id: driver.id,
              amount: driver.wallet_balance,
              method: driver.payout_method,
              account: driver.payout_account,
              status: 'pending',
              created_at: new Date(),
            },
          });

          // Reset driver wallet
          await this.prisma.driver.update({
            where: { id: driver.id },
            data: { wallet_balance: 0 },
          });

          this.logger.log(`Created payout for driver ${driver.id}: ${driver.wallet_balance}`);
        } catch (error) {
          this.logger.error(`Failed to process payout for driver ${driver.id}: ${error.message}`);
        }
      }
    } catch (error) {
      this.logger.error(`processDriverPayouts failed: ${error.message}`);
    }
  }

  /**
   * Check pending payment status
   * Runs every 5 minutes
   */
  @Cron(CronExpression.EVERY_5_MINUTES)
  async checkPendingPayments() {
    this.logger.debug('Running: checkPendingPayments');

    try {
      const pendingTime = new Date(Date.now() - 5 * 60 * 1000); // 5 minutes old

      const pendingPayments = await this.prisma.payment.findMany({
        where: {
          status: 'pending',
          created_at: { lt: pendingTime },
        },
        take: 50,
      });

      for (const payment of pendingPayments) {
        // TODO: Check status with payment provider
        this.logger.debug(`Checking payment status: ${payment.reference}`);
      }
    } catch (error) {
      this.logger.error(`checkPendingPayments failed: ${error.message}`);
    }
  }

  /**
   * Cancel expired payment intents
   * Runs every 30 minutes
   */
  @Cron(CronExpression.EVERY_30_MINUTES)
  async cancelExpiredPaymentIntents() {
    this.logger.debug('Running: cancelExpiredPaymentIntents');

    try {
      const expiredTime = new Date(Date.now() - 60 * 60 * 1000); // 1 hour

      const result = await this.prisma.payment.updateMany({
        where: {
          status: 'pending',
          created_at: { lt: expiredTime },
        },
        data: {
          status: 'expired',
          updated_at: new Date(),
        },
      });

      if (result.count > 0) {
        this.logger.log(`Expired ${result.count} payment intents`);
      }
    } catch (error) {
      this.logger.error(`cancelExpiredPaymentIntents failed: ${error.message}`);
    }
  }

  /**
   * Calculate merchant commissions
   * Runs daily at 1 AM
   */
  @Cron('0 1 * * *')
  async calculateMerchantCommissions() {
    this.logger.debug('Running: calculateMerchantCommissions');

    try {
      const yesterday = new Date();
      yesterday.setDate(yesterday.getDate() - 1);
      yesterday.setHours(0, 0, 0, 0);

      const today = new Date();
      today.setHours(0, 0, 0, 0);

      // Get all merchants
      const merchants = await this.prisma.merchant.findMany({
        select: { id: true, commission_rate: true },
      });

      for (const merchant of merchants) {
        // Calculate total bookings revenue
        const revenue = await this.prisma.booking.aggregate({
          where: {
            merchant_id: merchant.id,
            status: 'completed',
            payment_status: 'paid',
            ride_ended_at: {
              gte: yesterday,
              lt: today,
            },
          },
          _sum: {
            total_fare: true,
          },
        });

        const totalRevenue = revenue._sum.total_fare || 0;
        const commission = totalRevenue * (merchant.commission_rate / 100);

        if (totalRevenue > 0) {
          await this.prisma.merchantCommission.create({
            data: {
              merchant_id: merchant.id,
              date: yesterday,
              total_revenue: totalRevenue,
              commission_amount: commission,
              commission_rate: merchant.commission_rate,
              status: 'pending',
              created_at: new Date(),
            },
          });
        }
      }

      this.logger.log(`Calculated commissions for ${merchants.length} merchants`);
    } catch (error) {
      this.logger.error(`calculateMerchantCommissions failed: ${error.message}`);
    }
  }

  /**
   * Process refunds
   * Runs every 15 minutes
   */
  @Cron(CronExpression.EVERY_10_MINUTES)
  async processRefunds() {
    this.logger.debug('Running: processRefunds');

    try {
      const pendingRefunds = await this.prisma.refund.findMany({
        where: {
          status: 'approved',
          processed_at: null,
        },
        include: {
          payment: true,
        },
        take: 20,
      });

      for (const refund of pendingRefunds) {
        try {
          // TODO: Process refund with payment provider

          await this.prisma.refund.update({
            where: { id: refund.id },
            data: {
              status: 'processed',
              processed_at: new Date(),
            },
          });

          this.logger.log(`Processed refund ${refund.id}`);
        } catch (error) {
          this.logger.error(`Failed to process refund ${refund.id}: ${error.message}`);
        }
      }
    } catch (error) {
      this.logger.error(`processRefunds failed: ${error.message}`);
    }
  }

  /**
   * Expire unused promo codes
   * Runs daily at midnight
   */
  @Cron('0 0 * * *')
  async expirePromoCodes() {
    this.logger.debug('Running: expirePromoCodes');

    try {
      const result = await this.prisma.promoCode.updateMany({
        where: {
          status: 'active',
          end_date: { lt: new Date() },
        },
        data: {
          status: 'expired',
        },
      });

      if (result.count > 0) {
        this.logger.log(`Expired ${result.count} promo codes`);
      }
    } catch (error) {
      this.logger.error(`expirePromoCodes failed: ${error.message}`);
    }
  }
}
