import { Injectable, Logger, OnModuleInit } from '@nestjs/common';
import { Cron, CronExpression } from '@nestjs/schedule';
import { PrismaService } from '../../../common/prisma/prisma.service';

@Injectable()
export class CleanupTasksService implements OnModuleInit {
  private readonly logger = new Logger(CleanupTasksService.name);

  constructor(private prisma: PrismaService) {}

  onModuleInit() {
    this.logger.log('Cleanup Tasks initialized');
  }

  /**
   * Clean expired OTPs
   * Runs every 30 minutes
   */
  @Cron(CronExpression.EVERY_30_MINUTES)
  async cleanExpiredOtps() {
    this.logger.debug('Running: cleanExpiredOtps');

    try {
      const result = await this.prisma.otp.deleteMany({
        where: {
          expires_at: { lt: new Date() },
        },
      });

      if (result.count > 0) {
        this.logger.log(`Deleted ${result.count} expired OTPs`);
      }
    } catch (error) {
      this.logger.error(`cleanExpiredOtps failed: ${error.message}`);
    }
  }

  /**
   * Clean expired sessions
   * Runs daily at 2 AM
   */
  @Cron('0 2 * * *')
  async cleanExpiredSessions() {
    this.logger.debug('Running: cleanExpiredSessions');

    try {
      const result = await this.prisma.session.deleteMany({
        where: {
          expires_at: { lt: new Date() },
        },
      });

      if (result.count > 0) {
        this.logger.log(`Deleted ${result.count} expired sessions`);
      }
    } catch (error) {
      this.logger.error(`cleanExpiredSessions failed: ${error.message}`);
    }
  }

  /**
   * Clean old driver location history
   * Runs daily at 3 AM
   */
  @Cron('0 3 * * *')
  async cleanOldLocationHistory() {
    this.logger.debug('Running: cleanOldLocationHistory');

    try {
      const sevenDaysAgo = new Date();
      sevenDaysAgo.setDate(sevenDaysAgo.getDate() - 7);

      const result = await this.prisma.driverLocationHistory.deleteMany({
        where: {
          created_at: { lt: sevenDaysAgo },
        },
      });

      if (result.count > 0) {
        this.logger.log(`Deleted ${result.count} old location records`);
      }
    } catch (error) {
      this.logger.error(`cleanOldLocationHistory failed: ${error.message}`);
    }
  }

  /**
   * Clean old API logs
   * Runs weekly on Sunday at 4 AM
   */
  @Cron('0 4 * * 0')
  async cleanOldApiLogs() {
    this.logger.debug('Running: cleanOldApiLogs');

    try {
      const thirtyDaysAgo = new Date();
      thirtyDaysAgo.setDate(thirtyDaysAgo.getDate() - 30);

      const result = await this.prisma.apiLog.deleteMany({
        where: {
          created_at: { lt: thirtyDaysAgo },
        },
      });

      if (result.count > 0) {
        this.logger.log(`Deleted ${result.count} old API logs`);
      }
    } catch (error) {
      this.logger.error(`cleanOldApiLogs failed: ${error.message}`);
    }
  }

  /**
   * Clean old SMS logs
   * Runs weekly on Sunday at 4:30 AM
   */
  @Cron('30 4 * * 0')
  async cleanOldSmsLogs() {
    this.logger.debug('Running: cleanOldSmsLogs');

    try {
      const sixtyDaysAgo = new Date();
      sixtyDaysAgo.setDate(sixtyDaysAgo.getDate() - 60);

      const result = await this.prisma.smsLog.deleteMany({
        where: {
          created_at: { lt: sixtyDaysAgo },
        },
      });

      if (result.count > 0) {
        this.logger.log(`Deleted ${result.count} old SMS logs`);
      }
    } catch (error) {
      this.logger.error(`cleanOldSmsLogs failed: ${error.message}`);
    }
  }

  /**
   * Clean orphaned files
   * Runs weekly on Sunday at 5 AM
   */
  @Cron('0 5 * * 0')
  async cleanOrphanedFiles() {
    this.logger.debug('Running: cleanOrphanedFiles');

    try {
      const thirtyDaysAgo = new Date();
      thirtyDaysAgo.setDate(thirtyDaysAgo.getDate() - 30);

      // Find files not associated with any entity
      const orphanedFiles = await this.prisma.file.findMany({
        where: {
          entity_type: null,
          entity_id: null,
          created_at: { lt: thirtyDaysAgo },
        },
      });

      for (const file of orphanedFiles) {
        // TODO: Delete from storage provider
        await this.prisma.file.delete({
          where: { id: file.id },
        });
      }

      if (orphanedFiles.length > 0) {
        this.logger.log(`Deleted ${orphanedFiles.length} orphaned files`);
      }
    } catch (error) {
      this.logger.error(`cleanOrphanedFiles failed: ${error.message}`);
    }
  }

  /**
   * Clean soft-deleted records (permanent delete)
   * Runs monthly on the 1st at 6 AM
   */
  @Cron('0 6 1 * *')
  async permanentlyDeleteSoftDeleted() {
    this.logger.debug('Running: permanentlyDeleteSoftDeleted');

    try {
      const ninetyDaysAgo = new Date();
      ninetyDaysAgo.setDate(ninetyDaysAgo.getDate() - 90);

      // Permanently delete soft-deleted users
      const usersResult = await this.prisma.user.deleteMany({
        where: {
          deleted_at: { lt: ninetyDaysAgo },
        },
      });

      // Permanently delete soft-deleted drivers
      const driversResult = await this.prisma.driver.deleteMany({
        where: {
          deleted_at: { lt: ninetyDaysAgo },
        },
      });

      this.logger.log(
        `Permanently deleted: ${usersResult.count} users, ${driversResult.count} drivers`,
      );
    } catch (error) {
      this.logger.error(`permanentlyDeleteSoftDeleted failed: ${error.message}`);
    }
  }

  /**
   * Vacuum database (optimize)
   * Runs weekly on Sunday at 6 AM
   */
  @Cron('0 6 * * 0')
  async optimizeDatabase() {
    this.logger.debug('Running: optimizeDatabase');

    try {
      // This is MySQL specific - adjust for your database
      await this.prisma.$executeRaw`OPTIMIZE TABLE bookings, users, drivers, payments`;
      this.logger.log('Database optimization completed');
    } catch (error) {
      this.logger.error(`optimizeDatabase failed: ${error.message}`);
    }
  }

  /**
   * Clean expired password reset tokens
   * Runs every hour
   */
  @Cron(CronExpression.EVERY_HOUR)
  async cleanExpiredPasswordResets() {
    this.logger.debug('Running: cleanExpiredPasswordResets');

    try {
      const result = await this.prisma.passwordReset.deleteMany({
        where: {
          expires_at: { lt: new Date() },
        },
      });

      if (result.count > 0) {
        this.logger.log(`Deleted ${result.count} expired password reset tokens`);
      }
    } catch (error) {
      this.logger.error(`cleanExpiredPasswordResets failed: ${error.message}`);
    }
  }

  /**
   * Archive old bookings
   * Runs monthly on the 1st at 7 AM
   */
  @Cron('0 7 1 * *')
  async archiveOldBookings() {
    this.logger.debug('Running: archiveOldBookings');

    try {
      const oneYearAgo = new Date();
      oneYearAgo.setFullYear(oneYearAgo.getFullYear() - 1);

      // Move old bookings to archive table
      const oldBookings = await this.prisma.booking.findMany({
        where: {
          created_at: { lt: oneYearAgo },
          status: { in: ['completed', 'cancelled'] },
        },
        take: 10000,
      });

      if (oldBookings.length > 0) {
        // TODO: Insert into archive table and delete from main table
        this.logger.log(`Found ${oldBookings.length} bookings to archive`);
      }
    } catch (error) {
      this.logger.error(`archiveOldBookings failed: ${error.message}`);
    }
  }
}
