import { Injectable, Logger, OnModuleInit } from '@nestjs/common';
import { SchedulerRegistry } from '@nestjs/schedule';
import { CronJob } from 'cron';

export interface ScheduledTask {
  name: string;
  cronExpression: string;
  description: string;
  isActive: boolean;
  lastRun?: Date;
  nextRun?: Date;
  handler: () => Promise<void>;
}

@Injectable()
export class CronService implements OnModuleInit {
  private readonly logger = new Logger(CronService.name);
  private tasks: Map<string, ScheduledTask> = new Map();

  constructor(private schedulerRegistry: SchedulerRegistry) {}

  onModuleInit() {
    this.logger.log('Cron Service initialized');
    this.logScheduledJobs();
  }

  /**
   * Register a new scheduled task
   */
  registerTask(task: ScheduledTask): void {
    this.tasks.set(task.name, task);

    if (task.isActive) {
      const job = new CronJob(task.cronExpression, async () => {
        this.logger.log(`Running task: ${task.name}`);
        const startTime = Date.now();

        try {
          await task.handler();
          task.lastRun = new Date();
          this.logger.log(`Task ${task.name} completed in ${Date.now() - startTime}ms`);
        } catch (error) {
          this.logger.error(`Task ${task.name} failed: ${error.message}`);
        }
      });

      this.schedulerRegistry.addCronJob(task.name, job);
      job.start();

      this.logger.log(`Registered task: ${task.name} (${task.cronExpression})`);
    }
  }

  /**
   * Get all registered tasks
   */
  getTasks(): ScheduledTask[] {
    return Array.from(this.tasks.values()).map(task => ({
      ...task,
      nextRun: this.getNextRunTime(task.name),
    }));
  }

  /**
   * Get task by name
   */
  getTask(name: string): ScheduledTask | undefined {
    const task = this.tasks.get(name);
    if (task) {
      task.nextRun = this.getNextRunTime(name);
    }
    return task;
  }

  /**
   * Start a task
   */
  startTask(name: string): boolean {
    try {
      const job = this.schedulerRegistry.getCronJob(name);
      job.start();
      const task = this.tasks.get(name);
      if (task) task.isActive = true;
      this.logger.log(`Started task: ${name}`);
      return true;
    } catch {
      return false;
    }
  }

  /**
   * Stop a task
   */
  stopTask(name: string): boolean {
    try {
      const job = this.schedulerRegistry.getCronJob(name);
      job.stop();
      const task = this.tasks.get(name);
      if (task) task.isActive = false;
      this.logger.log(`Stopped task: ${name}`);
      return true;
    } catch {
      return false;
    }
  }

  /**
   * Run a task manually
   */
  async runTask(name: string): Promise<{ success: boolean; message?: string; duration?: number }> {
    const task = this.tasks.get(name);
    if (!task) {
      return { success: false, message: 'Task not found' };
    }

    this.logger.log(`Manually running task: ${name}`);
    const startTime = Date.now();

    try {
      await task.handler();
      task.lastRun = new Date();
      const duration = Date.now() - startTime;
      this.logger.log(`Task ${name} completed in ${duration}ms`);
      return { success: true, duration };
    } catch (error) {
      this.logger.error(`Task ${name} failed: ${error.message}`);
      return { success: false, message: error.message };
    }
  }

  /**
   * Delete a task
   */
  deleteTask(name: string): boolean {
    try {
      this.schedulerRegistry.deleteCronJob(name);
      this.tasks.delete(name);
      this.logger.log(`Deleted task: ${name}`);
      return true;
    } catch {
      return false;
    }
  }

  /**
   * Get next run time for a task
   */
  private getNextRunTime(name: string): Date | undefined {
    try {
      const job = this.schedulerRegistry.getCronJob(name);
      return job.nextDate().toJSDate();
    } catch {
      return undefined;
    }
  }

  /**
   * Log all scheduled jobs
   */
  private logScheduledJobs(): void {
    const jobs = this.schedulerRegistry.getCronJobs();
    this.logger.log(`Total scheduled jobs: ${jobs.size}`);
    jobs.forEach((job, name) => {
      this.logger.debug(`Job: ${name}, Next run: ${job.nextDate()}`);
    });
  }
}
