import {
  Controller,
  Get,
  Post,
  Put,
  Delete,
  Body,
  Param,
  Query,
  UseGuards,
  ParseIntPipe,
} from '@nestjs/common';
import { ApiTags, ApiOperation, ApiBearerAuth } from '@nestjs/swagger';
import { CorporateService } from './corporate.service';
import { JwtAuthGuard } from '../../common/guards/jwt-auth.guard';
import { MerchantGuard } from '../../common/guards/merchant.guard';
import { MerchantId, CurrentUser } from '../../common/decorators/merchant.decorator';

@ApiTags('Corporate')
@Controller('api/corporate')
@UseGuards(JwtAuthGuard, MerchantGuard)
@ApiBearerAuth()
export class CorporateController {
  constructor(private corporateService: CorporateService) {}

  // ============================================================================
  // EMPLOYEE ENDPOINTS (User who is corporate employee)
  // ============================================================================

  @Get('my-account')
  @ApiOperation({ summary: 'Get employee corporate account info' })
  async getMyAccount(
    @CurrentUser() user: any,
    @MerchantId() merchantId: number,
  ) {
    if (!user.corporate_id) {
      return { isCorporateEmployee: false };
    }

    const account = await this.corporateService.getCorporateAccount(
      user.corporate_id,
      merchantId,
    );

    return {
      isCorporateEmployee: true,
      corporate: account,
    };
  }

  @Post('check-booking')
  @ApiOperation({ summary: 'Check if employee can make corporate booking' })
  async checkBooking(
    @CurrentUser() user: any,
    @Body() body: { fareAmount: number },
  ) {
    return this.corporateService.canEmployeeBook(user.id, body.fareAmount);
  }

  // ============================================================================
  // ADMIN ENDPOINTS - Account Management
  // ============================================================================

  @Get('admin/accounts')
  @ApiOperation({ summary: 'List corporate accounts (admin)' })
  async listAccounts(
    @MerchantId() merchantId: number,
    @Query('page') page?: string,
    @Query('limit') limit?: string,
  ) {
    // TODO: Add proper list method
    return { message: 'List accounts' };
  }

  @Post('admin/accounts')
  @ApiOperation({ summary: 'Create corporate account (admin)' })
  async createAccount(
    @MerchantId() merchantId: number,
    @Body()
    body: {
      name: string;
      email: string;
      phone: string;
      address: string;
      taxId?: string;
      billingType: 'prepaid' | 'postpaid';
      creditLimit?: number;
      discountPercentage?: number;
    },
  ) {
    return this.corporateService.createCorporateAccount(merchantId, body);
  }

  @Get('admin/accounts/:id')
  @ApiOperation({ summary: 'Get corporate account details (admin)' })
  async getAccount(
    @Param('id', ParseIntPipe) id: number,
    @MerchantId() merchantId: number,
  ) {
    return this.corporateService.getCorporateAccount(id, merchantId);
  }

  @Put('admin/accounts/:id')
  @ApiOperation({ summary: 'Update corporate account (admin)' })
  async updateAccount(
    @Param('id', ParseIntPipe) id: number,
    @MerchantId() merchantId: number,
    @Body()
    body: {
      name?: string;
      email?: string;
      phone?: string;
      address?: string;
      taxId?: string;
      billingType?: 'prepaid' | 'postpaid';
      creditLimit?: number;
      discountPercentage?: number;
      isActive?: boolean;
    },
  ) {
    return this.corporateService.updateCorporateAccount(id, merchantId, body);
  }

  @Post('admin/accounts/:id/add-funds')
  @ApiOperation({ summary: 'Add funds to prepaid account (admin)' })
  async addFunds(
    @Param('id', ParseIntPipe) id: number,
    @MerchantId() merchantId: number,
    @Body() body: { amount: number; paymentReference: string },
  ) {
    return this.corporateService.addFunds(
      id,
      merchantId,
      body.amount,
      body.paymentReference,
    );
  }

  // ============================================================================
  // ADMIN ENDPOINTS - Employee Management
  // ============================================================================

  @Get('admin/accounts/:id/employees')
  @ApiOperation({ summary: 'Get corporate employees (admin)' })
  async getEmployees(
    @Param('id', ParseIntPipe) corporateId: number,
    @Query('page') page?: string,
    @Query('limit') limit?: string,
    @Query('active_only') activeOnly?: string,
  ) {
    return this.corporateService.getEmployees(corporateId, {
      page: page ? parseInt(page) : 1,
      limit: limit ? parseInt(limit) : 20,
      activeOnly: activeOnly === 'true',
    });
  }

  @Post('admin/accounts/:id/employees')
  @ApiOperation({ summary: 'Add employee to corporate account (admin)' })
  async addEmployee(
    @Param('id', ParseIntPipe) corporateId: number,
    @MerchantId() merchantId: number,
    @Body()
    body: {
      userId: number;
      employeeId?: string;
      department?: string;
      monthlyLimit?: number;
    },
  ) {
    return this.corporateService.addEmployee(corporateId, merchantId, body);
  }

  @Put('admin/accounts/:id/employees/:employeeId')
  @ApiOperation({ summary: 'Update employee monthly limit (admin)' })
  async updateEmployeeLimit(
    @Param('id', ParseIntPipe) corporateId: number,
    @Param('employeeId', ParseIntPipe) employeeId: number,
    @Body() body: { monthlyLimit: number },
  ) {
    return this.corporateService.updateEmployeeLimit(
      corporateId,
      employeeId,
      body.monthlyLimit,
    );
  }

  @Delete('admin/accounts/:id/employees/:employeeId')
  @ApiOperation({ summary: 'Remove employee from corporate (admin)' })
  async removeEmployee(
    @Param('id', ParseIntPipe) corporateId: number,
    @Param('employeeId', ParseIntPipe) employeeId: number,
    @MerchantId() merchantId: number,
  ) {
    await this.corporateService.removeEmployee(corporateId, employeeId, merchantId);
    return { message: 'Employe supprime' };
  }

  // ============================================================================
  // ADMIN ENDPOINTS - Reporting
  // ============================================================================

  @Get('admin/accounts/:id/report')
  @ApiOperation({ summary: 'Get corporate usage report (admin)' })
  async getUsageReport(
    @Param('id', ParseIntPipe) corporateId: number,
    @Query('start_date') startDate: string,
    @Query('end_date') endDate: string,
  ) {
    return this.corporateService.getUsageReport(
      corporateId,
      new Date(startDate),
      new Date(endDate),
    );
  }

  @Post('admin/accounts/:id/invoice')
  @ApiOperation({ summary: 'Generate invoice for postpaid account (admin)' })
  async generateInvoice(
    @Param('id', ParseIntPipe) corporateId: number,
    @MerchantId() merchantId: number,
    @Body() body: { startDate: string; endDate: string },
  ) {
    return this.corporateService.generateInvoice(
      corporateId,
      merchantId,
      new Date(body.startDate),
      new Date(body.endDate),
    );
  }
}
