import { Injectable, Logger, NotFoundException, BadRequestException } from '@nestjs/common';
import { PrismaService } from '../../common/prisma/prisma.service';
import { WebsocketService } from '../websocket/websocket.service';
import { QueueService } from '../queue/queue.service';

interface Message {
  id: number;
  conversationId: number;
  senderId: number;
  senderType: 'user' | 'driver' | 'support';
  content: string;
  messageType: 'text' | 'image' | 'audio' | 'location' | 'system';
  mediaUrl?: string;
  locationData?: { latitude: number; longitude: number };
  isRead: boolean;
  createdAt: Date;
}

interface Conversation {
  id: number;
  bookingId?: number;
  userId: number;
  driverId?: number;
  supportAgentId?: number;
  type: 'booking' | 'support';
  status: 'active' | 'closed';
  lastMessageAt: Date;
  unreadCount: number;
}

@Injectable()
export class ChatService {
  private readonly logger = new Logger(ChatService.name);

  constructor(
    private prisma: PrismaService,
    private websocketService: WebsocketService,
    private queueService: QueueService,
  ) {}

  // ============================================================================
  // CONVERSATION MANAGEMENT
  // ============================================================================

  /**
   * Get or create conversation for a booking
   */
  async getOrCreateBookingConversation(
    bookingId: number,
    userId: number,
    driverId: number,
  ): Promise<Conversation> {
    // Check for existing conversation
    let conversation = await this.prisma.chatConversation.findFirst({
      where: { booking_id: bookingId },
    });

    if (!conversation) {
      // Create new conversation
      conversation = await this.prisma.chatConversation.create({
        data: {
          booking_id: bookingId,
          user_id: userId,
          driver_id: driverId,
          type: 'booking',
          status: 'active',
          created_at: new Date(),
        },
      });

      this.logger.log(`Created chat conversation for booking #${bookingId}`);
    }

    return this.formatConversation(conversation);
  }

  /**
   * Create support conversation
   */
  async createSupportConversation(
    userId: number,
    userType: 'user' | 'driver',
    subject: string,
    initialMessage: string,
  ): Promise<Conversation> {
    const conversation = await this.prisma.chatConversation.create({
      data: {
        user_id: userType === 'user' ? userId : null,
        driver_id: userType === 'driver' ? userId : null,
        type: 'support',
        subject,
        status: 'active',
        created_at: new Date(),
      },
    });

    // Add initial message
    await this.sendMessage(
      conversation.id,
      userId,
      userType,
      initialMessage,
      'text',
    );

    // Add system message
    await this.addSystemMessage(
      conversation.id,
      'Votre demande a ete recue. Un agent vous repondra bientot.',
    );

    this.logger.log(`Created support conversation #${conversation.id}`);
    return this.formatConversation(conversation);
  }

  /**
   * Get user's conversations
   */
  async getUserConversations(
    userId: number,
    userType: 'user' | 'driver',
    options?: { type?: 'booking' | 'support'; status?: 'active' | 'closed' },
  ): Promise<Conversation[]> {
    const where: any = {};

    if (userType === 'user') {
      where.user_id = userId;
    } else {
      where.driver_id = userId;
    }

    if (options?.type) {
      where.type = options.type;
    }

    if (options?.status) {
      where.status = options.status;
    }

    const conversations = await this.prisma.chatConversation.findMany({
      where,
      orderBy: { last_message_at: 'desc' },
      include: {
        _count: {
          select: {
            messages: {
              where: {
                is_read: 0,
                sender_id: { not: userId },
              },
            },
          },
        },
      },
    });

    return conversations.map((c) => ({
      ...this.formatConversation(c),
      unreadCount: c._count.messages,
    }));
  }

  /**
   * Close conversation
   */
  async closeConversation(conversationId: number): Promise<void> {
    await this.prisma.chatConversation.update({
      where: { id: conversationId },
      data: {
        status: 'closed',
        closed_at: new Date(),
      },
    });

    await this.addSystemMessage(
      conversationId,
      'Cette conversation a ete fermee.',
    );
  }

  // ============================================================================
  // MESSAGE MANAGEMENT
  // ============================================================================

  /**
   * Send a message
   */
  async sendMessage(
    conversationId: number,
    senderId: number,
    senderType: 'user' | 'driver' | 'support',
    content: string,
    messageType: 'text' | 'image' | 'audio' | 'location' = 'text',
    mediaUrl?: string,
    locationData?: { latitude: number; longitude: number },
  ): Promise<Message> {
    // Verify conversation exists
    const conversation = await this.prisma.chatConversation.findUnique({
      where: { id: conversationId },
    });

    if (!conversation) {
      throw new NotFoundException('Conversation non trouvee');
    }

    if (conversation.status === 'closed') {
      throw new BadRequestException('Cette conversation est fermee');
    }

    // Create message
    const message = await this.prisma.chatMessage.create({
      data: {
        conversation_id: conversationId,
        sender_id: senderId,
        sender_type: senderType,
        content,
        message_type: messageType,
        media_url: mediaUrl,
        location_data: locationData ? JSON.stringify(locationData) : null,
        is_read: 0,
        created_at: new Date(),
      },
    });

    // Update conversation last message time
    await this.prisma.chatConversation.update({
      where: { id: conversationId },
      data: { last_message_at: new Date() },
    });

    // Send via WebSocket
    this.broadcastMessage(conversation, message, senderType, senderId);

    // Send push notification to recipient
    await this.sendMessageNotification(conversation, message, senderType, senderId);

    return this.formatMessage(message);
  }

  /**
   * Add system message
   */
  async addSystemMessage(conversationId: number, content: string): Promise<void> {
    await this.prisma.chatMessage.create({
      data: {
        conversation_id: conversationId,
        sender_id: 0,
        sender_type: 'system',
        content,
        message_type: 'system',
        is_read: 0,
        created_at: new Date(),
      },
    });
  }

  /**
   * Get conversation messages
   */
  async getMessages(
    conversationId: number,
    options?: { page?: number; limit?: number; before?: Date },
  ): Promise<{ data: Message[]; total: number }> {
    const limit = options?.limit || 50;
    const page = options?.page || 1;
    const skip = (page - 1) * limit;

    const where: any = { conversation_id: conversationId };

    if (options?.before) {
      where.created_at = { lt: options.before };
    }

    const [messages, total] = await Promise.all([
      this.prisma.chatMessage.findMany({
        where,
        skip,
        take: limit,
        orderBy: { created_at: 'desc' },
      }),
      this.prisma.chatMessage.count({ where: { conversation_id: conversationId } }),
    ]);

    return {
      data: messages.reverse().map((m) => this.formatMessage(m)),
      total,
    };
  }

  /**
   * Mark messages as read
   */
  async markAsRead(
    conversationId: number,
    readerId: number,
    readerType: 'user' | 'driver' | 'support',
  ): Promise<{ markedCount: number }> {
    const result = await this.prisma.chatMessage.updateMany({
      where: {
        conversation_id: conversationId,
        is_read: 0,
        sender_id: { not: readerId },
      },
      data: {
        is_read: 1,
        read_at: new Date(),
      },
    });

    // Notify sender that messages were read
    const conversation = await this.prisma.chatConversation.findUnique({
      where: { id: conversationId },
    });

    if (conversation) {
      const recipientId = readerType === 'user'
        ? conversation.driver_id
        : conversation.user_id;

      if (recipientId) {
        const recipientType = readerType === 'user' ? 'driver' : 'user';
        this.websocketService.sendToUser
          ? this.websocketService.sendToUser(recipientId, 'chat:messages_read', {
              conversationId,
              readBy: readerId,
              readByType: readerType,
            })
          : null;
      }
    }

    return { markedCount: result.count };
  }

  /**
   * Delete message (soft delete)
   */
  async deleteMessage(
    messageId: number,
    deleterId: number,
  ): Promise<void> {
    const message = await this.prisma.chatMessage.findFirst({
      where: { id: messageId, sender_id: deleterId },
    });

    if (!message) {
      throw new NotFoundException('Message non trouve');
    }

    await this.prisma.chatMessage.update({
      where: { id: messageId },
      data: {
        is_deleted: 1,
        deleted_at: new Date(),
        content: 'Message supprime',
      },
    });
  }

  // ============================================================================
  // QUICK REPLIES (pre-defined messages)
  // ============================================================================

  /**
   * Get quick replies
   */
  getQuickReplies(forType: 'user' | 'driver'): string[] {
    if (forType === 'user') {
      return [
        'Ou etes-vous?',
        'J\'arrive dans 2 minutes',
        'Je suis au point de rendez-vous',
        'Pouvez-vous m\'appeler?',
        'J\'ai un bagage supplementaire',
        'Merci!',
      ];
    }

    return [
      'Je suis en route',
      'Je suis arrive',
      'Pouvez-vous descendre?',
      'Quel est le numero du batiment?',
      'Je vous appelle',
      'Merci et bonne journee!',
    ];
  }

  // ============================================================================
  // HELPERS
  // ============================================================================

  /**
   * Broadcast message via WebSocket
   */
  private broadcastMessage(
    conversation: any,
    message: any,
    senderType: string,
    senderId: number,
  ): void {
    const formattedMessage = this.formatMessage(message);

    // Determine recipient
    if (senderType === 'user' && conversation.driver_id) {
      this.websocketService.sendToDriver(conversation.driver_id, 'chat:message:new', {
        conversationId: conversation.id,
        message: formattedMessage,
      });
    } else if (senderType === 'driver' && conversation.user_id) {
      this.websocketService.sendToUser(conversation.user_id, 'chat:message:new', {
        conversationId: conversation.id,
        message: formattedMessage,
      });
    }

    // Also send to booking room if applicable
    if (conversation.booking_id) {
      this.websocketService.sendToBooking(conversation.booking_id, 'chat:message:new', {
        conversationId: conversation.id,
        message: formattedMessage,
      });
    }
  }

  /**
   * Send push notification for new message
   */
  private async sendMessageNotification(
    conversation: any,
    message: any,
    senderType: string,
    senderId: number,
  ): Promise<void> {
    let recipientId: number | null = null;
    let recipientType: 'user' | 'driver' | null = null;

    if (senderType === 'user' && conversation.driver_id) {
      recipientId = conversation.driver_id;
      recipientType = 'driver';
    } else if (senderType === 'driver' && conversation.user_id) {
      recipientId = conversation.user_id;
      recipientType = 'user';
    }

    if (recipientId && recipientType) {
      // Check if recipient is not online (don't send push if they're viewing the chat)
      const isOnline = this.websocketService.isConnected(recipientId, recipientType);

      if (!isOnline) {
        await this.queueService.addNotificationJob('send-push', {
          userId: recipientId,
          userType: recipientType,
          title: 'Nouveau message',
          body: message.content.substring(0, 100),
          data: {
            type: 'chat_message',
            conversation_id: String(conversation.id),
            booking_id: conversation.booking_id ? String(conversation.booking_id) : undefined,
          },
        });
      }
    }
  }

  /**
   * Format conversation for response
   */
  private formatConversation(conversation: any): Conversation {
    return {
      id: conversation.id,
      bookingId: conversation.booking_id,
      userId: conversation.user_id,
      driverId: conversation.driver_id,
      supportAgentId: conversation.support_agent_id,
      type: conversation.type,
      status: conversation.status,
      lastMessageAt: conversation.last_message_at || conversation.created_at,
      unreadCount: 0,
    };
  }

  /**
   * Format message for response
   */
  private formatMessage(message: any): Message {
    return {
      id: message.id,
      conversationId: message.conversation_id,
      senderId: message.sender_id,
      senderType: message.sender_type,
      content: message.content,
      messageType: message.message_type,
      mediaUrl: message.media_url,
      locationData: message.location_data ? JSON.parse(message.location_data) : undefined,
      isRead: message.is_read === 1,
      createdAt: message.created_at,
    };
  }
}
