import {
  Controller,
  Get,
  Post,
  Delete,
  Body,
  Param,
  Query,
  UseGuards,
  ParseIntPipe,
} from '@nestjs/common';
import { ApiTags, ApiOperation, ApiBearerAuth } from '@nestjs/swagger';
import { ChatService } from './chat.service';
import { JwtAuthGuard } from '../../common/guards/jwt-auth.guard';
import { MerchantGuard } from '../../common/guards/merchant.guard';
import { CurrentUser } from '../../common/decorators/merchant.decorator';

@ApiTags('Chat')
@Controller('api/chat')
@UseGuards(JwtAuthGuard, MerchantGuard)
@ApiBearerAuth()
export class ChatController {
  constructor(private chatService: ChatService) {}

  // ============================================================================
  // CONVERSATION ENDPOINTS
  // ============================================================================

  @Get('conversations')
  @ApiOperation({ summary: 'Get my conversations' })
  async getConversations(
    @CurrentUser() user: any,
    @Query('type') type?: 'booking' | 'support',
    @Query('status') status?: 'active' | 'closed',
  ) {
    const userType = user.type || 'user';
    return this.chatService.getUserConversations(user.id, userType, {
      type,
      status,
    });
  }

  @Get('booking/:bookingId')
  @ApiOperation({ summary: 'Get or create conversation for a booking' })
  async getBookingConversation(
    @Param('bookingId', ParseIntPipe) bookingId: number,
    @CurrentUser() user: any,
  ) {
    // In a real implementation, we'd fetch the booking to get user and driver IDs
    // For now, assuming user is creating/accessing their own booking chat
    const userType = user.type || 'user';

    // This is a simplified version - in production, verify booking ownership
    return this.chatService.getOrCreateBookingConversation(
      bookingId,
      userType === 'user' ? user.id : 0,
      userType === 'driver' ? user.id : 0,
    );
  }

  @Post('support')
  @ApiOperation({ summary: 'Create support conversation' })
  async createSupportConversation(
    @CurrentUser() user: any,
    @Body() body: { subject: string; message: string },
  ) {
    const userType = user.type || 'user';
    return this.chatService.createSupportConversation(
      user.id,
      userType,
      body.subject,
      body.message,
    );
  }

  @Post('conversations/:id/close')
  @ApiOperation({ summary: 'Close a conversation' })
  async closeConversation(@Param('id', ParseIntPipe) conversationId: number) {
    await this.chatService.closeConversation(conversationId);
    return { message: 'Conversation fermee' };
  }

  // ============================================================================
  // MESSAGE ENDPOINTS
  // ============================================================================

  @Get('conversations/:id/messages')
  @ApiOperation({ summary: 'Get conversation messages' })
  async getMessages(
    @Param('id', ParseIntPipe) conversationId: number,
    @Query('page') page?: string,
    @Query('limit') limit?: string,
    @Query('before') before?: string,
  ) {
    return this.chatService.getMessages(conversationId, {
      page: page ? parseInt(page) : 1,
      limit: limit ? parseInt(limit) : 50,
      before: before ? new Date(before) : undefined,
    });
  }

  @Post('conversations/:id/messages')
  @ApiOperation({ summary: 'Send a message' })
  async sendMessage(
    @Param('id', ParseIntPipe) conversationId: number,
    @CurrentUser() user: any,
    @Body()
    body: {
      content: string;
      messageType?: 'text' | 'image' | 'audio' | 'location';
      mediaUrl?: string;
      locationData?: { latitude: number; longitude: number };
    },
  ) {
    const userType = user.type || 'user';
    return this.chatService.sendMessage(
      conversationId,
      user.id,
      userType,
      body.content,
      body.messageType || 'text',
      body.mediaUrl,
      body.locationData,
    );
  }

  @Post('conversations/:id/read')
  @ApiOperation({ summary: 'Mark messages as read' })
  async markAsRead(
    @Param('id', ParseIntPipe) conversationId: number,
    @CurrentUser() user: any,
  ) {
    const userType = user.type || 'user';
    return this.chatService.markAsRead(conversationId, user.id, userType);
  }

  @Delete('messages/:id')
  @ApiOperation({ summary: 'Delete a message' })
  async deleteMessage(
    @Param('id', ParseIntPipe) messageId: number,
    @CurrentUser() user: any,
  ) {
    await this.chatService.deleteMessage(messageId, user.id);
    return { message: 'Message supprime' };
  }

  // ============================================================================
  // UTILITY ENDPOINTS
  // ============================================================================

  @Get('quick-replies')
  @ApiOperation({ summary: 'Get quick reply suggestions' })
  async getQuickReplies(@CurrentUser() user: any) {
    const userType = user.type || 'user';
    return { replies: this.chatService.getQuickReplies(userType) };
  }
}
