import { Injectable, Logger, OnModuleInit, OnModuleDestroy } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import Redis from 'ioredis';

export interface CacheOptions {
  ttl?: number; // Time to live in seconds
  prefix?: string;
  tags?: string[];
}

export interface CacheStats {
  hits: number;
  misses: number;
  keys: number;
  memory: string;
}

@Injectable()
export class CacheService implements OnModuleInit, OnModuleDestroy {
  private readonly logger = new Logger(CacheService.name);
  private redis: Redis;
  private readonly defaultTTL: number;
  private readonly keyPrefix: string;

  // Stats tracking
  private hits = 0;
  private misses = 0;

  constructor(private configService: ConfigService) {
    this.defaultTTL = this.configService.get<number>('CACHE_TTL', 3600);
    this.keyPrefix = this.configService.get<string>('CACHE_PREFIX', 'monkapi:');
  }

  async onModuleInit() {
    this.redis = new Redis({
      host: this.configService.get<string>('REDIS_HOST', 'localhost'),
      port: this.configService.get<number>('REDIS_PORT', 6379),
      password: this.configService.get<string>('REDIS_PASSWORD'),
      db: this.configService.get<number>('REDIS_CACHE_DB', 1),
      keyPrefix: this.keyPrefix,
      lazyConnect: true,
      retryStrategy: (times) => {
        if (times > 3) {
          this.logger.error('Redis connection failed after 3 retries');
          return null;
        }
        return Math.min(times * 100, 3000);
      },
    });

    this.redis.on('connect', () => {
      this.logger.log('Redis cache connected');
    });

    this.redis.on('error', (error) => {
      this.logger.error(`Redis cache error: ${error.message}`);
    });

    try {
      await this.redis.connect();
    } catch (error) {
      this.logger.warn(`Redis cache connection failed: ${error.message}`);
    }
  }

  async onModuleDestroy() {
    await this.redis?.quit();
  }

  // ============================================================================
  // BASIC OPERATIONS
  // ============================================================================

  /**
   * Get value from cache
   */
  async get<T>(key: string): Promise<T | null> {
    try {
      const value = await this.redis.get(key);

      if (value) {
        this.hits++;
        return JSON.parse(value);
      }

      this.misses++;
      return null;
    } catch (error) {
      this.logger.debug(`Cache get error: ${error.message}`);
      return null;
    }
  }

  /**
   * Set value in cache
   */
  async set<T>(key: string, value: T, options?: CacheOptions): Promise<boolean> {
    try {
      const ttl = options?.ttl || this.defaultTTL;
      const serialized = JSON.stringify(value);

      if (ttl > 0) {
        await this.redis.setex(key, ttl, serialized);
      } else {
        await this.redis.set(key, serialized);
      }

      // Store tags for invalidation
      if (options?.tags?.length) {
        await this.addKeyToTags(key, options.tags);
      }

      return true;
    } catch (error) {
      this.logger.debug(`Cache set error: ${error.message}`);
      return false;
    }
  }

  /**
   * Delete value from cache
   */
  async delete(key: string): Promise<boolean> {
    try {
      await this.redis.del(key);
      return true;
    } catch (error) {
      this.logger.debug(`Cache delete error: ${error.message}`);
      return false;
    }
  }

  /**
   * Check if key exists
   */
  async exists(key: string): Promise<boolean> {
    try {
      const result = await this.redis.exists(key);
      return result === 1;
    } catch (error) {
      return false;
    }
  }

  /**
   * Get or set (cache-through)
   */
  async remember<T>(
    key: string,
    factory: () => Promise<T>,
    options?: CacheOptions,
  ): Promise<T> {
    const cached = await this.get<T>(key);

    if (cached !== null) {
      return cached;
    }

    const value = await factory();
    await this.set(key, value, options);

    return value;
  }

  /**
   * Get or set forever (no expiration)
   */
  async rememberForever<T>(
    key: string,
    factory: () => Promise<T>,
    tags?: string[],
  ): Promise<T> {
    return this.remember(key, factory, { ttl: 0, tags });
  }

  // ============================================================================
  // BULK OPERATIONS
  // ============================================================================

  /**
   * Get multiple values
   */
  async getMany<T>(keys: string[]): Promise<Map<string, T | null>> {
    try {
      const values = await this.redis.mget(...keys);
      const result = new Map<string, T | null>();

      keys.forEach((key, index) => {
        const value = values[index];
        if (value) {
          this.hits++;
          result.set(key, JSON.parse(value));
        } else {
          this.misses++;
          result.set(key, null);
        }
      });

      return result;
    } catch (error) {
      this.logger.debug(`Cache getMany error: ${error.message}`);
      return new Map();
    }
  }

  /**
   * Set multiple values
   */
  async setMany<T>(
    items: Map<string, T> | Record<string, T>,
    options?: CacheOptions,
  ): Promise<boolean> {
    try {
      const pipeline = this.redis.pipeline();
      const ttl = options?.ttl || this.defaultTTL;

      const entries = items instanceof Map ? items.entries() : Object.entries(items);

      for (const [key, value] of entries) {
        const serialized = JSON.stringify(value);
        if (ttl > 0) {
          pipeline.setex(key, ttl, serialized);
        } else {
          pipeline.set(key, serialized);
        }
      }

      await pipeline.exec();
      return true;
    } catch (error) {
      this.logger.debug(`Cache setMany error: ${error.message}`);
      return false;
    }
  }

  /**
   * Delete multiple values
   */
  async deleteMany(keys: string[]): Promise<number> {
    try {
      if (!keys.length) return 0;
      return await this.redis.del(...keys);
    } catch (error) {
      this.logger.debug(`Cache deleteMany error: ${error.message}`);
      return 0;
    }
  }

  // ============================================================================
  // TAG-BASED INVALIDATION
  // ============================================================================

  /**
   * Add key to tags
   */
  private async addKeyToTags(key: string, tags: string[]): Promise<void> {
    const pipeline = this.redis.pipeline();

    for (const tag of tags) {
      pipeline.sadd(`tag:${tag}`, key);
    }

    await pipeline.exec();
  }

  /**
   * Invalidate by tag
   */
  async invalidateTag(tag: string): Promise<number> {
    try {
      const tagKey = `tag:${tag}`;
      const keys = await this.redis.smembers(tagKey);

      if (!keys.length) {
        return 0;
      }

      const pipeline = this.redis.pipeline();
      keys.forEach((key) => pipeline.del(key));
      pipeline.del(tagKey);

      await pipeline.exec();
      return keys.length;
    } catch (error) {
      this.logger.debug(`Cache invalidateTag error: ${error.message}`);
      return 0;
    }
  }

  /**
   * Invalidate multiple tags
   */
  async invalidateTags(tags: string[]): Promise<number> {
    let total = 0;

    for (const tag of tags) {
      total += await this.invalidateTag(tag);
    }

    return total;
  }

  // ============================================================================
  // PATTERN-BASED OPERATIONS
  // ============================================================================

  /**
   * Find keys matching pattern
   */
  async keys(pattern: string): Promise<string[]> {
    try {
      return await this.redis.keys(pattern);
    } catch (error) {
      return [];
    }
  }

  /**
   * Delete keys matching pattern
   */
  async deletePattern(pattern: string): Promise<number> {
    try {
      const keys = await this.keys(pattern);

      if (!keys.length) {
        return 0;
      }

      return await this.redis.del(...keys);
    } catch (error) {
      this.logger.debug(`Cache deletePattern error: ${error.message}`);
      return 0;
    }
  }

  /**
   * Clear all cache (dangerous!)
   */
  async flush(): Promise<boolean> {
    try {
      await this.redis.flushdb();
      this.hits = 0;
      this.misses = 0;
      return true;
    } catch (error) {
      this.logger.debug(`Cache flush error: ${error.message}`);
      return false;
    }
  }

  // ============================================================================
  // ATOMIC OPERATIONS
  // ============================================================================

  /**
   * Increment value
   */
  async increment(key: string, amount: number = 1): Promise<number> {
    try {
      return await this.redis.incrby(key, amount);
    } catch (error) {
      return 0;
    }
  }

  /**
   * Decrement value
   */
  async decrement(key: string, amount: number = 1): Promise<number> {
    try {
      return await this.redis.decrby(key, amount);
    } catch (error) {
      return 0;
    }
  }

  /**
   * Set expiration time
   */
  async expire(key: string, seconds: number): Promise<boolean> {
    try {
      const result = await this.redis.expire(key, seconds);
      return result === 1;
    } catch (error) {
      return false;
    }
  }

  /**
   * Get TTL of key
   */
  async ttl(key: string): Promise<number> {
    try {
      return await this.redis.ttl(key);
    } catch (error) {
      return -2;
    }
  }

  // ============================================================================
  // HASH OPERATIONS
  // ============================================================================

  /**
   * Set hash field
   */
  async hset(key: string, field: string, value: any): Promise<boolean> {
    try {
      await this.redis.hset(key, field, JSON.stringify(value));
      return true;
    } catch (error) {
      return false;
    }
  }

  /**
   * Get hash field
   */
  async hget<T>(key: string, field: string): Promise<T | null> {
    try {
      const value = await this.redis.hget(key, field);
      return value ? JSON.parse(value) : null;
    } catch (error) {
      return null;
    }
  }

  /**
   * Get all hash fields
   */
  async hgetall<T>(key: string): Promise<Record<string, T>> {
    try {
      const hash = await this.redis.hgetall(key);
      const result: Record<string, T> = {};

      for (const [field, value] of Object.entries(hash)) {
        result[field] = JSON.parse(value);
      }

      return result;
    } catch (error) {
      return {};
    }
  }

  /**
   * Delete hash field
   */
  async hdel(key: string, field: string): Promise<boolean> {
    try {
      await this.redis.hdel(key, field);
      return true;
    } catch (error) {
      return false;
    }
  }

  // ============================================================================
  // LIST OPERATIONS
  // ============================================================================

  /**
   * Push to list (left)
   */
  async lpush(key: string, value: any): Promise<number> {
    try {
      return await this.redis.lpush(key, JSON.stringify(value));
    } catch (error) {
      return 0;
    }
  }

  /**
   * Push to list (right)
   */
  async rpush(key: string, value: any): Promise<number> {
    try {
      return await this.redis.rpush(key, JSON.stringify(value));
    } catch (error) {
      return 0;
    }
  }

  /**
   * Get list range
   */
  async lrange<T>(key: string, start: number, stop: number): Promise<T[]> {
    try {
      const values = await this.redis.lrange(key, start, stop);
      return values.map((v) => JSON.parse(v));
    } catch (error) {
      return [];
    }
  }

  /**
   * Trim list
   */
  async ltrim(key: string, start: number, stop: number): Promise<boolean> {
    try {
      await this.redis.ltrim(key, start, stop);
      return true;
    } catch (error) {
      return false;
    }
  }

  // ============================================================================
  // SET OPERATIONS
  // ============================================================================

  /**
   * Add to set
   */
  async sadd(key: string, ...members: any[]): Promise<number> {
    try {
      return await this.redis.sadd(key, ...members.map((m) => JSON.stringify(m)));
    } catch (error) {
      return 0;
    }
  }

  /**
   * Get set members
   */
  async smembers<T>(key: string): Promise<T[]> {
    try {
      const members = await this.redis.smembers(key);
      return members.map((m) => JSON.parse(m));
    } catch (error) {
      return [];
    }
  }

  /**
   * Check if member exists in set
   */
  async sismember(key: string, member: any): Promise<boolean> {
    try {
      const result = await this.redis.sismember(key, JSON.stringify(member));
      return result === 1;
    } catch (error) {
      return false;
    }
  }

  // ============================================================================
  // STATISTICS
  // ============================================================================

  /**
   * Get cache statistics
   */
  async getStats(): Promise<CacheStats> {
    try {
      const info = await this.redis.info('memory');
      const dbSize = await this.redis.dbsize();

      const memoryMatch = info.match(/used_memory_human:(\S+)/);

      return {
        hits: this.hits,
        misses: this.misses,
        keys: dbSize,
        memory: memoryMatch ? memoryMatch[1] : 'unknown',
      };
    } catch (error) {
      return {
        hits: this.hits,
        misses: this.misses,
        keys: 0,
        memory: 'unknown',
      };
    }
  }

  /**
   * Reset stats
   */
  resetStats(): void {
    this.hits = 0;
    this.misses = 0;
  }

  // ============================================================================
  // LOCK OPERATIONS
  // ============================================================================

  /**
   * Acquire lock
   */
  async lock(key: string, ttl: number = 30): Promise<boolean> {
    try {
      const lockKey = `lock:${key}`;
      const result = await this.redis.set(lockKey, '1', 'EX', ttl, 'NX');
      return result === 'OK';
    } catch (error) {
      return false;
    }
  }

  /**
   * Release lock
   */
  async unlock(key: string): Promise<boolean> {
    try {
      const lockKey = `lock:${key}`;
      await this.redis.del(lockKey);
      return true;
    } catch (error) {
      return false;
    }
  }

  /**
   * Execute with lock
   */
  async withLock<T>(
    key: string,
    factory: () => Promise<T>,
    ttl: number = 30,
    retries: number = 3,
  ): Promise<T | null> {
    let attempts = 0;

    while (attempts < retries) {
      const locked = await this.lock(key, ttl);

      if (locked) {
        try {
          return await factory();
        } finally {
          await this.unlock(key);
        }
      }

      attempts++;
      await new Promise((resolve) => setTimeout(resolve, 100 * attempts));
    }

    return null;
  }
}
