import { ApiProperty, ApiPropertyOptional } from '@nestjs/swagger';
import {
  IsNumber,
  IsString,
  IsOptional,
  IsInt,
  IsDateString,
  IsArray,
  ValidateNested,
  Min,
  Max,
} from 'class-validator';
import { Type } from 'class-transformer';

export class WaypointDto {
  @ApiProperty({ example: 6.1775 })
  @IsNumber()
  latitude: number;

  @ApiProperty({ example: 1.2364 })
  @IsNumber()
  longitude: number;

  @ApiPropertyOptional({ example: 'Stop location' })
  @IsOptional()
  @IsString()
  address?: string;
}

export class CreateBookingDto {
  // Pickup location
  @ApiProperty({ example: 6.1725, description: 'Pickup latitude' })
  @IsNumber()
  pickup_latitude: number;

  @ApiProperty({ example: 1.2314, description: 'Pickup longitude' })
  @IsNumber()
  pickup_longitude: number;

  @ApiProperty({ example: 'Lome, Togo', description: 'Pickup address' })
  @IsString()
  pickup_address: string;

  // Drop location
  @ApiProperty({ example: 6.1825, description: 'Drop latitude' })
  @IsNumber()
  drop_latitude: number;

  @ApiProperty({ example: 1.2414, description: 'Drop longitude' })
  @IsNumber()
  drop_longitude: number;

  @ApiProperty({ example: 'Tokoin, Lome', description: 'Drop address' })
  @IsString()
  drop_address: string;

  // Required fields
  @ApiProperty({ example: 1, description: 'Vehicle type ID' })
  @IsInt()
  @Min(1)
  vehicle_type_id: number;

  // Optional fields
  @ApiPropertyOptional({ example: 1, description: 'Payment method ID (1=cash, 2=card, 3=wallet, 4=mobile money)' })
  @IsOptional()
  @IsInt()
  payment_method_id?: number;

  @ApiPropertyOptional({ example: '2024-01-20T10:30:00Z', description: 'Scheduled time for ride later' })
  @IsOptional()
  @IsDateString()
  schedule_time?: string;

  @ApiPropertyOptional({ example: 1, description: 'Promo code ID' })
  @IsOptional()
  @IsInt()
  promo_code_id?: number;

  @ApiPropertyOptional({ example: 'PROMO123', description: 'Promo code string (alternative to promo_code_id)' })
  @IsOptional()
  @IsString()
  promo_code?: string;

  @ApiPropertyOptional({ example: 'Please call when you arrive', description: 'Notes for driver' })
  @IsOptional()
  @IsString()
  notes?: string;

  @ApiPropertyOptional({ example: 2, description: 'Number of passengers' })
  @IsOptional()
  @IsInt()
  @Min(1)
  @Max(10)
  number_of_passengers?: number;

  @ApiPropertyOptional({ type: [WaypointDto], description: 'Waypoints for multi-stop rides' })
  @IsOptional()
  @IsArray()
  @ValidateNested({ each: true })
  @Type(() => WaypointDto)
  waypoints?: WaypointDto[];

  @ApiPropertyOptional({ example: false, description: 'Is round trip' })
  @IsOptional()
  is_round_trip?: boolean;

  @ApiPropertyOptional({ example: 1, description: 'Corporate ID for corporate bookings' })
  @IsOptional()
  @IsInt()
  corporate_id?: number;

  @ApiPropertyOptional({ example: 1, description: 'Favorite driver ID' })
  @IsOptional()
  @IsInt()
  fav_driver_id?: number;
}
