// =============================================================================
// MonkAPI - Booking Constants (matching Laravel exactly)
// =============================================================================

/**
 * Booking Status Codes (from Laravel)
 * These match the integer values in the bookings table
 */
export const BOOKING_STATUS = {
  // Initial states
  PENDING: 1001,                    // Booking created, waiting for driver
  SEARCHING: 1002,                  // Searching for drivers

  // Driver assignment
  DRIVER_ACCEPTED: 1003,            // Driver accepted the ride
  DRIVER_ARRIVED: 1004,             // Driver arrived at pickup

  // Ride in progress
  RIDE_STARTED: 1005,               // Ride has started
  RIDE_COMPLETED: 1006,             // Ride completed

  // Cancellation
  CANCELLED_BY_USER: 1007,          // User cancelled
  CANCELLED_BY_DRIVER: 1008,        // Driver cancelled
  CANCELLED_BY_SYSTEM: 1009,        // System cancelled (timeout, etc.)

  // Special states
  NO_DRIVER_FOUND: 1010,            // No driver available
  DRIVER_ON_THE_WAY: 1011,          // Driver is heading to pickup

  // Payment states (some systems combine these)
  PAYMENT_PENDING: 1012,
  PAYMENT_COMPLETED: 1013,
  PAYMENT_FAILED: 1014,

  // Schedule booking
  SCHEDULED: 1015,                  // Future booking

  // Pool/Shared rides
  POOL_WAITING: 1016,
  POOL_MATCHED: 1017,
} as const;

export type BookingStatusType = typeof BOOKING_STATUS[keyof typeof BOOKING_STATUS];

/**
 * Booking Status Labels (for display)
 */
export const BOOKING_STATUS_LABELS: Record<number, string> = {
  [BOOKING_STATUS.PENDING]: 'En attente',
  [BOOKING_STATUS.SEARCHING]: 'Recherche de chauffeur',
  [BOOKING_STATUS.DRIVER_ACCEPTED]: 'Chauffeur accepté',
  [BOOKING_STATUS.DRIVER_ARRIVED]: 'Chauffeur arrivé',
  [BOOKING_STATUS.RIDE_STARTED]: 'Course en cours',
  [BOOKING_STATUS.RIDE_COMPLETED]: 'Course terminée',
  [BOOKING_STATUS.CANCELLED_BY_USER]: 'Annulée par utilisateur',
  [BOOKING_STATUS.CANCELLED_BY_DRIVER]: 'Annulée par chauffeur',
  [BOOKING_STATUS.CANCELLED_BY_SYSTEM]: 'Annulée par système',
  [BOOKING_STATUS.NO_DRIVER_FOUND]: 'Aucun chauffeur trouvé',
  [BOOKING_STATUS.DRIVER_ON_THE_WAY]: 'Chauffeur en route',
  [BOOKING_STATUS.PAYMENT_PENDING]: 'Paiement en attente',
  [BOOKING_STATUS.PAYMENT_COMPLETED]: 'Paiement effectué',
  [BOOKING_STATUS.PAYMENT_FAILED]: 'Paiement échoué',
  [BOOKING_STATUS.SCHEDULED]: 'Programmée',
  [BOOKING_STATUS.POOL_WAITING]: 'Pool - En attente',
  [BOOKING_STATUS.POOL_MATCHED]: 'Pool - Correspondance trouvée',
};

/**
 * Valid status transitions
 * Key = current status, Value = array of allowed next statuses
 */
export const BOOKING_STATUS_TRANSITIONS: Record<number, number[]> = {
  [BOOKING_STATUS.PENDING]: [
    BOOKING_STATUS.SEARCHING,
    BOOKING_STATUS.CANCELLED_BY_USER,
    BOOKING_STATUS.SCHEDULED,
  ],
  [BOOKING_STATUS.SEARCHING]: [
    BOOKING_STATUS.DRIVER_ACCEPTED,
    BOOKING_STATUS.NO_DRIVER_FOUND,
    BOOKING_STATUS.CANCELLED_BY_USER,
    BOOKING_STATUS.CANCELLED_BY_SYSTEM,
  ],
  [BOOKING_STATUS.DRIVER_ACCEPTED]: [
    BOOKING_STATUS.DRIVER_ON_THE_WAY,
    BOOKING_STATUS.DRIVER_ARRIVED,
    BOOKING_STATUS.CANCELLED_BY_USER,
    BOOKING_STATUS.CANCELLED_BY_DRIVER,
  ],
  [BOOKING_STATUS.DRIVER_ON_THE_WAY]: [
    BOOKING_STATUS.DRIVER_ARRIVED,
    BOOKING_STATUS.CANCELLED_BY_USER,
    BOOKING_STATUS.CANCELLED_BY_DRIVER,
  ],
  [BOOKING_STATUS.DRIVER_ARRIVED]: [
    BOOKING_STATUS.RIDE_STARTED,
    BOOKING_STATUS.CANCELLED_BY_USER,
    BOOKING_STATUS.CANCELLED_BY_DRIVER,
  ],
  [BOOKING_STATUS.RIDE_STARTED]: [
    BOOKING_STATUS.RIDE_COMPLETED,
  ],
  [BOOKING_STATUS.RIDE_COMPLETED]: [
    BOOKING_STATUS.PAYMENT_PENDING,
    BOOKING_STATUS.PAYMENT_COMPLETED,
  ],
  [BOOKING_STATUS.PAYMENT_PENDING]: [
    BOOKING_STATUS.PAYMENT_COMPLETED,
    BOOKING_STATUS.PAYMENT_FAILED,
  ],
  [BOOKING_STATUS.NO_DRIVER_FOUND]: [
    BOOKING_STATUS.SEARCHING, // Retry search
    BOOKING_STATUS.CANCELLED_BY_USER,
  ],
  [BOOKING_STATUS.SCHEDULED]: [
    BOOKING_STATUS.SEARCHING,
    BOOKING_STATUS.CANCELLED_BY_USER,
  ],
};

/**
 * Statuses that allow cancellation
 */
export const CANCELLABLE_STATUSES = [
  BOOKING_STATUS.PENDING,
  BOOKING_STATUS.SEARCHING,
  BOOKING_STATUS.DRIVER_ACCEPTED,
  BOOKING_STATUS.DRIVER_ON_THE_WAY,
  BOOKING_STATUS.DRIVER_ARRIVED,
  BOOKING_STATUS.SCHEDULED,
  BOOKING_STATUS.NO_DRIVER_FOUND,
];

/**
 * Statuses where cancellation fee applies
 */
export const CANCELLATION_FEE_STATUSES = [
  BOOKING_STATUS.DRIVER_ACCEPTED,
  BOOKING_STATUS.DRIVER_ON_THE_WAY,
  BOOKING_STATUS.DRIVER_ARRIVED,
];

/**
 * Active booking statuses (ride is in progress)
 */
export const ACTIVE_BOOKING_STATUSES = [
  BOOKING_STATUS.PENDING,
  BOOKING_STATUS.SEARCHING,
  BOOKING_STATUS.DRIVER_ACCEPTED,
  BOOKING_STATUS.DRIVER_ON_THE_WAY,
  BOOKING_STATUS.DRIVER_ARRIVED,
  BOOKING_STATUS.RIDE_STARTED,
];

/**
 * Completed booking statuses
 */
export const COMPLETED_BOOKING_STATUSES = [
  BOOKING_STATUS.RIDE_COMPLETED,
  BOOKING_STATUS.PAYMENT_COMPLETED,
];

/**
 * Payment status codes
 */
export const PAYMENT_STATUS = {
  PENDING: 0,
  COMPLETED: 1,
  FAILED: 2,
  REFUNDED: 3,
  PARTIAL: 4,
} as const;

/**
 * Payment methods
 */
export const PAYMENT_METHOD = {
  CASH: 1,
  CARD: 2,
  WALLET: 3,
  MOBILE_MONEY: 4,
  CORPORATE: 5,
} as const;

/**
 * Booking types
 */
export const BOOKING_TYPE = {
  NORMAL: 'normal',
  LATER: 'later',
  POOL: 'pool',
  RENTAL: 'rental',
  OUTSTATION: 'outstation',
  DELIVERY: 'delivery',
} as const;

/**
 * Driver search configuration
 */
export const DRIVER_SEARCH_CONFIG = {
  INITIAL_RADIUS_KM: 3,           // Start searching within 3km
  MAX_RADIUS_KM: 15,              // Maximum search radius
  RADIUS_INCREMENT_KM: 2,         // Increase radius by 2km each round
  REQUEST_TIMEOUT_SEC: 30,        // Time for driver to respond
  MAX_DRIVERS_PER_ROUND: 5,       // Max drivers to request per round
  MAX_SEARCH_ROUNDS: 5,           // Maximum search rounds
  TOTAL_SEARCH_TIMEOUT_SEC: 180,  // Total search timeout (3 minutes)
} as const;

/**
 * Check if status transition is valid
 */
export function isValidStatusTransition(currentStatus: number, newStatus: number): boolean {
  const allowedTransitions = BOOKING_STATUS_TRANSITIONS[currentStatus];
  return allowedTransitions ? allowedTransitions.includes(newStatus) : false;
}

/**
 * Check if booking can be cancelled
 */
export function canCancelBooking(status: number): boolean {
  return CANCELLABLE_STATUSES.includes(status);
}

/**
 * Check if cancellation fee applies
 */
export function shouldApplyCancellationFee(status: number): boolean {
  return CANCELLATION_FEE_STATUSES.includes(status);
}

/**
 * Get status label
 */
export function getStatusLabel(status: number): string {
  return BOOKING_STATUS_LABELS[status] || 'Inconnu';
}
