import {
  Injectable,
  UnauthorizedException,
  BadRequestException,
  ConflictException,
} from '@nestjs/common';
import { JwtService } from '@nestjs/jwt';
import { ConfigService } from '@nestjs/config';
import * as bcrypt from 'bcrypt';

import { PrismaService } from '../../shared/database/prisma.service';

// DTOs
import { UserSignupDto } from './dto/user-signup.dto';
import { UserLoginDto } from './dto/user-login.dto';
import { DriverSignupDto } from './dto/driver-signup.dto';
import { DriverLoginDto } from './dto/driver-login.dto';
import { OtpDto } from './dto/otp.dto';
import { ForgotPasswordDto } from './dto/forgot-password.dto';
import { ResetPasswordDto } from './dto/reset-password.dto';

@Injectable()
export class AuthService {
  constructor(
    private prisma: PrismaService,
    private jwtService: JwtService,
    private configService: ConfigService,
  ) {}

  // ============================================================================
  // USER AUTHENTICATION
  // ============================================================================

  async userSignup(dto: UserSignupDto, merchantId: number) {
    // Check if email already exists
    if (dto.email) {
      const existingEmail = await this.prisma.user.findFirst({
        where: {
          email: dto.email,
          merchant_id: merchantId,
          user_delete: null,
        },
      });
      if (existingEmail) {
        throw new ConflictException('Email already registered');
      }
    }

    // Check if phone already exists
    if (dto.phone) {
      const existingPhone = await this.prisma.user.findFirst({
        where: {
          UserPhone: dto.phone,
          merchant_id: merchantId,
          user_delete: null,
        },
      });
      if (existingPhone) {
        throw new ConflictException('Phone number already registered');
      }
    }

    // Hash password
    const hashedPassword = await bcrypt.hash(dto.password, 10);

    // Generate referral code
    const referralCode = this.generateReferralCode();

    // Create user
    const user = await this.prisma.user.create({
      data: {
        merchant_id: merchantId,
        first_name: dto.first_name,
        last_name: dto.last_name,
        email: dto.email,
        UserPhone: dto.phone,
        password: hashedPassword,
        referral_code: referralCode,
        device_type: dto.device_type,
        player_id: dto.player_id,
        UserStatus: 1,
        login_logout: 1,
      },
    });

    // Save device if provided
    if (dto.player_id) {
      await this.saveUserDevice(user.id, dto.device_type, dto.player_id);
    }

    // Generate tokens
    const tokens = await this.generateTokens(user.id, 'user', merchantId);

    return {
      message: 'Registration successful',
      data: {
        user: this.sanitizeUser(user),
        ...tokens,
      },
    };
  }

  async userLogin(dto: UserLoginDto, merchantId: number) {
    // Find user by email or phone
    const user = await this.prisma.user.findFirst({
      where: {
        merchant_id: merchantId,
        user_delete: null,
        OR: [
          { email: dto.email_or_phone },
          { UserPhone: dto.email_or_phone },
        ],
      },
    });

    if (!user) {
      throw new UnauthorizedException('Invalid credentials');
    }

    // Check password
    const isPasswordValid = await bcrypt.compare(dto.password, user.password);
    if (!isPasswordValid) {
      throw new UnauthorizedException('Invalid credentials');
    }

    // Check if user is active
    if (user.UserStatus !== 1) {
      throw new UnauthorizedException('Account is not active');
    }

    // Update login status
    await this.prisma.user.update({
      where: { id: user.id },
      data: {
        login_logout: 1,
        online_offline: 1,
        player_id: dto.player_id,
        device_type: dto.device_type,
      },
    });

    // Save device
    if (dto.player_id) {
      await this.saveUserDevice(user.id, dto.device_type, dto.player_id);
    }

    // Generate tokens
    const tokens = await this.generateTokens(user.id, 'user', merchantId);

    return {
      message: 'Login successful',
      data: {
        user: this.sanitizeUser(user),
        ...tokens,
      },
    };
  }

  async userLoginOtp(dto: OtpDto, merchantId: number) {
    const user = await this.prisma.user.findFirst({
      where: {
        merchant_id: merchantId,
        user_delete: null,
        OR: [
          { email: dto.email_or_phone },
          { UserPhone: dto.email_or_phone },
        ],
      },
    });

    if (!user) {
      throw new UnauthorizedException('User not found');
    }

    // Verify OTP
    if (user.otp !== dto.otp) {
      throw new BadRequestException('Invalid OTP');
    }

    // Check OTP expiry
    if (user.otp_expires_at && new Date() > user.otp_expires_at) {
      throw new BadRequestException('OTP has expired');
    }

    // Clear OTP and update login status
    await this.prisma.user.update({
      where: { id: user.id },
      data: {
        otp: null,
        otp_expires_at: null,
        login_logout: 1,
        online_offline: 1,
        player_id: dto.player_id,
        device_type: dto.device_type,
      },
    });

    // Save device
    if (dto.player_id) {
      await this.saveUserDevice(user.id, dto.device_type, dto.player_id);
    }

    // Generate tokens
    const tokens = await this.generateTokens(user.id, 'user', merchantId);

    return {
      message: 'Login successful',
      data: {
        user: this.sanitizeUser(user),
        ...tokens,
      },
    };
  }

  async userForgotPassword(dto: ForgotPasswordDto, merchantId: number) {
    const user = await this.prisma.user.findFirst({
      where: {
        merchant_id: merchantId,
        user_delete: null,
        OR: [
          { email: dto.email_or_phone },
          { UserPhone: dto.email_or_phone },
        ],
      },
    });

    if (!user) {
      // Don't reveal if user exists or not
      return { message: 'If account exists, OTP has been sent' };
    }

    // Generate OTP
    const otp = this.generateOtp();
    const otpExpiresAt = new Date(Date.now() + 10 * 60 * 1000); // 10 minutes

    await this.prisma.user.update({
      where: { id: user.id },
      data: {
        otp,
        otp_expires_at: otpExpiresAt,
      },
    });

    // TODO: Send OTP via SMS/Email
    // await this.notificationService.sendOtp(user, otp);

    return { message: 'OTP has been sent' };
  }

  async userResetPassword(dto: ResetPasswordDto, merchantId: number) {
    const user = await this.prisma.user.findFirst({
      where: {
        merchant_id: merchantId,
        user_delete: null,
        OR: [
          { email: dto.email_or_phone },
          { UserPhone: dto.email_or_phone },
        ],
      },
    });

    if (!user) {
      throw new BadRequestException('Invalid request');
    }

    // Verify OTP
    if (user.otp !== dto.otp) {
      throw new BadRequestException('Invalid OTP');
    }

    // Check OTP expiry
    if (user.otp_expires_at && new Date() > user.otp_expires_at) {
      throw new BadRequestException('OTP has expired');
    }

    // Hash new password
    const hashedPassword = await bcrypt.hash(dto.new_password, 10);

    // Update password and clear OTP
    await this.prisma.user.update({
      where: { id: user.id },
      data: {
        password: hashedPassword,
        otp: null,
        otp_expires_at: null,
      },
    });

    return { message: 'Password reset successful' };
  }

  async userLogout(userId: number) {
    await this.prisma.user.update({
      where: { id: userId },
      data: {
        login_logout: 2,
        online_offline: 2,
        player_id: null,
      },
    });

    // Deactivate all devices
    await this.prisma.userDevice.updateMany({
      where: { user_id: userId },
      data: { is_active: 0 },
    });

    return { message: 'Logout successful' };
  }

  // ============================================================================
  // DRIVER AUTHENTICATION
  // ============================================================================

  async driverSignup(dto: DriverSignupDto, merchantId: number) {
    // Check if email already exists
    if (dto.email) {
      const existingEmail = await this.prisma.driver.findFirst({
        where: {
          email: dto.email,
          merchant_id: merchantId,
        },
      });
      if (existingEmail) {
        throw new ConflictException('Email already registered');
      }
    }

    // Check if phone already exists
    if (dto.phone) {
      const existingPhone = await this.prisma.driver.findFirst({
        where: {
          phoneNumber: dto.phone,
          merchant_id: merchantId,
        },
      });
      if (existingPhone) {
        throw new ConflictException('Phone number already registered');
      }
    }

    // Hash password
    const hashedPassword = await bcrypt.hash(dto.password, 10);

    // Generate referral code
    const referralCode = this.generateReferralCode();

    // Create driver (pending approval)
    const driver = await this.prisma.driver.create({
      data: {
        merchant_id: merchantId,
        first_name: dto.first_name,
        last_name: dto.last_name,
        email: dto.email,
        phoneNumber: dto.phone,
        password: hashedPassword,
        referral_code: referralCode,
        device_type: dto.device_type,
        player_id: dto.player_id,
        driver_status: 0, // Pending approval
        is_online: 2,
        free_busy: 2,
      },
    });

    // Save device if provided
    if (dto.player_id) {
      await this.saveDriverDevice(driver.id, dto.device_type, dto.player_id);
    }

    // Generate tokens
    const tokens = await this.generateTokens(driver.id, 'driver', merchantId);

    return {
      message: 'Registration successful. Pending approval.',
      data: {
        driver: this.sanitizeDriver(driver),
        ...tokens,
      },
    };
  }

  async driverLogin(dto: DriverLoginDto, merchantId: number) {
    const driver = await this.prisma.driver.findFirst({
      where: {
        merchant_id: merchantId,
        OR: [
          { email: dto.email_or_phone },
          { phoneNumber: dto.email_or_phone },
        ],
      },
    });

    if (!driver) {
      throw new UnauthorizedException('Invalid credentials');
    }

    // Check password
    const isPasswordValid = await bcrypt.compare(dto.password, driver.password);
    if (!isPasswordValid) {
      throw new UnauthorizedException('Invalid credentials');
    }

    // Check if driver is approved
    if (driver.driver_status !== 1) {
      throw new UnauthorizedException('Account is pending approval or rejected');
    }

    // Update login status
    await this.prisma.driver.update({
      where: { id: driver.id },
      data: {
        player_id: dto.player_id,
        device_type: dto.device_type,
      },
    });

    // Save device
    if (dto.player_id) {
      await this.saveDriverDevice(driver.id, dto.device_type, dto.player_id);
    }

    // Generate tokens
    const tokens = await this.generateTokens(driver.id, 'driver', merchantId);

    return {
      message: 'Login successful',
      data: {
        driver: this.sanitizeDriver(driver),
        ...tokens,
      },
    };
  }

  async driverLoginOtp(dto: OtpDto, merchantId: number) {
    const driver = await this.prisma.driver.findFirst({
      where: {
        merchant_id: merchantId,
        OR: [
          { email: dto.email_or_phone },
          { phoneNumber: dto.email_or_phone },
        ],
      },
    });

    if (!driver) {
      throw new UnauthorizedException('Driver not found');
    }

    // Verify OTP
    if (driver.otp !== dto.otp) {
      throw new BadRequestException('Invalid OTP');
    }

    // Check OTP expiry
    if (driver.otp_expires_at && new Date() > driver.otp_expires_at) {
      throw new BadRequestException('OTP has expired');
    }

    // Clear OTP
    await this.prisma.driver.update({
      where: { id: driver.id },
      data: {
        otp: null,
        otp_expires_at: null,
        player_id: dto.player_id,
        device_type: dto.device_type,
      },
    });

    // Save device
    if (dto.player_id) {
      await this.saveDriverDevice(driver.id, dto.device_type, dto.player_id);
    }

    // Generate tokens
    const tokens = await this.generateTokens(driver.id, 'driver', merchantId);

    return {
      message: 'Login successful',
      data: {
        driver: this.sanitizeDriver(driver),
        ...tokens,
      },
    };
  }

  async driverForgotPassword(dto: ForgotPasswordDto, merchantId: number) {
    const driver = await this.prisma.driver.findFirst({
      where: {
        merchant_id: merchantId,
        OR: [
          { email: dto.email_or_phone },
          { phoneNumber: dto.email_or_phone },
        ],
      },
    });

    if (!driver) {
      return { message: 'If account exists, OTP has been sent' };
    }

    const otp = this.generateOtp();
    const otpExpiresAt = new Date(Date.now() + 10 * 60 * 1000);

    await this.prisma.driver.update({
      where: { id: driver.id },
      data: {
        otp,
        otp_expires_at: otpExpiresAt,
      },
    });

    // TODO: Send OTP via SMS/Email

    return { message: 'OTP has been sent' };
  }

  async driverResetPassword(dto: ResetPasswordDto, merchantId: number) {
    const driver = await this.prisma.driver.findFirst({
      where: {
        merchant_id: merchantId,
        OR: [
          { email: dto.email_or_phone },
          { phoneNumber: dto.email_or_phone },
        ],
      },
    });

    if (!driver) {
      throw new BadRequestException('Invalid request');
    }

    if (driver.otp !== dto.otp) {
      throw new BadRequestException('Invalid OTP');
    }

    if (driver.otp_expires_at && new Date() > driver.otp_expires_at) {
      throw new BadRequestException('OTP has expired');
    }

    const hashedPassword = await bcrypt.hash(dto.new_password, 10);

    await this.prisma.driver.update({
      where: { id: driver.id },
      data: {
        password: hashedPassword,
        otp: null,
        otp_expires_at: null,
      },
    });

    return { message: 'Password reset successful' };
  }

  async driverLogout(driverId: number) {
    await this.prisma.driver.update({
      where: { id: driverId },
      data: {
        is_online: 2,
        player_id: null,
      },
    });

    await this.prisma.driverDevice.updateMany({
      where: { driver_id: driverId },
      data: { is_active: 0 },
    });

    return { message: 'Logout successful' };
  }

  // ============================================================================
  // TOKEN MANAGEMENT
  // ============================================================================

  async refreshToken(refreshToken: string) {
    try {
      const payload = this.jwtService.verify(refreshToken, {
        secret: this.configService.get<string>('jwt.refreshSecret'),
      });

      const tokens = await this.generateTokens(
        payload.sub,
        payload.type,
        payload.merchantId,
      );

      return {
        message: 'Token refreshed',
        data: tokens,
      };
    } catch (error) {
      throw new UnauthorizedException('Invalid refresh token');
    }
  }

  private async generateTokens(
    userId: number,
    type: 'user' | 'driver',
    merchantId: number,
  ) {
    const payload = {
      sub: userId,
      type,
      merchantId,
    };

    const [accessToken, refreshToken] = await Promise.all([
      this.jwtService.signAsync(payload),
      this.jwtService.signAsync(payload, {
        secret: this.configService.get<string>('jwt.refreshSecret'),
        expiresIn: this.configService.get<string>('jwt.refreshExpiresIn'),
      }),
    ]);

    return {
      access_token: accessToken,
      refresh_token: refreshToken,
      token_type: 'Bearer',
    };
  }

  // ============================================================================
  // HELPERS
  // ============================================================================

  private generateOtp(): string {
    return Math.floor(100000 + Math.random() * 900000).toString();
  }

  private generateReferralCode(): string {
    const chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
    let code = '';
    for (let i = 0; i < 8; i++) {
      code += chars.charAt(Math.floor(Math.random() * chars.length));
    }
    return code;
  }

  private async saveUserDevice(
    userId: number,
    deviceType: number,
    playerId: string,
  ) {
    // Deactivate old devices
    await this.prisma.userDevice.updateMany({
      where: { user_id: userId },
      data: { is_active: 0 },
    });

    // Create or update device
    await this.prisma.userDevice.upsert({
      where: {
        id: 0, // Force create
      },
      create: {
        user_id: userId,
        device_type: deviceType,
        player_id: playerId,
        is_active: 1,
      },
      update: {
        device_type: deviceType,
        player_id: playerId,
        is_active: 1,
      },
    });
  }

  private async saveDriverDevice(
    driverId: number,
    deviceType: number,
    playerId: string,
  ) {
    await this.prisma.driverDevice.updateMany({
      where: { driver_id: driverId },
      data: { is_active: 0 },
    });

    await this.prisma.driverDevice.create({
      data: {
        driver_id: driverId,
        device_type: deviceType,
        player_id: playerId,
        is_active: 1,
      },
    });
  }

  private sanitizeUser(user: any) {
    const { password, otp, otp_expires_at, ...sanitized } = user;
    return sanitized;
  }

  private sanitizeDriver(driver: any) {
    const { password, otp, otp_expires_at, ...sanitized } = driver;
    return sanitized;
  }
}
