// =============================================================================
// AuthService Unit Tests
// =============================================================================

import { Test, TestingModule } from '@nestjs/testing';
import { JwtService } from '@nestjs/jwt';
import { ConfigService } from '@nestjs/config';
import { UnauthorizedException, BadRequestException } from '@nestjs/common';
import * as bcrypt from 'bcrypt';
import { AuthService } from './auth.service';
import { PrismaService } from '../../shared/database/prisma.service';
import {
  mockPrismaService,
  mockJwtService,
  mockConfigService,
  createMockUser,
  createMockDriver,
  resetAllMocks,
} from '../../../test/setup';

jest.mock('bcrypt');

describe('AuthService', () => {
  let service: AuthService;
  let prisma: typeof mockPrismaService;
  let jwt: typeof mockJwtService;

  beforeEach(async () => {
    resetAllMocks();

    const module: TestingModule = await Test.createTestingModule({
      providers: [
        AuthService,
        { provide: PrismaService, useValue: mockPrismaService },
        { provide: JwtService, useValue: mockJwtService },
        { provide: ConfigService, useValue: mockConfigService },
      ],
    }).compile();

    service = module.get<AuthService>(AuthService);
    prisma = mockPrismaService;
    jwt = mockJwtService;
  });

  afterEach(() => {
    jest.clearAllMocks();
  });

  describe('validateUser', () => {
    it('should return user without password when credentials are valid', async () => {
      const mockUser = createMockUser();
      prisma.user.findFirst.mockResolvedValue(mockUser);
      (bcrypt.compare as jest.Mock).mockResolvedValue(true);

      const result = await service.validateUser(
        'john.doe@example.com',
        'password123',
        1,
      );

      expect(result).toBeDefined();
      expect(result.id).toBe(mockUser.id);
      expect(result.password).toBeUndefined();
    });

    it('should return null when user not found', async () => {
      prisma.user.findFirst.mockResolvedValue(null);

      const result = await service.validateUser(
        'nonexistent@example.com',
        'password123',
        1,
      );

      expect(result).toBeNull();
    });

    it('should return null when password is incorrect', async () => {
      const mockUser = createMockUser();
      prisma.user.findFirst.mockResolvedValue(mockUser);
      (bcrypt.compare as jest.Mock).mockResolvedValue(false);

      const result = await service.validateUser(
        'john.doe@example.com',
        'wrongpassword',
        1,
      );

      expect(result).toBeNull();
    });
  });

  describe('validateDriver', () => {
    it('should return driver without password when credentials are valid', async () => {
      const mockDriver = createMockDriver();
      prisma.driver.findFirst.mockResolvedValue(mockDriver);
      (bcrypt.compare as jest.Mock).mockResolvedValue(true);

      const result = await service.validateDriver(
        'driver@example.com',
        'password123',
        1,
      );

      expect(result).toBeDefined();
      expect(result.id).toBe(mockDriver.id);
      expect(result.password).toBeUndefined();
    });

    it('should return null when driver not approved', async () => {
      const mockDriver = createMockDriver({ driver_status: 0 }); // pending
      prisma.driver.findFirst.mockResolvedValue(mockDriver);
      (bcrypt.compare as jest.Mock).mockResolvedValue(true);

      const result = await service.validateDriver(
        'driver@example.com',
        'password123',
        1,
      );

      expect(result).toBeNull();
    });
  });

  describe('login', () => {
    it('should return tokens for valid user', async () => {
      const mockUser = createMockUser();
      jwt.signAsync.mockResolvedValue('mock-access-token');

      const result = await service.login(mockUser, 'user');

      expect(result).toHaveProperty('accessToken');
      expect(result).toHaveProperty('refreshToken');
      expect(result).toHaveProperty('expiresIn');
      expect(result.tokenType).toBe('Bearer');
    });

    it('should return tokens for valid driver', async () => {
      const mockDriver = createMockDriver();
      jwt.signAsync.mockResolvedValue('mock-access-token');

      const result = await service.login(mockDriver, 'driver');

      expect(result).toHaveProperty('accessToken');
      expect(result.tokenType).toBe('Bearer');
    });
  });

  describe('signup', () => {
    it('should create new user successfully', async () => {
      const signupData = {
        firstName: 'New',
        lastName: 'User',
        email: 'new@example.com',
        phone: '+22890111222',
        password: 'password123',
      };

      prisma.user.findFirst.mockResolvedValue(null);
      (bcrypt.hash as jest.Mock).mockResolvedValue('hashedpassword');
      prisma.user.create.mockResolvedValue(createMockUser(signupData));
      jwt.signAsync.mockResolvedValue('mock-token');

      const result = await service.signup(signupData, 1);

      expect(result).toHaveProperty('user');
      expect(result).toHaveProperty('accessToken');
      expect(prisma.user.create).toHaveBeenCalled();
    });

    it('should throw error when email already exists', async () => {
      const signupData = {
        firstName: 'New',
        lastName: 'User',
        email: 'existing@example.com',
        phone: '+22890111222',
        password: 'password123',
      };

      prisma.user.findFirst.mockResolvedValue(createMockUser());

      await expect(service.signup(signupData, 1)).rejects.toThrow(
        BadRequestException,
      );
    });
  });

  describe('verifyOtp', () => {
    it('should verify valid OTP', async () => {
      const mockUser = createMockUser({
        otp: '123456',
        otp_expires_at: new Date(Date.now() + 10 * 60 * 1000), // 10 min from now
      });
      prisma.user.findFirst.mockResolvedValue(mockUser);
      prisma.user.update.mockResolvedValue({ ...mockUser, otp: null });
      jwt.signAsync.mockResolvedValue('mock-token');

      const result = await service.verifyOtp(
        '+22890123456',
        '123456',
        'user',
        1,
      );

      expect(result).toHaveProperty('accessToken');
      expect(prisma.user.update).toHaveBeenCalledWith(
        expect.objectContaining({
          where: { id: mockUser.id },
          data: expect.objectContaining({ otp: null }),
        }),
      );
    });

    it('should throw error for invalid OTP', async () => {
      const mockUser = createMockUser({
        otp: '123456',
        otp_expires_at: new Date(Date.now() + 10 * 60 * 1000),
      });
      prisma.user.findFirst.mockResolvedValue(mockUser);

      await expect(
        service.verifyOtp('+22890123456', '000000', 'user', 1),
      ).rejects.toThrow(UnauthorizedException);
    });

    it('should throw error for expired OTP', async () => {
      const mockUser = createMockUser({
        otp: '123456',
        otp_expires_at: new Date(Date.now() - 10 * 60 * 1000), // 10 min ago
      });
      prisma.user.findFirst.mockResolvedValue(mockUser);

      await expect(
        service.verifyOtp('+22890123456', '123456', 'user', 1),
      ).rejects.toThrow(UnauthorizedException);
    });
  });

  describe('refreshToken', () => {
    it('should return new tokens for valid refresh token', async () => {
      const payload = { sub: 1, merchantId: 1, type: 'user' };
      jwt.verifyAsync.mockResolvedValue(payload);
      prisma.user.findUnique.mockResolvedValue(createMockUser());
      jwt.signAsync.mockResolvedValue('new-token');

      const result = await service.refreshToken('valid-refresh-token');

      expect(result).toHaveProperty('accessToken');
      expect(result).toHaveProperty('refreshToken');
    });

    it('should throw error for invalid refresh token', async () => {
      jwt.verifyAsync.mockRejectedValue(new Error('Invalid token'));

      await expect(
        service.refreshToken('invalid-token'),
      ).rejects.toThrow(UnauthorizedException);
    });
  });

  describe('generateReferralCode', () => {
    it('should generate unique referral code', async () => {
      prisma.user.findFirst.mockResolvedValue(null);

      const code = await service.generateReferralCode('John');

      expect(code).toMatch(/^JOHN[A-Z0-9]{4,6}$/);
    });

    it('should retry if code already exists', async () => {
      prisma.user.findFirst
        .mockResolvedValueOnce(createMockUser()) // First code exists
        .mockResolvedValueOnce(null); // Second code is unique

      const code = await service.generateReferralCode('John');

      expect(code).toBeDefined();
      expect(prisma.user.findFirst).toHaveBeenCalledTimes(2);
    });
  });
});
