import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../common/prisma/prisma.service';
import { ConfigService } from '@nestjs/config';

export type AuditAction =
  | 'create'
  | 'update'
  | 'delete'
  | 'login'
  | 'logout'
  | 'view'
  | 'export'
  | 'import'
  | 'approve'
  | 'reject'
  | 'cancel'
  | 'complete'
  | 'assign'
  | 'payment'
  | 'refund'
  | 'suspend'
  | 'reactivate'
  | 'send_notification'
  | 'change_status'
  | 'upload'
  | 'download'
  | 'custom';

export type AuditEntity =
  | 'user'
  | 'driver'
  | 'booking'
  | 'delivery'
  | 'payment'
  | 'vehicle'
  | 'document'
  | 'promo_code'
  | 'merchant'
  | 'franchise'
  | 'corporate'
  | 'wallet'
  | 'notification'
  | 'settings'
  | 'admin'
  | 'system';

export interface AuditLogInput {
  action: AuditAction;
  entity: AuditEntity;
  entityId?: number | string;
  userId?: number;
  userType?: 'user' | 'driver' | 'admin' | 'merchant' | 'system';
  merchantId?: number;
  description?: string;
  oldValues?: Record<string, any>;
  newValues?: Record<string, any>;
  metadata?: Record<string, any>;
  ipAddress?: string;
  userAgent?: string;
  requestId?: string;
}

export interface AuditLog {
  id: number;
  action: string;
  entity: string;
  entityId?: string;
  userId?: number;
  userType?: string;
  merchantId?: number;
  description?: string;
  oldValues?: any;
  newValues?: any;
  metadata?: any;
  ipAddress?: string;
  userAgent?: string;
  requestId?: string;
  createdAt: Date;
}

export interface AuditQueryOptions {
  action?: AuditAction | AuditAction[];
  entity?: AuditEntity | AuditEntity[];
  entityId?: number | string;
  userId?: number;
  userType?: string;
  merchantId?: number;
  startDate?: Date;
  endDate?: Date;
  search?: string;
  page?: number;
  limit?: number;
  orderBy?: 'asc' | 'desc';
}

@Injectable()
export class AuditService {
  private readonly logger = new Logger(AuditService.name);
  private readonly enabled: boolean;
  private readonly retentionDays: number;

  constructor(
    private prisma: PrismaService,
    private configService: ConfigService,
  ) {
    this.enabled = this.configService.get<boolean>('AUDIT_ENABLED', true);
    this.retentionDays = this.configService.get<number>('AUDIT_RETENTION_DAYS', 90);
  }

  /**
   * Log an audit event
   */
  async log(input: AuditLogInput): Promise<AuditLog | null> {
    if (!this.enabled) {
      return null;
    }

    try {
      const log = await this.prisma.auditLog.create({
        data: {
          action: input.action,
          entity: input.entity,
          entity_id: input.entityId?.toString(),
          user_id: input.userId,
          user_type: input.userType,
          merchant_id: input.merchantId,
          description: input.description || this.generateDescription(input),
          old_values: input.oldValues ? JSON.stringify(input.oldValues) : null,
          new_values: input.newValues ? JSON.stringify(input.newValues) : null,
          metadata: input.metadata ? JSON.stringify(input.metadata) : null,
          ip_address: input.ipAddress,
          user_agent: input.userAgent,
          request_id: input.requestId,
          created_at: new Date(),
        },
      });

      return this.mapToAuditLog(log);
    } catch (error) {
      this.logger.error(`Failed to create audit log: ${error.message}`);
      return null;
    }
  }

  /**
   * Log multiple events at once
   */
  async logBatch(inputs: AuditLogInput[]): Promise<number> {
    if (!this.enabled || !inputs.length) {
      return 0;
    }

    try {
      const result = await this.prisma.auditLog.createMany({
        data: inputs.map((input) => ({
          action: input.action,
          entity: input.entity,
          entity_id: input.entityId?.toString(),
          user_id: input.userId,
          user_type: input.userType,
          merchant_id: input.merchantId,
          description: input.description || this.generateDescription(input),
          old_values: input.oldValues ? JSON.stringify(input.oldValues) : null,
          new_values: input.newValues ? JSON.stringify(input.newValues) : null,
          metadata: input.metadata ? JSON.stringify(input.metadata) : null,
          ip_address: input.ipAddress,
          user_agent: input.userAgent,
          request_id: input.requestId,
          created_at: new Date(),
        })),
      });

      return result.count;
    } catch (error) {
      this.logger.error(`Failed to create batch audit logs: ${error.message}`);
      return 0;
    }
  }

  /**
   * Query audit logs
   */
  async query(options: AuditQueryOptions): Promise<{
    data: AuditLog[];
    total: number;
    page: number;
    limit: number;
    totalPages: number;
  }> {
    const page = options.page || 1;
    const limit = Math.min(options.limit || 50, 100);
    const skip = (page - 1) * limit;

    const where: any = {};

    if (options.action) {
      where.action = Array.isArray(options.action)
        ? { in: options.action }
        : options.action;
    }

    if (options.entity) {
      where.entity = Array.isArray(options.entity)
        ? { in: options.entity }
        : options.entity;
    }

    if (options.entityId) {
      where.entity_id = options.entityId.toString();
    }

    if (options.userId) {
      where.user_id = options.userId;
    }

    if (options.userType) {
      where.user_type = options.userType;
    }

    if (options.merchantId) {
      where.merchant_id = options.merchantId;
    }

    if (options.startDate || options.endDate) {
      where.created_at = {};
      if (options.startDate) {
        where.created_at.gte = options.startDate;
      }
      if (options.endDate) {
        where.created_at.lte = options.endDate;
      }
    }

    if (options.search) {
      where.OR = [
        { description: { contains: options.search } },
        { entity_id: { contains: options.search } },
      ];
    }

    const [logs, total] = await Promise.all([
      this.prisma.auditLog.findMany({
        where,
        orderBy: { created_at: options.orderBy || 'desc' },
        skip,
        take: limit,
      }),
      this.prisma.auditLog.count({ where }),
    ]);

    return {
      data: logs.map(this.mapToAuditLog),
      total,
      page,
      limit,
      totalPages: Math.ceil(total / limit),
    };
  }

  /**
   * Get audit log by ID
   */
  async getById(id: number): Promise<AuditLog | null> {
    const log = await this.prisma.auditLog.findUnique({
      where: { id },
    });

    return log ? this.mapToAuditLog(log) : null;
  }

  /**
   * Get audit history for an entity
   */
  async getEntityHistory(
    entity: AuditEntity,
    entityId: number | string,
    limit: number = 50,
  ): Promise<AuditLog[]> {
    const logs = await this.prisma.auditLog.findMany({
      where: {
        entity,
        entity_id: entityId.toString(),
      },
      orderBy: { created_at: 'desc' },
      take: limit,
    });

    return logs.map(this.mapToAuditLog);
  }

  /**
   * Get user activity
   */
  async getUserActivity(
    userId: number,
    userType: string,
    options: { startDate?: Date; endDate?: Date; limit?: number } = {},
  ): Promise<AuditLog[]> {
    const where: any = {
      user_id: userId,
      user_type: userType,
    };

    if (options.startDate || options.endDate) {
      where.created_at = {};
      if (options.startDate) where.created_at.gte = options.startDate;
      if (options.endDate) where.created_at.lte = options.endDate;
    }

    const logs = await this.prisma.auditLog.findMany({
      where,
      orderBy: { created_at: 'desc' },
      take: options.limit || 100,
    });

    return logs.map(this.mapToAuditLog);
  }

  /**
   * Get audit statistics
   */
  async getStatistics(
    merchantId?: number,
    startDate?: Date,
    endDate?: Date,
  ): Promise<{
    totalLogs: number;
    byAction: Record<string, number>;
    byEntity: Record<string, number>;
    byUserType: Record<string, number>;
    recentActivity: AuditLog[];
  }> {
    const where: any = {};
    if (merchantId) where.merchant_id = merchantId;
    if (startDate || endDate) {
      where.created_at = {};
      if (startDate) where.created_at.gte = startDate;
      if (endDate) where.created_at.lte = endDate;
    }

    const [totalLogs, actionGroups, entityGroups, userTypeGroups, recentLogs] =
      await Promise.all([
        this.prisma.auditLog.count({ where }),
        this.prisma.auditLog.groupBy({
          by: ['action'],
          where,
          _count: true,
        }),
        this.prisma.auditLog.groupBy({
          by: ['entity'],
          where,
          _count: true,
        }),
        this.prisma.auditLog.groupBy({
          by: ['user_type'],
          where,
          _count: true,
        }),
        this.prisma.auditLog.findMany({
          where,
          orderBy: { created_at: 'desc' },
          take: 10,
        }),
      ]);

    return {
      totalLogs,
      byAction: actionGroups.reduce(
        (acc, g) => ({ ...acc, [g.action]: g._count }),
        {},
      ),
      byEntity: entityGroups.reduce(
        (acc, g) => ({ ...acc, [g.entity]: g._count }),
        {},
      ),
      byUserType: userTypeGroups.reduce(
        (acc, g) => ({ ...acc, [g.user_type || 'unknown']: g._count }),
        {},
      ),
      recentActivity: recentLogs.map(this.mapToAuditLog),
    };
  }

  /**
   * Clean up old audit logs
   */
  async cleanup(daysToKeep?: number): Promise<number> {
    const retentionDays = daysToKeep || this.retentionDays;
    const cutoffDate = new Date();
    cutoffDate.setDate(cutoffDate.getDate() - retentionDays);

    const result = await this.prisma.auditLog.deleteMany({
      where: {
        created_at: { lt: cutoffDate },
      },
    });

    this.logger.log(`Cleaned up ${result.count} audit logs older than ${retentionDays} days`);
    return result.count;
  }

  /**
   * Export audit logs
   */
  async export(
    options: AuditQueryOptions,
    format: 'json' | 'csv' = 'json',
  ): Promise<string> {
    const { data } = await this.query({ ...options, limit: 10000 });

    if (format === 'csv') {
      const headers = [
        'ID',
        'Action',
        'Entity',
        'Entity ID',
        'User ID',
        'User Type',
        'Description',
        'IP Address',
        'Created At',
      ];

      let csv = headers.join(',') + '\n';
      for (const log of data) {
        csv += [
          log.id,
          log.action,
          log.entity,
          log.entityId || '',
          log.userId || '',
          log.userType || '',
          `"${(log.description || '').replace(/"/g, '""')}"`,
          log.ipAddress || '',
          log.createdAt.toISOString(),
        ].join(',') + '\n';
      }

      return csv;
    }

    return JSON.stringify(data, null, 2);
  }

  // ============================================================================
  // CONVENIENCE METHODS
  // ============================================================================

  async logCreate(
    entity: AuditEntity,
    entityId: number | string,
    newValues: Record<string, any>,
    context: Partial<AuditLogInput>,
  ): Promise<AuditLog | null> {
    return this.log({
      action: 'create',
      entity,
      entityId,
      newValues,
      ...context,
    });
  }

  async logUpdate(
    entity: AuditEntity,
    entityId: number | string,
    oldValues: Record<string, any>,
    newValues: Record<string, any>,
    context: Partial<AuditLogInput>,
  ): Promise<AuditLog | null> {
    return this.log({
      action: 'update',
      entity,
      entityId,
      oldValues,
      newValues,
      ...context,
    });
  }

  async logDelete(
    entity: AuditEntity,
    entityId: number | string,
    oldValues: Record<string, any>,
    context: Partial<AuditLogInput>,
  ): Promise<AuditLog | null> {
    return this.log({
      action: 'delete',
      entity,
      entityId,
      oldValues,
      ...context,
    });
  }

  async logLogin(
    userId: number,
    userType: 'user' | 'driver' | 'admin',
    context: Partial<AuditLogInput>,
  ): Promise<AuditLog | null> {
    return this.log({
      action: 'login',
      entity: userType === 'admin' ? 'admin' : userType,
      entityId: userId,
      userId,
      userType,
      ...context,
    });
  }

  async logLogout(
    userId: number,
    userType: 'user' | 'driver' | 'admin',
    context: Partial<AuditLogInput>,
  ): Promise<AuditLog | null> {
    return this.log({
      action: 'logout',
      entity: userType === 'admin' ? 'admin' : userType,
      entityId: userId,
      userId,
      userType,
      ...context,
    });
  }

  async logPayment(
    paymentId: number | string,
    action: 'payment' | 'refund',
    metadata: Record<string, any>,
    context: Partial<AuditLogInput>,
  ): Promise<AuditLog | null> {
    return this.log({
      action,
      entity: 'payment',
      entityId: paymentId,
      metadata,
      ...context,
    });
  }

  async logStatusChange(
    entity: AuditEntity,
    entityId: number | string,
    oldStatus: string,
    newStatus: string,
    context: Partial<AuditLogInput>,
  ): Promise<AuditLog | null> {
    return this.log({
      action: 'change_status',
      entity,
      entityId,
      oldValues: { status: oldStatus },
      newValues: { status: newStatus },
      description: `Status changed from ${oldStatus} to ${newStatus}`,
      ...context,
    });
  }

  // ============================================================================
  // PRIVATE METHODS
  // ============================================================================

  private generateDescription(input: AuditLogInput): string {
    const entityName = input.entity.replace(/_/g, ' ');
    const actionVerb = this.getActionVerb(input.action);

    if (input.entityId) {
      return `${actionVerb} ${entityName} #${input.entityId}`;
    }

    return `${actionVerb} ${entityName}`;
  }

  private getActionVerb(action: AuditAction): string {
    const verbs: Record<AuditAction, string> = {
      create: 'Created',
      update: 'Updated',
      delete: 'Deleted',
      login: 'Logged in',
      logout: 'Logged out',
      view: 'Viewed',
      export: 'Exported',
      import: 'Imported',
      approve: 'Approved',
      reject: 'Rejected',
      cancel: 'Cancelled',
      complete: 'Completed',
      assign: 'Assigned',
      payment: 'Processed payment for',
      refund: 'Refunded',
      suspend: 'Suspended',
      reactivate: 'Reactivated',
      send_notification: 'Sent notification to',
      change_status: 'Changed status of',
      upload: 'Uploaded',
      download: 'Downloaded',
      custom: 'Performed action on',
    };

    return verbs[action] || 'Performed action on';
  }

  private mapToAuditLog(log: any): AuditLog {
    return {
      id: log.id,
      action: log.action,
      entity: log.entity,
      entityId: log.entity_id,
      userId: log.user_id,
      userType: log.user_type,
      merchantId: log.merchant_id,
      description: log.description,
      oldValues: log.old_values ? JSON.parse(log.old_values) : null,
      newValues: log.new_values ? JSON.parse(log.new_values) : null,
      metadata: log.metadata ? JSON.parse(log.metadata) : null,
      ipAddress: log.ip_address,
      userAgent: log.user_agent,
      requestId: log.request_id,
      createdAt: log.created_at,
    };
  }
}
