import {
  Injectable,
  NestInterceptor,
  ExecutionContext,
  CallHandler,
} from '@nestjs/common';
import { Observable, tap } from 'rxjs';
import { Reflector } from '@nestjs/core';
import { AuditService, AuditAction, AuditEntity } from './audit.service';
import { v4 as uuidv4 } from 'uuid';

export const AUDIT_KEY = 'audit';

export interface AuditMetadata {
  action: AuditAction;
  entity: AuditEntity;
  getEntityId?: (req: any, result: any) => number | string | undefined;
  getOldValues?: (req: any) => Record<string, any> | undefined;
  getNewValues?: (req: any, result: any) => Record<string, any> | undefined;
  getDescription?: (req: any, result: any) => string | undefined;
  skip?: (req: any, result: any) => boolean;
}

export function Audit(metadata: AuditMetadata): MethodDecorator {
  return (target, propertyKey, descriptor) => {
    Reflect.defineMetadata(AUDIT_KEY, metadata, descriptor.value as object);
    return descriptor;
  };
}

@Injectable()
export class AuditInterceptor implements NestInterceptor {
  constructor(
    private auditService: AuditService,
    private reflector: Reflector,
  ) {}

  intercept(context: ExecutionContext, next: CallHandler): Observable<any> {
    const metadata = this.reflector.get<AuditMetadata>(
      AUDIT_KEY,
      context.getHandler(),
    );

    if (!metadata) {
      return next.handle();
    }

    const request = context.switchToHttp().getRequest();
    const requestId = uuidv4();
    request.requestId = requestId;

    const startTime = Date.now();

    return next.handle().pipe(
      tap({
        next: async (result) => {
          // Check if we should skip logging
          if (metadata.skip && metadata.skip(request, result)) {
            return;
          }

          try {
            await this.auditService.log({
              action: metadata.action,
              entity: metadata.entity,
              entityId: metadata.getEntityId
                ? metadata.getEntityId(request, result)
                : request.params?.id,
              userId: request.user?.id || request.user?.userId,
              userType: this.getUserType(request),
              merchantId: request.merchantId,
              description: metadata.getDescription
                ? metadata.getDescription(request, result)
                : undefined,
              oldValues: metadata.getOldValues
                ? metadata.getOldValues(request)
                : undefined,
              newValues: metadata.getNewValues
                ? metadata.getNewValues(request, result)
                : request.body,
              metadata: {
                method: request.method,
                path: request.path,
                duration: Date.now() - startTime,
                statusCode: context.switchToHttp().getResponse().statusCode,
              },
              ipAddress: this.getClientIp(request),
              userAgent: request.headers['user-agent'],
              requestId,
            });
          } catch (error) {
            // Don't fail the request if audit logging fails
            console.error('Audit logging failed:', error.message);
          }
        },
        error: async (error) => {
          // Log failed requests too
          try {
            await this.auditService.log({
              action: metadata.action,
              entity: metadata.entity,
              entityId: metadata.getEntityId
                ? metadata.getEntityId(request, null)
                : request.params?.id,
              userId: request.user?.id || request.user?.userId,
              userType: this.getUserType(request),
              merchantId: request.merchantId,
              description: `Failed: ${error.message}`,
              metadata: {
                method: request.method,
                path: request.path,
                duration: Date.now() - startTime,
                error: error.message,
                statusCode: error.status || 500,
              },
              ipAddress: this.getClientIp(request),
              userAgent: request.headers['user-agent'],
              requestId,
            });
          } catch (auditError) {
            console.error('Audit logging failed:', auditError.message);
          }
        },
      }),
    );
  }

  private getUserType(request: any): 'user' | 'driver' | 'admin' | 'system' {
    if (request.user?.role === 'admin') return 'admin';
    if (request.user?.driverId) return 'driver';
    if (request.user?.userId) return 'user';
    return 'system';
  }

  private getClientIp(request: any): string {
    const forwardedFor = request.headers['x-forwarded-for'];
    if (forwardedFor) {
      return forwardedFor.split(',')[0].trim();
    }
    return request.ip || request.connection?.remoteAddress || 'unknown';
  }
}
