import {
  Controller,
  Get,
  Post,
  Query,
  Param,
  Res,
  UseGuards,
  ParseIntPipe,
} from '@nestjs/common';
import { Response } from 'express';
import { AuditService, AuditAction, AuditEntity, AuditQueryOptions } from './audit.service';
import { JwtAuthGuard } from '../../common/guards/jwt-auth.guard';
import { MerchantId } from '../../common/decorators/merchant.decorator';

@Controller('admin/audit')
@UseGuards(JwtAuthGuard)
export class AuditController {
  constructor(private readonly auditService: AuditService) {}

  /**
   * Query audit logs
   */
  @Get()
  async queryLogs(
    @MerchantId() merchantId: number,
    @Query('action') action?: string,
    @Query('entity') entity?: string,
    @Query('entityId') entityId?: string,
    @Query('userId') userId?: string,
    @Query('userType') userType?: string,
    @Query('startDate') startDate?: string,
    @Query('endDate') endDate?: string,
    @Query('search') search?: string,
    @Query('page') page?: string,
    @Query('limit') limit?: string,
    @Query('order') order?: 'asc' | 'desc',
  ) {
    const options: AuditQueryOptions = {
      merchantId,
      page: page ? parseInt(page) : 1,
      limit: limit ? parseInt(limit) : 50,
      orderBy: order || 'desc',
    };

    if (action) {
      options.action = action.split(',') as AuditAction[];
    }

    if (entity) {
      options.entity = entity.split(',') as AuditEntity[];
    }

    if (entityId) {
      options.entityId = entityId;
    }

    if (userId) {
      options.userId = parseInt(userId);
    }

    if (userType) {
      options.userType = userType;
    }

    if (startDate) {
      options.startDate = new Date(startDate);
    }

    if (endDate) {
      options.endDate = new Date(endDate);
    }

    if (search) {
      options.search = search;
    }

    return this.auditService.query(options);
  }

  /**
   * Get audit log by ID
   */
  @Get(':id')
  async getLog(@Param('id', ParseIntPipe) id: number) {
    const log = await this.auditService.getById(id);

    if (!log) {
      return { error: 'Audit log not found' };
    }

    return log;
  }

  /**
   * Get entity history
   */
  @Get('entity/:entity/:entityId')
  async getEntityHistory(
    @Param('entity') entity: AuditEntity,
    @Param('entityId') entityId: string,
    @Query('limit') limit?: string,
  ) {
    const logs = await this.auditService.getEntityHistory(
      entity,
      entityId,
      limit ? parseInt(limit) : 50,
    );

    return {
      entity,
      entityId,
      count: logs.length,
      history: logs,
    };
  }

  /**
   * Get user activity
   */
  @Get('user/:userType/:userId')
  async getUserActivity(
    @Param('userType') userType: string,
    @Param('userId', ParseIntPipe) userId: number,
    @Query('startDate') startDate?: string,
    @Query('endDate') endDate?: string,
    @Query('limit') limit?: string,
  ) {
    const logs = await this.auditService.getUserActivity(userId, userType, {
      startDate: startDate ? new Date(startDate) : undefined,
      endDate: endDate ? new Date(endDate) : undefined,
      limit: limit ? parseInt(limit) : 100,
    });

    return {
      userId,
      userType,
      count: logs.length,
      activity: logs,
    };
  }

  /**
   * Get audit statistics
   */
  @Get('stats/summary')
  async getStatistics(
    @MerchantId() merchantId: number,
    @Query('startDate') startDate?: string,
    @Query('endDate') endDate?: string,
  ) {
    return this.auditService.getStatistics(
      merchantId,
      startDate ? new Date(startDate) : undefined,
      endDate ? new Date(endDate) : undefined,
    );
  }

  /**
   * Export audit logs
   */
  @Get('export')
  async exportLogs(
    @MerchantId() merchantId: number,
    @Query('format') format: 'json' | 'csv' = 'json',
    @Query('action') action?: string,
    @Query('entity') entity?: string,
    @Query('startDate') startDate?: string,
    @Query('endDate') endDate?: string,
    @Res() res: Response,
  ) {
    const options: AuditQueryOptions = {
      merchantId,
    };

    if (action) {
      options.action = action.split(',') as AuditAction[];
    }

    if (entity) {
      options.entity = entity.split(',') as AuditEntity[];
    }

    if (startDate) {
      options.startDate = new Date(startDate);
    }

    if (endDate) {
      options.endDate = new Date(endDate);
    }

    const content = await this.auditService.export(options, format);

    const filename = `audit_logs_${new Date().toISOString().split('T')[0]}.${format}`;
    const contentType = format === 'csv' ? 'text/csv' : 'application/json';

    res.setHeader('Content-Type', contentType);
    res.setHeader('Content-Disposition', `attachment; filename="${filename}"`);
    res.send(content);
  }

  /**
   * Cleanup old audit logs
   */
  @Post('cleanup')
  async cleanupLogs(@Query('days') days?: string) {
    const daysToKeep = days ? parseInt(days) : undefined;
    const deletedCount = await this.auditService.cleanup(daysToKeep);

    return {
      success: true,
      deletedCount,
      message: `Deleted ${deletedCount} audit logs`,
    };
  }

  /**
   * Get available actions
   */
  @Get('meta/actions')
  getActions() {
    return {
      actions: [
        'create',
        'update',
        'delete',
        'login',
        'logout',
        'view',
        'export',
        'import',
        'approve',
        'reject',
        'cancel',
        'complete',
        'assign',
        'payment',
        'refund',
        'suspend',
        'reactivate',
        'send_notification',
        'change_status',
        'upload',
        'download',
        'custom',
      ],
    };
  }

  /**
   * Get available entities
   */
  @Get('meta/entities')
  getEntities() {
    return {
      entities: [
        'user',
        'driver',
        'booking',
        'delivery',
        'payment',
        'vehicle',
        'document',
        'promo_code',
        'merchant',
        'franchise',
        'corporate',
        'wallet',
        'notification',
        'settings',
        'admin',
        'system',
      ],
    };
  }
}
