import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../common/prisma/prisma.service';
import { ConfigService } from '@nestjs/config';
import * as crypto from 'crypto';

export type ApiKeyPermission =
  | 'read:bookings'
  | 'write:bookings'
  | 'read:drivers'
  | 'write:drivers'
  | 'read:users'
  | 'write:users'
  | 'read:payments'
  | 'write:payments'
  | 'read:analytics'
  | 'read:settings'
  | 'write:settings'
  | 'webhooks'
  | '*';

export interface ApiKey {
  id: number;
  merchantId: number;
  name: string;
  keyPrefix: string;
  hashedKey: string;
  permissions: ApiKeyPermission[];
  rateLimit: number;
  rateLimitWindow: number;
  ipWhitelist: string[];
  expiresAt?: Date;
  lastUsedAt?: Date;
  usageCount: number;
  active: boolean;
  createdAt: Date;
  updatedAt: Date;
}

export interface ApiKeyUsage {
  id: number;
  apiKeyId: number;
  endpoint: string;
  method: string;
  statusCode: number;
  ipAddress: string;
  userAgent?: string;
  responseTime: number;
  createdAt: Date;
}

export interface CreateApiKeyInput {
  name: string;
  permissions: ApiKeyPermission[];
  rateLimit?: number;
  rateLimitWindow?: number;
  ipWhitelist?: string[];
  expiresAt?: Date;
}

@Injectable()
export class ApiKeyService {
  private readonly logger = new Logger(ApiKeyService.name);
  private readonly keyCache: Map<string, { key: ApiKey; expires: number }> = new Map();
  private readonly cacheTTL = 60000; // 1 minute

  constructor(
    private prisma: PrismaService,
    private configService: ConfigService,
  ) {}

  // ============================================================================
  // KEY MANAGEMENT
  // ============================================================================

  /**
   * Create new API key
   */
  async createKey(
    merchantId: number,
    input: CreateApiKeyInput,
  ): Promise<{ apiKey: ApiKey; rawKey: string }> {
    const rawKey = this.generateKey();
    const keyPrefix = rawKey.substring(0, 8);
    const hashedKey = this.hashKey(rawKey);

    const key = await this.prisma.apiKey.create({
      data: {
        merchant_id: merchantId,
        name: input.name,
        key_prefix: keyPrefix,
        hashed_key: hashedKey,
        permissions: JSON.stringify(input.permissions),
        rate_limit: input.rateLimit || 1000,
        rate_limit_window: input.rateLimitWindow || 3600,
        ip_whitelist: input.ipWhitelist ? JSON.stringify(input.ipWhitelist) : null,
        expires_at: input.expiresAt,
        active: true,
        usage_count: 0,
        created_at: new Date(),
        updated_at: new Date(),
      },
    });

    return {
      apiKey: this.mapToApiKey(key),
      rawKey: `mk_${rawKey}`,
    };
  }

  /**
   * Validate API key
   */
  async validateKey(rawKey: string): Promise<ApiKey | null> {
    // Remove prefix if present
    const key = rawKey.replace(/^mk_/, '');

    // Check cache first
    const cached = this.keyCache.get(key);
    if (cached && cached.expires > Date.now()) {
      return cached.key;
    }

    const keyPrefix = key.substring(0, 8);
    const hashedKey = this.hashKey(key);

    const apiKey = await this.prisma.apiKey.findFirst({
      where: {
        key_prefix: keyPrefix,
        hashed_key: hashedKey,
        active: true,
      },
    });

    if (!apiKey) {
      return null;
    }

    // Check expiration
    if (apiKey.expires_at && apiKey.expires_at < new Date()) {
      return null;
    }

    const mappedKey = this.mapToApiKey(apiKey);

    // Update cache
    this.keyCache.set(key, {
      key: mappedKey,
      expires: Date.now() + this.cacheTTL,
    });

    // Update last used (non-blocking)
    this.prisma.apiKey.update({
      where: { id: apiKey.id },
      data: {
        last_used_at: new Date(),
        usage_count: { increment: 1 },
      },
    }).catch(() => {});

    return mappedKey;
  }

  /**
   * Check if key has permission
   */
  hasPermission(apiKey: ApiKey, permission: ApiKeyPermission): boolean {
    if (apiKey.permissions.includes('*')) {
      return true;
    }
    return apiKey.permissions.includes(permission);
  }

  /**
   * Check if IP is allowed
   */
  isIpAllowed(apiKey: ApiKey, ipAddress: string): boolean {
    if (!apiKey.ipWhitelist?.length) {
      return true;
    }

    return apiKey.ipWhitelist.some((allowed) => {
      if (allowed.includes('/')) {
        // CIDR notation
        return this.ipInCidr(ipAddress, allowed);
      }
      return allowed === ipAddress;
    });
  }

  /**
   * Get API key by ID
   */
  async getKey(keyId: number, merchantId: number): Promise<ApiKey | null> {
    const key = await this.prisma.apiKey.findFirst({
      where: { id: keyId, merchant_id: merchantId },
    });

    return key ? this.mapToApiKey(key) : null;
  }

  /**
   * List API keys for merchant
   */
  async listKeys(merchantId: number): Promise<ApiKey[]> {
    const keys = await this.prisma.apiKey.findMany({
      where: { merchant_id: merchantId },
      orderBy: { created_at: 'desc' },
    });

    return keys.map(this.mapToApiKey);
  }

  /**
   * Update API key
   */
  async updateKey(
    keyId: number,
    merchantId: number,
    data: Partial<{
      name: string;
      permissions: ApiKeyPermission[];
      rateLimit: number;
      rateLimitWindow: number;
      ipWhitelist: string[];
      expiresAt: Date | null;
      active: boolean;
    }>,
  ): Promise<ApiKey | null> {
    const updateData: any = { updated_at: new Date() };

    if (data.name) updateData.name = data.name;
    if (data.permissions) updateData.permissions = JSON.stringify(data.permissions);
    if (data.rateLimit !== undefined) updateData.rate_limit = data.rateLimit;
    if (data.rateLimitWindow !== undefined) updateData.rate_limit_window = data.rateLimitWindow;
    if (data.ipWhitelist !== undefined) {
      updateData.ip_whitelist = data.ipWhitelist.length
        ? JSON.stringify(data.ipWhitelist)
        : null;
    }
    if (data.expiresAt !== undefined) updateData.expires_at = data.expiresAt;
    if (typeof data.active === 'boolean') updateData.active = data.active;

    const result = await this.prisma.apiKey.updateMany({
      where: { id: keyId, merchant_id: merchantId },
      data: updateData,
    });

    if (result.count === 0) return null;

    // Clear cache
    this.clearKeyCache(keyId);

    return this.getKey(keyId, merchantId);
  }

  /**
   * Revoke API key
   */
  async revokeKey(keyId: number, merchantId: number): Promise<boolean> {
    const result = await this.prisma.apiKey.updateMany({
      where: { id: keyId, merchant_id: merchantId },
      data: { active: false, updated_at: new Date() },
    });

    if (result.count > 0) {
      this.clearKeyCache(keyId);
      return true;
    }

    return false;
  }

  /**
   * Delete API key
   */
  async deleteKey(keyId: number, merchantId: number): Promise<boolean> {
    const result = await this.prisma.apiKey.deleteMany({
      where: { id: keyId, merchant_id: merchantId },
    });

    if (result.count > 0) {
      this.clearKeyCache(keyId);
      return true;
    }

    return false;
  }

  /**
   * Regenerate API key
   */
  async regenerateKey(
    keyId: number,
    merchantId: number,
  ): Promise<{ apiKey: ApiKey; rawKey: string } | null> {
    const existingKey = await this.getKey(keyId, merchantId);

    if (!existingKey) {
      return null;
    }

    const rawKey = this.generateKey();
    const keyPrefix = rawKey.substring(0, 8);
    const hashedKey = this.hashKey(rawKey);

    await this.prisma.apiKey.update({
      where: { id: keyId },
      data: {
        key_prefix: keyPrefix,
        hashed_key: hashedKey,
        updated_at: new Date(),
      },
    });

    this.clearKeyCache(keyId);

    const updatedKey = await this.getKey(keyId, merchantId);

    return {
      apiKey: updatedKey!,
      rawKey: `mk_${rawKey}`,
    };
  }

  // ============================================================================
  // USAGE TRACKING
  // ============================================================================

  /**
   * Log API key usage
   */
  async logUsage(
    apiKeyId: number,
    data: {
      endpoint: string;
      method: string;
      statusCode: number;
      ipAddress: string;
      userAgent?: string;
      responseTime: number;
    },
  ): Promise<void> {
    await this.prisma.apiKeyUsage.create({
      data: {
        api_key_id: apiKeyId,
        endpoint: data.endpoint,
        method: data.method,
        status_code: data.statusCode,
        ip_address: data.ipAddress,
        user_agent: data.userAgent,
        response_time: data.responseTime,
        created_at: new Date(),
      },
    });
  }

  /**
   * Get usage statistics
   */
  async getUsageStats(
    keyId: number,
    merchantId: number,
    startDate?: Date,
    endDate?: Date,
  ): Promise<{
    totalRequests: number;
    successfulRequests: number;
    failedRequests: number;
    avgResponseTime: number;
    byEndpoint: Record<string, number>;
    byDay: Record<string, number>;
    topIPs: Array<{ ip: string; count: number }>;
  }> {
    // Verify key belongs to merchant
    const key = await this.getKey(keyId, merchantId);
    if (!key) {
      return {
        totalRequests: 0,
        successfulRequests: 0,
        failedRequests: 0,
        avgResponseTime: 0,
        byEndpoint: {},
        byDay: {},
        topIPs: [],
      };
    }

    const where: any = { api_key_id: keyId };
    if (startDate || endDate) {
      where.created_at = {};
      if (startDate) where.created_at.gte = startDate;
      if (endDate) where.created_at.lte = endDate;
    }

    const [total, successful, failed, avgTime, endpointGroups, ipGroups] =
      await Promise.all([
        this.prisma.apiKeyUsage.count({ where }),
        this.prisma.apiKeyUsage.count({
          where: { ...where, status_code: { lt: 400 } },
        }),
        this.prisma.apiKeyUsage.count({
          where: { ...where, status_code: { gte: 400 } },
        }),
        this.prisma.apiKeyUsage.aggregate({
          where,
          _avg: { response_time: true },
        }),
        this.prisma.apiKeyUsage.groupBy({
          by: ['endpoint'],
          where,
          _count: true,
        }),
        this.prisma.apiKeyUsage.groupBy({
          by: ['ip_address'],
          where,
          _count: true,
          orderBy: { _count: { ip_address: 'desc' } },
          take: 10,
        }),
      ]);

    // Get daily breakdown
    const dailyUsage = await this.prisma.$queryRaw<Array<{ date: string; count: bigint }>>`
      SELECT DATE(created_at) as date, COUNT(*) as count
      FROM api_key_usage
      WHERE api_key_id = ${keyId}
      ${startDate ? `AND created_at >= ${startDate}` : ''}
      ${endDate ? `AND created_at <= ${endDate}` : ''}
      GROUP BY DATE(created_at)
      ORDER BY date DESC
      LIMIT 30
    `;

    return {
      totalRequests: total,
      successfulRequests: successful,
      failedRequests: failed,
      avgResponseTime: Math.round(avgTime._avg.response_time || 0),
      byEndpoint: endpointGroups.reduce(
        (acc, g) => ({ ...acc, [g.endpoint]: g._count }),
        {},
      ),
      byDay: dailyUsage.reduce(
        (acc, d) => ({ ...acc, [d.date]: Number(d.count) }),
        {},
      ),
      topIPs: ipGroups.map((g) => ({
        ip: g.ip_address,
        count: g._count,
      })),
    };
  }

  // ============================================================================
  // RATE LIMITING
  // ============================================================================

  /**
   * Check rate limit
   */
  async checkRateLimit(apiKey: ApiKey, ipAddress: string): Promise<{
    allowed: boolean;
    remaining: number;
    resetAt: Date;
  }> {
    const windowKey = `rate_limit:${apiKey.id}:${Math.floor(Date.now() / (apiKey.rateLimitWindow * 1000))}`;

    // This would ideally use Redis
    // For now, return a simple implementation
    const count = await this.prisma.apiKeyUsage.count({
      where: {
        api_key_id: apiKey.id,
        created_at: {
          gte: new Date(Date.now() - apiKey.rateLimitWindow * 1000),
        },
      },
    });

    const remaining = Math.max(0, apiKey.rateLimit - count);
    const resetAt = new Date(
      Math.ceil(Date.now() / (apiKey.rateLimitWindow * 1000)) *
        apiKey.rateLimitWindow *
        1000,
    );

    return {
      allowed: remaining > 0,
      remaining,
      resetAt,
    };
  }

  // ============================================================================
  // PRIVATE METHODS
  // ============================================================================

  private generateKey(): string {
    return crypto.randomBytes(32).toString('hex');
  }

  private hashKey(key: string): string {
    return crypto.createHash('sha256').update(key).digest('hex');
  }

  private clearKeyCache(keyId: number): void {
    // Simple cache clear - in production, use key prefix lookup
    for (const [cacheKey, cached] of this.keyCache) {
      if (cached.key.id === keyId) {
        this.keyCache.delete(cacheKey);
        break;
      }
    }
  }

  private ipInCidr(ip: string, cidr: string): boolean {
    const [range, bits] = cidr.split('/');
    const mask = ~(2 ** (32 - parseInt(bits)) - 1);

    const ipNum = this.ipToNumber(ip);
    const rangeNum = this.ipToNumber(range);

    return (ipNum & mask) === (rangeNum & mask);
  }

  private ipToNumber(ip: string): number {
    return ip.split('.').reduce((acc, octet) => (acc << 8) + parseInt(octet), 0);
  }

  private mapToApiKey(key: any): ApiKey {
    return {
      id: key.id,
      merchantId: key.merchant_id,
      name: key.name,
      keyPrefix: key.key_prefix,
      hashedKey: key.hashed_key,
      permissions: JSON.parse(key.permissions),
      rateLimit: key.rate_limit,
      rateLimitWindow: key.rate_limit_window,
      ipWhitelist: key.ip_whitelist ? JSON.parse(key.ip_whitelist) : [],
      expiresAt: key.expires_at,
      lastUsedAt: key.last_used_at,
      usageCount: key.usage_count,
      active: key.active,
      createdAt: key.created_at,
      updatedAt: key.updated_at,
    };
  }
}
