import {
  Injectable,
  CanActivate,
  ExecutionContext,
  UnauthorizedException,
  ForbiddenException,
  SetMetadata,
} from '@nestjs/common';
import { Reflector } from '@nestjs/core';
import { ApiKeyService, ApiKeyPermission } from './api-key.service';

export const API_KEY_PERMISSIONS = 'api_key_permissions';
export const RequireApiKeyPermissions = (...permissions: ApiKeyPermission[]) =>
  SetMetadata(API_KEY_PERMISSIONS, permissions);

@Injectable()
export class ApiKeyGuard implements CanActivate {
  constructor(
    private apiKeyService: ApiKeyService,
    private reflector: Reflector,
  ) {}

  async canActivate(context: ExecutionContext): Promise<boolean> {
    const request = context.switchToHttp().getRequest();
    const startTime = Date.now();

    // Extract API key from header
    const apiKey = this.extractApiKey(request);

    if (!apiKey) {
      throw new UnauthorizedException('API key is required');
    }

    // Validate key
    const validatedKey = await this.apiKeyService.validateKey(apiKey);

    if (!validatedKey) {
      throw new UnauthorizedException('Invalid or expired API key');
    }

    // Check IP whitelist
    const clientIp = this.getClientIp(request);
    if (!this.apiKeyService.isIpAllowed(validatedKey, clientIp)) {
      throw new ForbiddenException('IP address not allowed');
    }

    // Check rate limit
    const rateLimit = await this.apiKeyService.checkRateLimit(
      validatedKey,
      clientIp,
    );

    // Set rate limit headers
    const response = context.switchToHttp().getResponse();
    response.setHeader('X-RateLimit-Limit', validatedKey.rateLimit);
    response.setHeader('X-RateLimit-Remaining', rateLimit.remaining);
    response.setHeader(
      'X-RateLimit-Reset',
      Math.floor(rateLimit.resetAt.getTime() / 1000),
    );

    if (!rateLimit.allowed) {
      response.setHeader('Retry-After', Math.ceil((rateLimit.resetAt.getTime() - Date.now()) / 1000));
      throw new ForbiddenException('Rate limit exceeded');
    }

    // Check permissions
    const requiredPermissions = this.reflector.get<ApiKeyPermission[]>(
      API_KEY_PERMISSIONS,
      context.getHandler(),
    );

    if (requiredPermissions?.length) {
      const hasAllPermissions = requiredPermissions.every((permission) =>
        this.apiKeyService.hasPermission(validatedKey, permission),
      );

      if (!hasAllPermissions) {
        throw new ForbiddenException('Insufficient permissions');
      }
    }

    // Attach key info to request
    request.apiKey = validatedKey;
    request.merchantId = validatedKey.merchantId;

    // Log usage (non-blocking)
    const endpoint = request.route?.path || request.path;
    const method = request.method;

    response.on('finish', () => {
      this.apiKeyService.logUsage(validatedKey.id, {
        endpoint,
        method,
        statusCode: response.statusCode,
        ipAddress: clientIp,
        userAgent: request.headers['user-agent'],
        responseTime: Date.now() - startTime,
      }).catch(() => {});
    });

    return true;
  }

  private extractApiKey(request: any): string | null {
    // Check Authorization header (Bearer token style)
    const authHeader = request.headers.authorization;
    if (authHeader?.startsWith('Bearer ')) {
      const token = authHeader.substring(7);
      if (token.startsWith('mk_')) {
        return token;
      }
    }

    // Check X-API-Key header
    const apiKeyHeader = request.headers['x-api-key'];
    if (apiKeyHeader) {
      return apiKeyHeader;
    }

    // Check query parameter (not recommended for production)
    if (request.query.api_key) {
      return request.query.api_key;
    }

    return null;
  }

  private getClientIp(request: any): string {
    const forwardedFor = request.headers['x-forwarded-for'];
    if (forwardedFor) {
      return forwardedFor.split(',')[0].trim();
    }
    return request.ip || request.connection?.remoteAddress || 'unknown';
  }
}
