import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../common/prisma/prisma.service';

export interface DateRange {
  startDate: Date;
  endDate: Date;
}

export interface AnalyticsFilter {
  merchantId: number;
  dateRange?: DateRange;
  serviceType?: string;
  zoneId?: number;
}

@Injectable()
export class AnalyticsService {
  private readonly logger = new Logger(AnalyticsService.name);

  constructor(private prisma: PrismaService) {}

  /**
   * Get booking analytics
   */
  async getBookingAnalytics(filter: AnalyticsFilter) {
    const { merchantId, dateRange, serviceType, zoneId } = filter;

    const where: any = { merchant_id: merchantId };

    if (dateRange) {
      where.created_at = {
        gte: dateRange.startDate,
        lte: dateRange.endDate,
      };
    }

    if (serviceType) where.service_type = serviceType;
    if (zoneId) where.zone_id = zoneId;

    // Total bookings by status
    const byStatus = await this.prisma.booking.groupBy({
      by: ['status'],
      where,
      _count: { id: true },
    });

    // Total revenue
    const revenue = await this.prisma.booking.aggregate({
      where: { ...where, status: 'completed' },
      _sum: { total_fare: true, driver_earning: true, commission: true },
      _avg: { total_fare: true },
    });

    // Bookings over time (daily)
    const dailyBookings = await this.getBookingsOverTime(where, 'day');

    // Average booking metrics
    const avgMetrics = await this.prisma.booking.aggregate({
      where: { ...where, status: 'completed' },
      _avg: {
        distance: true,
        duration: true,
        wait_time: true,
        rating: true,
      },
    });

    // Cancellation rate
    const totalBookings = byStatus.reduce((sum, s) => sum + s._count.id, 0);
    const cancelledBookings = byStatus.find(s => s.status === 'cancelled')?._count.id || 0;
    const cancellationRate = totalBookings > 0 ? (cancelledBookings / totalBookings) * 100 : 0;

    return {
      summary: {
        total: totalBookings,
        completed: byStatus.find(s => s.status === 'completed')?._count.id || 0,
        cancelled: cancelledBookings,
        cancellationRate: Math.round(cancellationRate * 100) / 100,
      },
      revenue: {
        total: revenue._sum.total_fare || 0,
        driverEarnings: revenue._sum.driver_earning || 0,
        commission: revenue._sum.commission || 0,
        averageFare: Math.round((revenue._avg.total_fare || 0) * 100) / 100,
      },
      averages: {
        distance: Math.round((avgMetrics._avg.distance || 0) * 100) / 100,
        duration: Math.round((avgMetrics._avg.duration || 0) * 100) / 100,
        waitTime: Math.round((avgMetrics._avg.wait_time || 0) * 100) / 100,
        rating: Math.round((avgMetrics._avg.rating || 0) * 100) / 100,
      },
      byStatus,
      dailyTrend: dailyBookings,
    };
  }

  /**
   * Get driver analytics
   */
  async getDriverAnalytics(filter: AnalyticsFilter) {
    const { merchantId, dateRange } = filter;

    const where: any = { merchant_id: merchantId };
    if (dateRange) {
      where.created_at = { gte: dateRange.startDate, lte: dateRange.endDate };
    }

    // Driver counts
    const totalDrivers = await this.prisma.driver.count({ where });
    const activeDrivers = await this.prisma.driver.count({
      where: { ...where, is_active: true },
    });
    const onlineDrivers = await this.prisma.driver.count({
      where: { ...where, is_online: true },
    });

    // New drivers over time
    const newDrivers = await this.prisma.driver.groupBy({
      by: [],
      where: dateRange ? {
        merchant_id: merchantId,
        created_at: { gte: dateRange.startDate, lte: dateRange.endDate },
      } : where,
      _count: { id: true },
    });

    // Top performers
    const topDrivers = await this.prisma.driver.findMany({
      where: { merchant_id: merchantId, is_active: true },
      select: {
        id: true,
        name: true,
        rating: true,
        total_trips: true,
        _count: {
          select: {
            bookings: {
              where: dateRange ? {
                status: 'completed',
                ride_ended_at: { gte: dateRange.startDate, lte: dateRange.endDate },
              } : { status: 'completed' },
            },
          },
        },
      },
      orderBy: { rating: 'desc' },
      take: 10,
    });

    // Average rating
    const avgRating = await this.prisma.driver.aggregate({
      where: { merchant_id: merchantId, is_active: true },
      _avg: { rating: true },
    });

    // Drivers by level
    const byLevel = await this.prisma.driver.groupBy({
      by: ['level'],
      where: { merchant_id: merchantId },
      _count: { id: true },
    });

    // Drivers by vehicle type
    const byVehicleType = await this.prisma.driver.groupBy({
      by: ['vehicle_type'],
      where: { merchant_id: merchantId, vehicle_type: { not: null } },
      _count: { id: true },
    });

    return {
      summary: {
        total: totalDrivers,
        active: activeDrivers,
        online: onlineDrivers,
        inactive: totalDrivers - activeDrivers,
        averageRating: Math.round((avgRating._avg.rating || 0) * 100) / 100,
      },
      newDrivers: newDrivers[0]?._count.id || 0,
      topDrivers: topDrivers.map(d => ({
        id: d.id,
        name: d.name,
        rating: d.rating,
        trips: d._count.bookings,
      })),
      byLevel,
      byVehicleType,
    };
  }

  /**
   * Get user analytics
   */
  async getUserAnalytics(filter: AnalyticsFilter) {
    const { merchantId, dateRange } = filter;

    const where: any = { merchant_id: merchantId };

    // User counts
    const totalUsers = await this.prisma.user.count({ where });

    // New users in period
    const newUsersWhere: any = { merchant_id: merchantId };
    if (dateRange) {
      newUsersWhere.created_at = { gte: dateRange.startDate, lte: dateRange.endDate };
    }
    const newUsers = await this.prisma.user.count({ where: newUsersWhere });

    // Active users (made booking in period)
    let activeUsers = 0;
    if (dateRange) {
      const activeUsersResult = await this.prisma.booking.findMany({
        where: {
          merchant_id: merchantId,
          created_at: { gte: dateRange.startDate, lte: dateRange.endDate },
        },
        select: { user_id: true },
        distinct: ['user_id'],
      });
      activeUsers = activeUsersResult.length;
    }

    // Users by platform
    const byPlatform = await this.prisma.user.groupBy({
      by: ['device_type'],
      where,
      _count: { id: true },
    });

    // User retention
    const retention = await this.calculateUserRetention(merchantId, dateRange);

    // Top users
    const topUsers = await this.prisma.user.findMany({
      where,
      select: {
        id: true,
        name: true,
        total_bookings: true,
        total_spent: true,
      },
      orderBy: { total_bookings: 'desc' },
      take: 10,
    });

    return {
      summary: {
        total: totalUsers,
        new: newUsers,
        active: activeUsers,
      },
      byPlatform,
      retention,
      topUsers,
    };
  }

  /**
   * Get revenue analytics
   */
  async getRevenueAnalytics(filter: AnalyticsFilter) {
    const { merchantId, dateRange } = filter;

    const where: any = {
      merchant_id: merchantId,
      status: 'completed',
    };

    if (dateRange) {
      where.ride_ended_at = { gte: dateRange.startDate, lte: dateRange.endDate };
    }

    // Total revenue
    const totalRevenue = await this.prisma.booking.aggregate({
      where,
      _sum: {
        total_fare: true,
        base_fare: true,
        distance_fare: true,
        time_fare: true,
        surge_fare: true,
        discount: true,
        tip_amount: true,
        commission: true,
      },
    });

    // Revenue by service type
    const byServiceType = await this.prisma.booking.groupBy({
      by: ['service_type'],
      where,
      _sum: { total_fare: true },
      _count: { id: true },
    });

    // Revenue by payment method
    const byPaymentMethod = await this.prisma.booking.groupBy({
      by: ['payment_method'],
      where,
      _sum: { total_fare: true },
      _count: { id: true },
    });

    // Daily revenue trend
    const dailyRevenue = await this.getRevenueOverTime(where, 'day');

    // Revenue by zone
    const byZone = await this.prisma.booking.groupBy({
      by: ['zone_id'],
      where: { ...where, zone_id: { not: null } },
      _sum: { total_fare: true },
      _count: { id: true },
    });

    return {
      summary: {
        total: totalRevenue._sum.total_fare || 0,
        baseFare: totalRevenue._sum.base_fare || 0,
        distanceFare: totalRevenue._sum.distance_fare || 0,
        timeFare: totalRevenue._sum.time_fare || 0,
        surgeFare: totalRevenue._sum.surge_fare || 0,
        discount: totalRevenue._sum.discount || 0,
        tips: totalRevenue._sum.tip_amount || 0,
        commission: totalRevenue._sum.commission || 0,
      },
      byServiceType: byServiceType.map(s => ({
        serviceType: s.service_type,
        revenue: s._sum.total_fare || 0,
        count: s._count.id,
      })),
      byPaymentMethod: byPaymentMethod.map(p => ({
        method: p.payment_method,
        revenue: p._sum.total_fare || 0,
        count: p._count.id,
      })),
      dailyTrend: dailyRevenue,
      byZone,
    };
  }

  /**
   * Get geographic analytics
   */
  async getGeographicAnalytics(filter: AnalyticsFilter) {
    const { merchantId, dateRange } = filter;

    const where: any = { merchant_id: merchantId };
    if (dateRange) {
      where.created_at = { gte: dateRange.startDate, lte: dateRange.endDate };
    }

    // Popular pickup locations (clustered)
    const pickupLocations = await this.prisma.booking.findMany({
      where: { ...where, status: 'completed' },
      select: {
        pickup_lat: true,
        pickup_lng: true,
        pickup_address: true,
      },
      take: 1000,
    });

    // Popular dropoff locations
    const dropoffLocations = await this.prisma.booking.findMany({
      where: { ...where, status: 'completed' },
      select: {
        dropoff_lat: true,
        dropoff_lng: true,
        dropoff_address: true,
      },
      take: 1000,
    });

    // Bookings by zone
    const byZone = await this.prisma.booking.groupBy({
      by: ['zone_id'],
      where: { ...where, zone_id: { not: null } },
      _count: { id: true },
      _sum: { total_fare: true },
    });

    // Driver distribution
    const driverLocations = await this.prisma.driver.findMany({
      where: { merchant_id: merchantId, is_online: true },
      select: {
        current_lat: true,
        current_lng: true,
        is_available: true,
      },
    });

    return {
      pickupHeatmap: this.clusterLocations(pickupLocations, 'pickup'),
      dropoffHeatmap: this.clusterLocations(dropoffLocations, 'dropoff'),
      byZone,
      driverDistribution: driverLocations,
    };
  }

  /**
   * Get time-based analytics
   */
  async getTimeAnalytics(filter: AnalyticsFilter) {
    const { merchantId, dateRange } = filter;

    const where: any = {
      merchant_id: merchantId,
      status: 'completed',
    };

    if (dateRange) {
      where.created_at = { gte: dateRange.startDate, lte: dateRange.endDate };
    }

    // Bookings by hour of day
    const byHour = await this.prisma.$queryRaw`
      SELECT
        HOUR(created_at) as hour,
        COUNT(*) as count,
        SUM(total_fare) as revenue
      FROM bookings
      WHERE merchant_id = ${merchantId}
        AND status = 'completed'
        ${dateRange ? `AND created_at BETWEEN ${dateRange.startDate} AND ${dateRange.endDate}` : ''}
      GROUP BY HOUR(created_at)
      ORDER BY hour
    `;

    // Bookings by day of week
    const byDayOfWeek = await this.prisma.$queryRaw`
      SELECT
        DAYOFWEEK(created_at) as day_of_week,
        COUNT(*) as count,
        SUM(total_fare) as revenue
      FROM bookings
      WHERE merchant_id = ${merchantId}
        AND status = 'completed'
        ${dateRange ? `AND created_at BETWEEN ${dateRange.startDate} AND ${dateRange.endDate}` : ''}
      GROUP BY DAYOFWEEK(created_at)
      ORDER BY day_of_week
    `;

    // Peak hours
    const peakHours = (byHour as any[])
      .sort((a, b) => b.count - a.count)
      .slice(0, 5)
      .map(h => ({ hour: h.hour, count: Number(h.count) }));

    return {
      byHour,
      byDayOfWeek,
      peakHours,
    };
  }

  // ============================================================================
  // PRIVATE METHODS
  // ============================================================================

  private async getBookingsOverTime(where: any, granularity: 'day' | 'week' | 'month') {
    const groupBy = granularity === 'day' ? 'DATE(created_at)'
      : granularity === 'week' ? 'YEARWEEK(created_at)'
      : 'DATE_FORMAT(created_at, "%Y-%m")';

    return this.prisma.$queryRawUnsafe(`
      SELECT
        ${groupBy} as period,
        COUNT(*) as count,
        SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed,
        SUM(CASE WHEN status = 'cancelled' THEN 1 ELSE 0 END) as cancelled
      FROM bookings
      WHERE merchant_id = ${where.merchant_id}
        ${where.created_at?.gte ? `AND created_at >= '${where.created_at.gte.toISOString()}'` : ''}
        ${where.created_at?.lte ? `AND created_at <= '${where.created_at.lte.toISOString()}'` : ''}
      GROUP BY ${groupBy}
      ORDER BY period
    `);
  }

  private async getRevenueOverTime(where: any, granularity: 'day' | 'week' | 'month') {
    const groupBy = granularity === 'day' ? 'DATE(ride_ended_at)'
      : granularity === 'week' ? 'YEARWEEK(ride_ended_at)'
      : 'DATE_FORMAT(ride_ended_at, "%Y-%m")';

    return this.prisma.$queryRawUnsafe(`
      SELECT
        ${groupBy} as period,
        SUM(total_fare) as revenue,
        COUNT(*) as count
      FROM bookings
      WHERE merchant_id = ${where.merchant_id}
        AND status = 'completed'
        ${where.ride_ended_at?.gte ? `AND ride_ended_at >= '${where.ride_ended_at.gte.toISOString()}'` : ''}
        ${where.ride_ended_at?.lte ? `AND ride_ended_at <= '${where.ride_ended_at.lte.toISOString()}'` : ''}
      GROUP BY ${groupBy}
      ORDER BY period
    `);
  }

  private async calculateUserRetention(merchantId: number, dateRange?: DateRange) {
    // Simplified retention calculation
    if (!dateRange) return { week1: 0, week2: 0, week4: 0 };

    const startDate = dateRange.startDate;

    // Users who booked in week 1
    const week1End = new Date(startDate);
    week1End.setDate(week1End.getDate() + 7);

    const week1Users = await this.prisma.booking.findMany({
      where: {
        merchant_id: merchantId,
        created_at: { gte: startDate, lt: week1End },
      },
      select: { user_id: true },
      distinct: ['user_id'],
    });

    const week1UserIds = week1Users.map(u => u.user_id);

    // Check retention in subsequent weeks
    const week2Start = week1End;
    const week2End = new Date(week2Start);
    week2End.setDate(week2End.getDate() + 7);

    const week2Retained = await this.prisma.booking.findMany({
      where: {
        merchant_id: merchantId,
        user_id: { in: week1UserIds },
        created_at: { gte: week2Start, lt: week2End },
      },
      select: { user_id: true },
      distinct: ['user_id'],
    });

    return {
      week1: week1UserIds.length,
      week2: week2Retained.length,
      retention: week1UserIds.length > 0
        ? Math.round((week2Retained.length / week1UserIds.length) * 100)
        : 0,
    };
  }

  private clusterLocations(locations: any[], type: 'pickup' | 'dropoff') {
    // Simple clustering by rounding coordinates
    const clusters: Record<string, { lat: number; lng: number; count: number }> = {};

    const latKey = type === 'pickup' ? 'pickup_lat' : 'dropoff_lat';
    const lngKey = type === 'pickup' ? 'pickup_lng' : 'dropoff_lng';

    for (const loc of locations) {
      if (!loc[latKey] || !loc[lngKey]) continue;

      const roundedLat = Math.round(loc[latKey] * 100) / 100;
      const roundedLng = Math.round(loc[lngKey] * 100) / 100;
      const key = `${roundedLat},${roundedLng}`;

      if (!clusters[key]) {
        clusters[key] = { lat: roundedLat, lng: roundedLng, count: 0 };
      }
      clusters[key].count++;
    }

    return Object.values(clusters).sort((a, b) => b.count - a.count).slice(0, 50);
  }
}
