import {
  Controller,
  Get,
  Query,
  UseGuards,
} from '@nestjs/common';
import { AnalyticsService } from './analytics.service';
import { MetricsService } from './metrics.service';
import { RealtimeAnalyticsService } from './realtime.service';
import { JwtAuthGuard } from '../../common/guards/jwt-auth.guard';
import { MerchantGuard } from '../../common/guards/merchant.guard';
import { CurrentMerchant } from '../../common/decorators/merchant.decorator';

@Controller('analytics')
@UseGuards(JwtAuthGuard, MerchantGuard)
export class AnalyticsController {
  constructor(
    private readonly analyticsService: AnalyticsService,
    private readonly metricsService: MetricsService,
    private readonly realtimeService: RealtimeAnalyticsService,
  ) {}

  // ============================================================================
  // REAL-TIME
  // ============================================================================

  /**
   * Get real-time dashboard stats
   */
  @Get('realtime/stats')
  async getRealtimeStats(@CurrentMerchant() merchantId: number) {
    return this.realtimeService.getRealtimeStats(merchantId);
  }

  /**
   * Get live bookings
   */
  @Get('realtime/bookings')
  async getLiveBookings(
    @CurrentMerchant() merchantId: number,
    @Query('limit') limit?: string,
  ) {
    return this.realtimeService.getLiveBookings(
      merchantId,
      limit ? parseInt(limit) : 20,
    );
  }

  /**
   * Get live driver locations
   */
  @Get('realtime/drivers')
  async getLiveDrivers(
    @CurrentMerchant() merchantId: number,
    @Query('minLat') minLat?: string,
    @Query('maxLat') maxLat?: string,
    @Query('minLng') minLng?: string,
    @Query('maxLng') maxLng?: string,
  ) {
    const bounds = minLat && maxLat && minLng && maxLng
      ? {
          minLat: parseFloat(minLat),
          maxLat: parseFloat(maxLat),
          minLng: parseFloat(minLng),
          maxLng: parseFloat(maxLng),
        }
      : undefined;

    return this.realtimeService.getLiveDrivers(merchantId, bounds);
  }

  /**
   * Get activity feed
   */
  @Get('realtime/activity')
  async getActivityFeed(
    @CurrentMerchant() merchantId: number,
    @Query('limit') limit?: string,
  ) {
    return this.realtimeService.getActivityFeed(
      merchantId,
      limit ? parseInt(limit) : 50,
    );
  }

  /**
   * Get demand/supply by zone
   */
  @Get('realtime/demand-supply')
  async getDemandSupply(@CurrentMerchant() merchantId: number) {
    return this.realtimeService.getDemandSupplyByZone(merchantId);
  }

  /**
   * Get booking funnel
   */
  @Get('realtime/funnel')
  async getBookingFunnel(@CurrentMerchant() merchantId: number) {
    return this.realtimeService.getBookingFunnel(merchantId);
  }

  /**
   * Get average wait times
   */
  @Get('realtime/wait-times')
  async getWaitTimes(@CurrentMerchant() merchantId: number) {
    return this.realtimeService.getAverageWaitTimes(merchantId);
  }

  // ============================================================================
  // BOOKING ANALYTICS
  // ============================================================================

  /**
   * Get booking analytics
   */
  @Get('bookings')
  async getBookingAnalytics(
    @CurrentMerchant() merchantId: number,
    @Query('startDate') startDate?: string,
    @Query('endDate') endDate?: string,
    @Query('serviceType') serviceType?: string,
    @Query('zoneId') zoneId?: string,
  ) {
    return this.analyticsService.getBookingAnalytics({
      merchantId,
      dateRange: startDate && endDate
        ? { startDate: new Date(startDate), endDate: new Date(endDate) }
        : undefined,
      serviceType,
      zoneId: zoneId ? parseInt(zoneId) : undefined,
    });
  }

  // ============================================================================
  // DRIVER ANALYTICS
  // ============================================================================

  /**
   * Get driver analytics
   */
  @Get('drivers')
  async getDriverAnalytics(
    @CurrentMerchant() merchantId: number,
    @Query('startDate') startDate?: string,
    @Query('endDate') endDate?: string,
  ) {
    return this.analyticsService.getDriverAnalytics({
      merchantId,
      dateRange: startDate && endDate
        ? { startDate: new Date(startDate), endDate: new Date(endDate) }
        : undefined,
    });
  }

  // ============================================================================
  // USER ANALYTICS
  // ============================================================================

  /**
   * Get user analytics
   */
  @Get('users')
  async getUserAnalytics(
    @CurrentMerchant() merchantId: number,
    @Query('startDate') startDate?: string,
    @Query('endDate') endDate?: string,
  ) {
    return this.analyticsService.getUserAnalytics({
      merchantId,
      dateRange: startDate && endDate
        ? { startDate: new Date(startDate), endDate: new Date(endDate) }
        : undefined,
    });
  }

  // ============================================================================
  // REVENUE ANALYTICS
  // ============================================================================

  /**
   * Get revenue analytics
   */
  @Get('revenue')
  async getRevenueAnalytics(
    @CurrentMerchant() merchantId: number,
    @Query('startDate') startDate?: string,
    @Query('endDate') endDate?: string,
  ) {
    return this.analyticsService.getRevenueAnalytics({
      merchantId,
      dateRange: startDate && endDate
        ? { startDate: new Date(startDate), endDate: new Date(endDate) }
        : undefined,
    });
  }

  // ============================================================================
  // GEOGRAPHIC ANALYTICS
  // ============================================================================

  /**
   * Get geographic analytics
   */
  @Get('geographic')
  async getGeographicAnalytics(
    @CurrentMerchant() merchantId: number,
    @Query('startDate') startDate?: string,
    @Query('endDate') endDate?: string,
  ) {
    return this.analyticsService.getGeographicAnalytics({
      merchantId,
      dateRange: startDate && endDate
        ? { startDate: new Date(startDate), endDate: new Date(endDate) }
        : undefined,
    });
  }

  // ============================================================================
  // TIME ANALYTICS
  // ============================================================================

  /**
   * Get time-based analytics
   */
  @Get('time')
  async getTimeAnalytics(
    @CurrentMerchant() merchantId: number,
    @Query('startDate') startDate?: string,
    @Query('endDate') endDate?: string,
  ) {
    return this.analyticsService.getTimeAnalytics({
      merchantId,
      dateRange: startDate && endDate
        ? { startDate: new Date(startDate), endDate: new Date(endDate) }
        : undefined,
    });
  }

  // ============================================================================
  // METRICS
  // ============================================================================

  /**
   * Get dashboard metrics summary
   */
  @Get('metrics/dashboard')
  async getDashboardMetrics(@CurrentMerchant() merchantId: number) {
    return this.metricsService.getDashboardMetrics(merchantId);
  }

  /**
   * Get specific metric over time
   */
  @Get('metrics/:name')
  async getMetric(
    @CurrentMerchant() merchantId: number,
    @Query('name') name: string,
    @Query('startDate') startDate: string,
    @Query('endDate') endDate: string,
    @Query('aggregation') aggregation?: string,
    @Query('granularity') granularity?: string,
  ) {
    return this.metricsService.getMetric(
      name,
      new Date(startDate),
      new Date(endDate),
      (aggregation as 'sum' | 'avg' | 'min' | 'max' | 'count') || 'sum',
      (granularity as 'minute' | 'hour' | 'day') || 'hour',
    );
  }
}
