import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../common/prisma/prisma.service';
import * as ExcelJS from 'exceljs';

export interface ReportFilters {
  startDate?: Date;
  endDate?: Date;
  status?: string;
  driverId?: number;
  userId?: number;
  paymentMethod?: string;
}

@Injectable()
export class ReportsService {
  private readonly logger = new Logger(ReportsService.name);

  constructor(private prisma: PrismaService) {}

  /**
   * Generate bookings report
   */
  async generateBookingsReport(
    merchantId: number,
    filters: ReportFilters,
  ): Promise<Buffer> {
    const workbook = new ExcelJS.Workbook();
    const worksheet = workbook.addWorksheet('Reservations');

    // Define columns
    worksheet.columns = [
      { header: 'ID', key: 'id', width: 10 },
      { header: 'Date', key: 'date', width: 20 },
      { header: 'Client', key: 'user', width: 25 },
      { header: 'Chauffeur', key: 'driver', width: 25 },
      { header: 'Depart', key: 'pickup', width: 30 },
      { header: 'Destination', key: 'drop', width: 30 },
      { header: 'Distance (km)', key: 'distance', width: 15 },
      { header: 'Montant', key: 'amount', width: 15 },
      { header: 'Paiement', key: 'payment', width: 15 },
      { header: 'Statut', key: 'status', width: 15 },
    ];

    // Style header row
    worksheet.getRow(1).font = { bold: true };
    worksheet.getRow(1).fill = {
      type: 'pattern',
      pattern: 'solid',
      fgColor: { argb: 'FFE0E0E0' },
    };

    // Get bookings data
    const where: any = { merchant_id: merchantId };

    if (filters.startDate) {
      where.created_at = { ...where.created_at, gte: filters.startDate };
    }
    if (filters.endDate) {
      where.created_at = { ...where.created_at, lte: filters.endDate };
    }
    if (filters.status) {
      where.status = filters.status;
    }
    if (filters.driverId) {
      where.driver_id = filters.driverId;
    }
    if (filters.paymentMethod) {
      where.payment_method = filters.paymentMethod;
    }

    const bookings = await this.prisma.booking.findMany({
      where,
      include: {
        user: { select: { first_name: true, last_name: true, mobile: true } },
        driver: { select: { first_name: true, last_name: true, mobile: true } },
      },
      orderBy: { created_at: 'desc' },
      take: 10000,
    });

    // Add data rows
    for (const booking of bookings) {
      worksheet.addRow({
        id: booking.id,
        date: booking.created_at.toISOString(),
        user: booking.user
          ? `${booking.user.first_name} ${booking.user.last_name}`
          : 'N/A',
        driver: booking.driver
          ? `${booking.driver.first_name} ${booking.driver.last_name}`
          : 'Non assigne',
        pickup: booking.pickup_address || 'N/A',
        drop: booking.drop_address || 'N/A',
        distance: Number(booking.distance) || 0,
        amount: Number(booking.total_amount) || 0,
        payment: booking.payment_method || 'N/A',
        status: this.translateStatus(booking.status),
      });
    }

    // Add summary row
    const totalAmount = bookings.reduce(
      (sum, b) => sum + (Number(b.total_amount) || 0),
      0,
    );
    worksheet.addRow({});
    worksheet.addRow({
      id: 'TOTAL',
      amount: totalAmount,
    });

    return workbook.xlsx.writeBuffer() as Promise<Buffer>;
  }

  /**
   * Generate drivers report
   */
  async generateDriversReport(
    merchantId: number,
    filters: ReportFilters,
  ): Promise<Buffer> {
    const workbook = new ExcelJS.Workbook();
    const worksheet = workbook.addWorksheet('Chauffeurs');

    worksheet.columns = [
      { header: 'ID', key: 'id', width: 10 },
      { header: 'Nom', key: 'name', width: 25 },
      { header: 'Telephone', key: 'phone', width: 15 },
      { header: 'Email', key: 'email', width: 25 },
      { header: 'Vehicule', key: 'vehicle', width: 20 },
      { header: 'Courses', key: 'bookings', width: 12 },
      { header: 'Revenu', key: 'revenue', width: 15 },
      { header: 'Note', key: 'rating', width: 10 },
      { header: 'Statut', key: 'status', width: 12 },
      { header: "Date d'inscription", key: 'created', width: 20 },
    ];

    worksheet.getRow(1).font = { bold: true };
    worksheet.getRow(1).fill = {
      type: 'pattern',
      pattern: 'solid',
      fgColor: { argb: 'FFE0E0E0' },
    };

    const drivers = await this.prisma.driver.findMany({
      where: { merchant_id: merchantId },
      include: {
        vehicle: { select: { make: true, model: true, registration: true } },
        _count: { select: { bookings: true } },
      },
      orderBy: { created_at: 'desc' },
    });

    // Get revenue per driver
    const revenueData = await this.prisma.booking.groupBy({
      by: ['driver_id'],
      where: {
        merchant_id: merchantId,
        status: 'completed',
        ...(filters.startDate && { completed_at: { gte: filters.startDate } }),
        ...(filters.endDate && { completed_at: { lte: filters.endDate } }),
      },
      _sum: { total_amount: true },
    });

    const revenueMap = new Map(
      revenueData.map(r => [r.driver_id, Number(r._sum.total_amount) || 0]),
    );

    for (const driver of drivers) {
      worksheet.addRow({
        id: driver.id,
        name: `${driver.first_name} ${driver.last_name}`,
        phone: driver.mobile,
        email: driver.email,
        vehicle: driver.vehicle
          ? `${driver.vehicle.make} ${driver.vehicle.model} (${driver.vehicle.registration})`
          : 'N/A',
        bookings: driver._count.bookings,
        revenue: revenueMap.get(driver.id) || 0,
        rating: Number(driver.rating) || 0,
        status: driver.is_active ? 'Actif' : 'Inactif',
        created: driver.created_at.toISOString().split('T')[0],
      });
    }

    return workbook.xlsx.writeBuffer() as Promise<Buffer>;
  }

  /**
   * Generate financial report
   */
  async generateFinancialReport(
    merchantId: number,
    filters: ReportFilters,
  ): Promise<Buffer> {
    const workbook = new ExcelJS.Workbook();

    // Revenue summary sheet
    const summarySheet = workbook.addWorksheet('Resume');
    summarySheet.columns = [
      { header: 'Metrique', key: 'metric', width: 30 },
      { header: 'Valeur', key: 'value', width: 20 },
    ];

    const where: any = {
      merchant_id: merchantId,
      status: 'completed',
    };

    if (filters.startDate) {
      where.completed_at = { ...where.completed_at, gte: filters.startDate };
    }
    if (filters.endDate) {
      where.completed_at = { ...where.completed_at, lte: filters.endDate };
    }

    const [totalRevenue, bookingsCount, avgAmount] = await Promise.all([
      this.prisma.booking.aggregate({
        where,
        _sum: { total_amount: true },
      }),
      this.prisma.booking.count({ where }),
      this.prisma.booking.aggregate({
        where,
        _avg: { total_amount: true },
      }),
    ]);

    summarySheet.addRow({ metric: 'Total Reservations', value: bookingsCount });
    summarySheet.addRow({
      metric: 'Revenu Total',
      value: `${Number(totalRevenue._sum.total_amount) || 0} XOF`,
    });
    summarySheet.addRow({
      metric: 'Montant Moyen par Course',
      value: `${Math.round(Number(avgAmount._avg.total_amount) || 0)} XOF`,
    });

    // Daily breakdown sheet
    const dailySheet = workbook.addWorksheet('Par Jour');
    dailySheet.columns = [
      { header: 'Date', key: 'date', width: 15 },
      { header: 'Courses', key: 'bookings', width: 12 },
      { header: 'Revenu', key: 'revenue', width: 15 },
    ];

    const bookings = await this.prisma.booking.findMany({
      where,
      select: { completed_at: true, total_amount: true },
    });

    const dailyMap = new Map<string, { count: number; revenue: number }>();

    for (const booking of bookings) {
      if (booking.completed_at) {
        const date = booking.completed_at.toISOString().split('T')[0];
        const current = dailyMap.get(date) || { count: 0, revenue: 0 };
        current.count++;
        current.revenue += Number(booking.total_amount) || 0;
        dailyMap.set(date, current);
      }
    }

    const sortedDates = Array.from(dailyMap.entries()).sort((a, b) =>
      a[0].localeCompare(b[0]),
    );

    for (const [date, data] of sortedDates) {
      dailySheet.addRow({
        date,
        bookings: data.count,
        revenue: data.revenue,
      });
    }

    // Payment methods sheet
    const paymentSheet = workbook.addWorksheet('Par Mode de Paiement');
    paymentSheet.columns = [
      { header: 'Mode de Paiement', key: 'method', width: 20 },
      { header: 'Courses', key: 'count', width: 12 },
      { header: 'Montant', key: 'amount', width: 15 },
    ];

    const paymentData = await this.prisma.booking.groupBy({
      by: ['payment_method'],
      where,
      _sum: { total_amount: true },
      _count: true,
    });

    for (const row of paymentData) {
      paymentSheet.addRow({
        method: row.payment_method || 'Non specifie',
        count: row._count,
        amount: Number(row._sum.total_amount) || 0,
      });
    }

    return workbook.xlsx.writeBuffer() as Promise<Buffer>;
  }

  /**
   * Generate users report
   */
  async generateUsersReport(
    merchantId: number,
    filters: ReportFilters,
  ): Promise<Buffer> {
    const workbook = new ExcelJS.Workbook();
    const worksheet = workbook.addWorksheet('Utilisateurs');

    worksheet.columns = [
      { header: 'ID', key: 'id', width: 10 },
      { header: 'Nom', key: 'name', width: 25 },
      { header: 'Telephone', key: 'phone', width: 15 },
      { header: 'Email', key: 'email', width: 25 },
      { header: 'Courses', key: 'bookings', width: 12 },
      { header: 'Total Depense', key: 'spent', width: 15 },
      { header: 'Statut', key: 'status', width: 12 },
      { header: "Date d'inscription", key: 'created', width: 20 },
    ];

    worksheet.getRow(1).font = { bold: true };

    const users = await this.prisma.user.findMany({
      where: { merchant_id: merchantId },
      include: {
        _count: { select: { bookings: true } },
      },
      orderBy: { created_at: 'desc' },
    });

    // Get spending per user
    const spendingData = await this.prisma.booking.groupBy({
      by: ['user_id'],
      where: {
        merchant_id: merchantId,
        status: 'completed',
      },
      _sum: { total_amount: true },
    });

    const spendingMap = new Map(
      spendingData.map(r => [r.user_id, Number(r._sum.total_amount) || 0]),
    );

    for (const user of users) {
      worksheet.addRow({
        id: user.id,
        name: `${user.first_name} ${user.last_name}`,
        phone: user.mobile,
        email: user.email,
        bookings: user._count.bookings,
        spent: spendingMap.get(user.id) || 0,
        status: user.is_active ? 'Actif' : 'Inactif',
        created: user.created_at.toISOString().split('T')[0],
      });
    }

    return workbook.xlsx.writeBuffer() as Promise<Buffer>;
  }

  /**
   * Get report data for API (JSON format)
   */
  async getReportData(
    merchantId: number,
    reportType: 'bookings' | 'drivers' | 'users' | 'financial',
    filters: ReportFilters,
  ): Promise<any> {
    switch (reportType) {
      case 'bookings':
        return this.getBookingsReportData(merchantId, filters);
      case 'drivers':
        return this.getDriversReportData(merchantId, filters);
      case 'users':
        return this.getUsersReportData(merchantId, filters);
      case 'financial':
        return this.getFinancialReportData(merchantId, filters);
      default:
        throw new Error('Invalid report type');
    }
  }

  private async getBookingsReportData(merchantId: number, filters: ReportFilters) {
    const where: any = { merchant_id: merchantId };
    if (filters.startDate) where.created_at = { gte: filters.startDate };
    if (filters.endDate) where.created_at = { ...where.created_at, lte: filters.endDate };
    if (filters.status) where.status = filters.status;

    return this.prisma.booking.findMany({
      where,
      include: {
        user: { select: { first_name: true, last_name: true } },
        driver: { select: { first_name: true, last_name: true } },
      },
      orderBy: { created_at: 'desc' },
      take: 1000,
    });
  }

  private async getDriversReportData(merchantId: number, filters: ReportFilters) {
    return this.prisma.driver.findMany({
      where: { merchant_id: merchantId },
      include: {
        vehicle: true,
        _count: { select: { bookings: true } },
      },
    });
  }

  private async getUsersReportData(merchantId: number, filters: ReportFilters) {
    return this.prisma.user.findMany({
      where: { merchant_id: merchantId },
      include: {
        _count: { select: { bookings: true } },
      },
    });
  }

  private async getFinancialReportData(merchantId: number, filters: ReportFilters) {
    const where: any = { merchant_id: merchantId, status: 'completed' };
    if (filters.startDate) where.completed_at = { gte: filters.startDate };
    if (filters.endDate) where.completed_at = { ...where.completed_at, lte: filters.endDate };

    const [revenue, byPayment, byDay] = await Promise.all([
      this.prisma.booking.aggregate({
        where,
        _sum: { total_amount: true },
        _count: true,
        _avg: { total_amount: true },
      }),
      this.prisma.booking.groupBy({
        by: ['payment_method'],
        where,
        _sum: { total_amount: true },
        _count: true,
      }),
      this.prisma.booking.groupBy({
        by: ['completed_at'],
        where,
        _sum: { total_amount: true },
        _count: true,
      }),
    ]);

    return { revenue, byPayment, byDay };
  }

  private translateStatus(status: string): string {
    const translations: Record<string, string> = {
      pending: 'En attente',
      accepted: 'Acceptee',
      arrived: 'Arrive',
      started: 'En cours',
      completed: 'Terminee',
      cancelled: 'Annulee',
    };
    return translations[status] || status;
  }
}
