import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../common/prisma/prisma.service';

export interface DashboardStats {
  bookings: {
    total: number;
    today: number;
    thisWeek: number;
    thisMonth: number;
    pending: number;
    ongoing: number;
    completed: number;
    cancelled: number;
  };
  revenue: {
    today: number;
    thisWeek: number;
    thisMonth: number;
    thisYear: number;
  };
  drivers: {
    total: number;
    active: number;
    online: number;
    offline: number;
    pendingApproval: number;
  };
  users: {
    total: number;
    active: number;
    newToday: number;
    newThisWeek: number;
    newThisMonth: number;
  };
  deliveries: {
    total: number;
    pending: number;
    inTransit: number;
    completed: number;
  };
}

@Injectable()
export class DashboardService {
  private readonly logger = new Logger(DashboardService.name);

  constructor(private prisma: PrismaService) {}

  /**
   * Get comprehensive dashboard statistics
   */
  async getDashboardStats(merchantId: number): Promise<DashboardStats> {
    const now = new Date();
    const startOfToday = new Date(now.getFullYear(), now.getMonth(), now.getDate());
    const startOfWeek = new Date(now);
    startOfWeek.setDate(now.getDate() - now.getDay());
    startOfWeek.setHours(0, 0, 0, 0);
    const startOfMonth = new Date(now.getFullYear(), now.getMonth(), 1);
    const startOfYear = new Date(now.getFullYear(), 0, 1);

    const [
      bookingStats,
      revenueStats,
      driverStats,
      userStats,
      deliveryStats,
    ] = await Promise.all([
      this.getBookingStats(merchantId, startOfToday, startOfWeek, startOfMonth),
      this.getRevenueStats(merchantId, startOfToday, startOfWeek, startOfMonth, startOfYear),
      this.getDriverStats(merchantId),
      this.getUserStats(merchantId, startOfToday, startOfWeek, startOfMonth),
      this.getDeliveryStats(merchantId),
    ]);

    return {
      bookings: bookingStats,
      revenue: revenueStats,
      drivers: driverStats,
      users: userStats,
      deliveries: deliveryStats,
    };
  }

  /**
   * Get booking statistics
   */
  private async getBookingStats(
    merchantId: number,
    startOfToday: Date,
    startOfWeek: Date,
    startOfMonth: Date,
  ) {
    const [
      total,
      today,
      thisWeek,
      thisMonth,
      pending,
      ongoing,
      completed,
      cancelled,
    ] = await Promise.all([
      this.prisma.booking.count({ where: { merchant_id: merchantId } }),
      this.prisma.booking.count({
        where: { merchant_id: merchantId, created_at: { gte: startOfToday } },
      }),
      this.prisma.booking.count({
        where: { merchant_id: merchantId, created_at: { gte: startOfWeek } },
      }),
      this.prisma.booking.count({
        where: { merchant_id: merchantId, created_at: { gte: startOfMonth } },
      }),
      this.prisma.booking.count({
        where: { merchant_id: merchantId, status: 'pending' },
      }),
      this.prisma.booking.count({
        where: {
          merchant_id: merchantId,
          status: { in: ['accepted', 'arrived', 'started'] },
        },
      }),
      this.prisma.booking.count({
        where: { merchant_id: merchantId, status: 'completed' },
      }),
      this.prisma.booking.count({
        where: { merchant_id: merchantId, status: 'cancelled' },
      }),
    ]);

    return { total, today, thisWeek, thisMonth, pending, ongoing, completed, cancelled };
  }

  /**
   * Get revenue statistics
   */
  private async getRevenueStats(
    merchantId: number,
    startOfToday: Date,
    startOfWeek: Date,
    startOfMonth: Date,
    startOfYear: Date,
  ) {
    const [todayRevenue, weekRevenue, monthRevenue, yearRevenue] = await Promise.all([
      this.prisma.booking.aggregate({
        where: {
          merchant_id: merchantId,
          status: 'completed',
          completed_at: { gte: startOfToday },
        },
        _sum: { total_amount: true },
      }),
      this.prisma.booking.aggregate({
        where: {
          merchant_id: merchantId,
          status: 'completed',
          completed_at: { gte: startOfWeek },
        },
        _sum: { total_amount: true },
      }),
      this.prisma.booking.aggregate({
        where: {
          merchant_id: merchantId,
          status: 'completed',
          completed_at: { gte: startOfMonth },
        },
        _sum: { total_amount: true },
      }),
      this.prisma.booking.aggregate({
        where: {
          merchant_id: merchantId,
          status: 'completed',
          completed_at: { gte: startOfYear },
        },
        _sum: { total_amount: true },
      }),
    ]);

    return {
      today: Number(todayRevenue._sum.total_amount) || 0,
      thisWeek: Number(weekRevenue._sum.total_amount) || 0,
      thisMonth: Number(monthRevenue._sum.total_amount) || 0,
      thisYear: Number(yearRevenue._sum.total_amount) || 0,
    };
  }

  /**
   * Get driver statistics
   */
  private async getDriverStats(merchantId: number) {
    const [total, active, online, pendingApproval] = await Promise.all([
      this.prisma.driver.count({ where: { merchant_id: merchantId } }),
      this.prisma.driver.count({ where: { merchant_id: merchantId, is_active: 1 } }),
      this.prisma.driver.count({
        where: { merchant_id: merchantId, is_active: 1, is_online: 1 },
      }),
      this.prisma.driver.count({
        where: { merchant_id: merchantId, is_approved: 0 },
      }),
    ]);

    return {
      total,
      active,
      online,
      offline: active - online,
      pendingApproval,
    };
  }

  /**
   * Get user statistics
   */
  private async getUserStats(
    merchantId: number,
    startOfToday: Date,
    startOfWeek: Date,
    startOfMonth: Date,
  ) {
    const [total, active, newToday, newThisWeek, newThisMonth] = await Promise.all([
      this.prisma.user.count({ where: { merchant_id: merchantId } }),
      this.prisma.user.count({ where: { merchant_id: merchantId, is_active: 1 } }),
      this.prisma.user.count({
        where: { merchant_id: merchantId, created_at: { gte: startOfToday } },
      }),
      this.prisma.user.count({
        where: { merchant_id: merchantId, created_at: { gte: startOfWeek } },
      }),
      this.prisma.user.count({
        where: { merchant_id: merchantId, created_at: { gte: startOfMonth } },
      }),
    ]);

    return { total, active, newToday, newThisWeek, newThisMonth };
  }

  /**
   * Get delivery statistics
   */
  private async getDeliveryStats(merchantId: number) {
    const [total, pending, inTransit, completed] = await Promise.all([
      this.prisma.delivery.count({ where: { merchant_id: merchantId } }),
      this.prisma.delivery.count({
        where: { merchant_id: merchantId, status: 'pending' },
      }),
      this.prisma.delivery.count({
        where: {
          merchant_id: merchantId,
          status: { in: ['picked_up', 'in_transit'] },
        },
      }),
      this.prisma.delivery.count({
        where: { merchant_id: merchantId, status: 'delivered' },
      }),
    ]);

    return { total, pending, inTransit, completed };
  }

  /**
   * Get booking trends (last 30 days)
   */
  async getBookingTrends(
    merchantId: number,
    days: number = 30,
  ): Promise<Array<{ date: string; count: number; revenue: number }>> {
    const endDate = new Date();
    const startDate = new Date();
    startDate.setDate(endDate.getDate() - days);

    const bookings = await this.prisma.booking.groupBy({
      by: ['created_at'],
      where: {
        merchant_id: merchantId,
        created_at: { gte: startDate, lte: endDate },
      },
      _count: true,
      _sum: { total_amount: true },
    });

    // Aggregate by date
    const trendMap = new Map<string, { count: number; revenue: number }>();

    for (let d = new Date(startDate); d <= endDate; d.setDate(d.getDate() + 1)) {
      const dateStr = d.toISOString().split('T')[0];
      trendMap.set(dateStr, { count: 0, revenue: 0 });
    }

    // This is a simplified aggregation - in production, use a proper date aggregation
    const rawBookings = await this.prisma.booking.findMany({
      where: {
        merchant_id: merchantId,
        created_at: { gte: startDate, lte: endDate },
      },
      select: { created_at: true, total_amount: true },
    });

    for (const booking of rawBookings) {
      const dateStr = booking.created_at.toISOString().split('T')[0];
      const current = trendMap.get(dateStr) || { count: 0, revenue: 0 };
      current.count++;
      current.revenue += Number(booking.total_amount) || 0;
      trendMap.set(dateStr, current);
    }

    return Array.from(trendMap.entries())
      .map(([date, data]) => ({ date, ...data }))
      .sort((a, b) => a.date.localeCompare(b.date));
  }

  /**
   * Get revenue by payment method
   */
  async getRevenueByPaymentMethod(
    merchantId: number,
    startDate?: Date,
    endDate?: Date,
  ): Promise<Array<{ method: string; amount: number; count: number }>> {
    const where: any = {
      merchant_id: merchantId,
      status: 'completed',
    };

    if (startDate || endDate) {
      where.completed_at = {};
      if (startDate) where.completed_at.gte = startDate;
      if (endDate) where.completed_at.lte = endDate;
    }

    const results = await this.prisma.booking.groupBy({
      by: ['payment_method'],
      where,
      _sum: { total_amount: true },
      _count: true,
    });

    return results.map(r => ({
      method: r.payment_method || 'unknown',
      amount: Number(r._sum.total_amount) || 0,
      count: r._count,
    }));
  }

  /**
   * Get top drivers
   */
  async getTopDrivers(
    merchantId: number,
    limit: number = 10,
    startDate?: Date,
  ): Promise<any[]> {
    const where: any = {
      merchant_id: merchantId,
      status: 'completed',
    };

    if (startDate) {
      where.completed_at = { gte: startDate };
    }

    const results = await this.prisma.booking.groupBy({
      by: ['driver_id'],
      where,
      _sum: { total_amount: true },
      _count: true,
      orderBy: { _count: { driver_id: 'desc' } },
      take: limit,
    });

    const driverIds = results.map(r => r.driver_id).filter(Boolean);
    const drivers = await this.prisma.driver.findMany({
      where: { id: { in: driverIds as number[] } },
      select: {
        id: true,
        first_name: true,
        last_name: true,
        profile_image: true,
        rating: true,
      },
    });

    const driverMap = new Map(drivers.map(d => [d.id, d]));

    return results
      .filter(r => r.driver_id)
      .map(r => ({
        driver: driverMap.get(r.driver_id as number),
        bookings: r._count,
        revenue: Number(r._sum.total_amount) || 0,
      }));
  }

  /**
   * Get peak hours analysis
   */
  async getPeakHoursAnalysis(
    merchantId: number,
    days: number = 30,
  ): Promise<Array<{ hour: number; bookings: number; avgWaitTime: number }>> {
    const startDate = new Date();
    startDate.setDate(startDate.getDate() - days);

    const bookings = await this.prisma.booking.findMany({
      where: {
        merchant_id: merchantId,
        created_at: { gte: startDate },
      },
      select: {
        created_at: true,
        accepted_at: true,
      },
    });

    const hourStats = new Map<number, { count: number; totalWaitTime: number }>();

    for (let h = 0; h < 24; h++) {
      hourStats.set(h, { count: 0, totalWaitTime: 0 });
    }

    for (const booking of bookings) {
      const hour = booking.created_at.getHours();
      const current = hourStats.get(hour)!;
      current.count++;

      if (booking.accepted_at) {
        const waitTime =
          (booking.accepted_at.getTime() - booking.created_at.getTime()) / 60000; // minutes
        current.totalWaitTime += waitTime;
      }
    }

    return Array.from(hourStats.entries()).map(([hour, data]) => ({
      hour,
      bookings: data.count,
      avgWaitTime: data.count > 0 ? Math.round(data.totalWaitTime / data.count) : 0,
    }));
  }

  /**
   * Get service zone heatmap data
   */
  async getHeatmapData(
    merchantId: number,
    startDate?: Date,
  ): Promise<Array<{ lat: number; lng: number; weight: number }>> {
    const where: any = { merchant_id: merchantId };
    if (startDate) {
      where.created_at = { gte: startDate };
    }

    const bookings = await this.prisma.booking.findMany({
      where,
      select: {
        pickup_latitude: true,
        pickup_longitude: true,
      },
      take: 1000,
    });

    return bookings
      .filter(b => b.pickup_latitude && b.pickup_longitude)
      .map(b => ({
        lat: Number(b.pickup_latitude),
        lng: Number(b.pickup_longitude),
        weight: 1,
      }));
  }
}
