import { Injectable, Logger, BadRequestException, NotFoundException } from '@nestjs/common';
import { PrismaService } from '../../common/prisma/prisma.service';
import { NotificationService } from '../notification/notification.service';

@Injectable()
export class AdminService {
  private readonly logger = new Logger(AdminService.name);

  constructor(
    private prisma: PrismaService,
    private notificationService: NotificationService,
  ) {}

  // ============================================================================
  // DRIVER MANAGEMENT
  // ============================================================================

  /**
   * Get drivers list with filters
   */
  async getDrivers(
    merchantId: number,
    options?: {
      status?: 'active' | 'inactive' | 'pending';
      search?: string;
      page?: number;
      limit?: number;
    },
  ): Promise<{ data: any[]; total: number }> {
    const page = options?.page || 1;
    const limit = options?.limit || 20;
    const skip = (page - 1) * limit;

    const where: any = { merchant_id: merchantId };

    if (options?.status === 'active') {
      where.is_active = 1;
      where.is_approved = 1;
    } else if (options?.status === 'inactive') {
      where.is_active = 0;
    } else if (options?.status === 'pending') {
      where.is_approved = 0;
    }

    if (options?.search) {
      where.OR = [
        { first_name: { contains: options.search } },
        { last_name: { contains: options.search } },
        { email: { contains: options.search } },
        { mobile: { contains: options.search } },
      ];
    }

    const [drivers, total] = await Promise.all([
      this.prisma.driver.findMany({
        where,
        include: {
          vehicle: { select: { make: true, model: true, registration: true } },
        },
        skip,
        take: limit,
        orderBy: { created_at: 'desc' },
      }),
      this.prisma.driver.count({ where }),
    ]);

    return { data: drivers, total };
  }

  /**
   * Approve driver
   */
  async approveDriver(driverId: number, adminId: number): Promise<any> {
    const driver = await this.prisma.driver.findUnique({
      where: { id: driverId },
    });

    if (!driver) {
      throw new NotFoundException('Chauffeur non trouve');
    }

    const updated = await this.prisma.driver.update({
      where: { id: driverId },
      data: {
        is_approved: 1,
        is_active: 1,
        approved_by: adminId,
        approved_at: new Date(),
      },
    });

    await this.notificationService.sendToUser(driverId, 'driver', {
      title: 'Compte approuve',
      body: 'Votre compte chauffeur a ete approuve. Vous pouvez maintenant recevoir des courses.',
      data: { type: 'driver_approved' },
    });

    return updated;
  }

  /**
   * Reject driver
   */
  async rejectDriver(driverId: number, adminId: number, reason: string): Promise<any> {
    const driver = await this.prisma.driver.findUnique({
      where: { id: driverId },
    });

    if (!driver) {
      throw new NotFoundException('Chauffeur non trouve');
    }

    const updated = await this.prisma.driver.update({
      where: { id: driverId },
      data: {
        is_approved: 0,
        rejection_reason: reason,
        rejected_by: adminId,
        rejected_at: new Date(),
      },
    });

    await this.notificationService.sendToUser(driverId, 'driver', {
      title: 'Compte refuse',
      body: `Votre demande d'inscription a ete refusee. Raison: ${reason}`,
      data: { type: 'driver_rejected', reason },
    });

    return updated;
  }

  /**
   * Suspend driver
   */
  async suspendDriver(driverId: number, adminId: number, reason: string): Promise<any> {
    const updated = await this.prisma.driver.update({
      where: { id: driverId },
      data: {
        is_active: 0,
        suspension_reason: reason,
        suspended_by: adminId,
        suspended_at: new Date(),
      },
    });

    await this.notificationService.sendToUser(driverId, 'driver', {
      title: 'Compte suspendu',
      body: `Votre compte a ete suspendu. Raison: ${reason}`,
      data: { type: 'driver_suspended', reason },
    });

    return updated;
  }

  /**
   * Reactivate driver
   */
  async reactivateDriver(driverId: number): Promise<any> {
    const updated = await this.prisma.driver.update({
      where: { id: driverId },
      data: {
        is_active: 1,
        suspension_reason: null,
        suspended_at: null,
        reactivated_at: new Date(),
      },
    });

    await this.notificationService.sendToUser(driverId, 'driver', {
      title: 'Compte reactive',
      body: 'Votre compte a ete reactive. Vous pouvez maintenant recevoir des courses.',
      data: { type: 'driver_reactivated' },
    });

    return updated;
  }

  // ============================================================================
  // USER MANAGEMENT
  // ============================================================================

  /**
   * Get users list
   */
  async getUsers(
    merchantId: number,
    options?: {
      status?: 'active' | 'inactive';
      search?: string;
      page?: number;
      limit?: number;
    },
  ): Promise<{ data: any[]; total: number }> {
    const page = options?.page || 1;
    const limit = options?.limit || 20;
    const skip = (page - 1) * limit;

    const where: any = { merchant_id: merchantId };

    if (options?.status === 'active') {
      where.is_active = 1;
    } else if (options?.status === 'inactive') {
      where.is_active = 0;
    }

    if (options?.search) {
      where.OR = [
        { first_name: { contains: options.search } },
        { last_name: { contains: options.search } },
        { email: { contains: options.search } },
        { mobile: { contains: options.search } },
      ];
    }

    const [users, total] = await Promise.all([
      this.prisma.user.findMany({
        where,
        skip,
        take: limit,
        orderBy: { created_at: 'desc' },
      }),
      this.prisma.user.count({ where }),
    ]);

    return { data: users, total };
  }

  /**
   * Block user
   */
  async blockUser(userId: number, adminId: number, reason: string): Promise<any> {
    const updated = await this.prisma.user.update({
      where: { id: userId },
      data: {
        is_active: 0,
        blocked_reason: reason,
        blocked_by: adminId,
        blocked_at: new Date(),
      },
    });

    await this.notificationService.sendToUser(userId, 'user', {
      title: 'Compte bloque',
      body: `Votre compte a ete bloque. Raison: ${reason}`,
      data: { type: 'user_blocked', reason },
    });

    return updated;
  }

  /**
   * Unblock user
   */
  async unblockUser(userId: number): Promise<any> {
    return this.prisma.user.update({
      where: { id: userId },
      data: {
        is_active: 1,
        blocked_reason: null,
        blocked_at: null,
      },
    });
  }

  // ============================================================================
  // BOOKING MANAGEMENT
  // ============================================================================

  /**
   * Get bookings list
   */
  async getBookings(
    merchantId: number,
    options?: {
      status?: string;
      startDate?: Date;
      endDate?: Date;
      driverId?: number;
      userId?: number;
      page?: number;
      limit?: number;
    },
  ): Promise<{ data: any[]; total: number }> {
    const page = options?.page || 1;
    const limit = options?.limit || 20;
    const skip = (page - 1) * limit;

    const where: any = { merchant_id: merchantId };

    if (options?.status) {
      where.status = options.status;
    }
    if (options?.driverId) {
      where.driver_id = options.driverId;
    }
    if (options?.userId) {
      where.user_id = options.userId;
    }
    if (options?.startDate || options?.endDate) {
      where.created_at = {};
      if (options.startDate) where.created_at.gte = options.startDate;
      if (options.endDate) where.created_at.lte = options.endDate;
    }

    const [bookings, total] = await Promise.all([
      this.prisma.booking.findMany({
        where,
        include: {
          user: { select: { first_name: true, last_name: true, mobile: true } },
          driver: { select: { first_name: true, last_name: true, mobile: true } },
        },
        skip,
        take: limit,
        orderBy: { created_at: 'desc' },
      }),
      this.prisma.booking.count({ where }),
    ]);

    return { data: bookings, total };
  }

  /**
   * Cancel booking (admin)
   */
  async cancelBooking(bookingId: number, adminId: number, reason: string): Promise<any> {
    const booking = await this.prisma.booking.findUnique({
      where: { id: bookingId },
    });

    if (!booking) {
      throw new NotFoundException('Reservation non trouvee');
    }

    if (booking.status === 'completed' || booking.status === 'cancelled') {
      throw new BadRequestException('Cette reservation ne peut pas etre annulee');
    }

    const updated = await this.prisma.booking.update({
      where: { id: bookingId },
      data: {
        status: 'cancelled',
        cancellation_reason: reason,
        cancelled_by: 'admin',
        cancelled_by_id: adminId,
        cancelled_at: new Date(),
      },
    });

    // Notify user and driver
    if (booking.user_id) {
      await this.notificationService.sendToUser(booking.user_id, 'user', {
        title: 'Reservation annulee',
        body: `Votre reservation a ete annulee par l'administration. Raison: ${reason}`,
        data: { type: 'booking_cancelled', booking_id: String(bookingId) },
      });
    }

    if (booking.driver_id) {
      await this.notificationService.sendToUser(booking.driver_id, 'driver', {
        title: 'Course annulee',
        body: `La course #${bookingId} a ete annulee par l'administration.`,
        data: { type: 'booking_cancelled', booking_id: String(bookingId) },
      });
    }

    return updated;
  }

  // ============================================================================
  // CONFIGURATION
  // ============================================================================

  /**
   * Get merchant settings
   */
  async getMerchantSettings(merchantId: number): Promise<any> {
    return this.prisma.merchantSetting.findMany({
      where: { merchant_id: merchantId },
    });
  }

  /**
   * Update merchant setting
   */
  async updateMerchantSetting(
    merchantId: number,
    key: string,
    value: string,
  ): Promise<any> {
    return this.prisma.merchantSetting.upsert({
      where: {
        merchant_id_key: { merchant_id: merchantId, key },
      },
      create: {
        merchant_id: merchantId,
        key,
        value,
        created_at: new Date(),
      },
      update: {
        value,
        updated_at: new Date(),
      },
    });
  }

  /**
   * Get vehicle types
   */
  async getVehicleTypes(merchantId: number): Promise<any[]> {
    return this.prisma.vehicleType.findMany({
      where: { merchant_id: merchantId },
      orderBy: { sort_order: 'asc' },
    });
  }

  /**
   * Update vehicle type
   */
  async updateVehicleType(vehicleTypeId: number, data: any): Promise<any> {
    return this.prisma.vehicleType.update({
      where: { id: vehicleTypeId },
      data: {
        ...(data.name && { name: data.name }),
        ...(data.baseFare !== undefined && { base_fare: data.baseFare }),
        ...(data.perKmRate !== undefined && { per_km_rate: data.perKmRate }),
        ...(data.perMinRate !== undefined && { per_min_rate: data.perMinRate }),
        ...(data.minFare !== undefined && { min_fare: data.minFare }),
        ...(data.capacity !== undefined && { capacity: data.capacity }),
        ...(data.isActive !== undefined && { is_active: data.isActive ? 1 : 0 }),
        updated_at: new Date(),
      },
    });
  }

  // ============================================================================
  // ACTIVITY LOG
  // ============================================================================

  /**
   * Log admin activity
   */
  async logActivity(
    adminId: number,
    merchantId: number,
    action: string,
    entityType: string,
    entityId: number,
    details?: any,
  ): Promise<void> {
    await this.prisma.adminActivityLog.create({
      data: {
        admin_id: adminId,
        merchant_id: merchantId,
        action,
        entity_type: entityType,
        entity_id: entityId,
        details: details ? JSON.stringify(details) : null,
        created_at: new Date(),
      },
    });
  }

  /**
   * Get activity logs
   */
  async getActivityLogs(
    merchantId: number,
    options?: {
      adminId?: number;
      action?: string;
      page?: number;
      limit?: number;
    },
  ): Promise<{ data: any[]; total: number }> {
    const page = options?.page || 1;
    const limit = options?.limit || 50;
    const skip = (page - 1) * limit;

    const where: any = { merchant_id: merchantId };
    if (options?.adminId) where.admin_id = options.adminId;
    if (options?.action) where.action = options.action;

    const [logs, total] = await Promise.all([
      this.prisma.adminActivityLog.findMany({
        where,
        skip,
        take: limit,
        orderBy: { created_at: 'desc' },
      }),
      this.prisma.adminActivityLog.count({ where }),
    ]);

    return { data: logs, total };
  }

  // ============================================================================
  // BROADCAST NOTIFICATIONS
  // ============================================================================

  /**
   * Send broadcast notification
   */
  async sendBroadcastNotification(
    merchantId: number,
    target: 'all' | 'users' | 'drivers',
    title: string,
    body: string,
    data?: any,
  ): Promise<{ sent: number }> {
    let users: { id: number }[] = [];
    let drivers: { id: number }[] = [];

    if (target === 'all' || target === 'users') {
      users = await this.prisma.user.findMany({
        where: { merchant_id: merchantId, is_active: 1 },
        select: { id: true },
      });
    }

    if (target === 'all' || target === 'drivers') {
      drivers = await this.prisma.driver.findMany({
        where: { merchant_id: merchantId, is_active: 1 },
        select: { id: true },
      });
    }

    let sent = 0;

    for (const user of users) {
      try {
        await this.notificationService.sendToUser(user.id, 'user', {
          title,
          body,
          data,
        });
        sent++;
      } catch (error) {
        this.logger.error(`Failed to send to user ${user.id}: ${error.message}`);
      }
    }

    for (const driver of drivers) {
      try {
        await this.notificationService.sendToUser(driver.id, 'driver', {
          title,
          body,
          data,
        });
        sent++;
      } catch (error) {
        this.logger.error(`Failed to send to driver ${driver.id}: ${error.message}`);
      }
    }

    return { sent };
  }
}
