import {
  Controller,
  Get,
  Post,
  Put,
  Body,
  Param,
  Query,
  Res,
  UseGuards,
  ParseIntPipe,
} from '@nestjs/common';
import { Response } from 'express';
import { ApiTags, ApiOperation, ApiBearerAuth } from '@nestjs/swagger';
import { AdminService } from './admin.service';
import { DashboardService } from './dashboard.service';
import { ReportsService } from './reports.service';
import { JwtAuthGuard } from '../../common/guards/jwt-auth.guard';
import { MerchantGuard } from '../../common/guards/merchant.guard';
import { MerchantId, CurrentUser } from '../../common/decorators/merchant.decorator';

@ApiTags('Admin')
@Controller('api/admin')
@UseGuards(JwtAuthGuard, MerchantGuard)
@ApiBearerAuth()
export class AdminController {
  constructor(
    private adminService: AdminService,
    private dashboardService: DashboardService,
    private reportsService: ReportsService,
  ) {}

  // ============================================================================
  // DASHBOARD
  // ============================================================================

  @Get('dashboard')
  @ApiOperation({ summary: 'Get dashboard statistics' })
  async getDashboardStats(@MerchantId() merchantId: number) {
    return this.dashboardService.getDashboardStats(merchantId);
  }

  @Get('dashboard/trends')
  @ApiOperation({ summary: 'Get booking trends' })
  async getBookingTrends(
    @MerchantId() merchantId: number,
    @Query('days') days?: string,
  ) {
    return this.dashboardService.getBookingTrends(
      merchantId,
      days ? parseInt(days) : 30,
    );
  }

  @Get('dashboard/peak-hours')
  @ApiOperation({ summary: 'Get peak hours analysis' })
  async getPeakHours(
    @MerchantId() merchantId: number,
    @Query('days') days?: string,
  ) {
    return this.dashboardService.getPeakHoursAnalysis(
      merchantId,
      days ? parseInt(days) : 30,
    );
  }

  @Get('dashboard/top-drivers')
  @ApiOperation({ summary: 'Get top drivers' })
  async getTopDrivers(
    @MerchantId() merchantId: number,
    @Query('limit') limit?: string,
  ) {
    return this.dashboardService.getTopDrivers(
      merchantId,
      limit ? parseInt(limit) : 10,
    );
  }

  @Get('dashboard/revenue-by-payment')
  @ApiOperation({ summary: 'Get revenue by payment method' })
  async getRevenueByPayment(
    @MerchantId() merchantId: number,
    @Query('startDate') startDate?: string,
    @Query('endDate') endDate?: string,
  ) {
    return this.dashboardService.getRevenueByPaymentMethod(
      merchantId,
      startDate ? new Date(startDate) : undefined,
      endDate ? new Date(endDate) : undefined,
    );
  }

  @Get('dashboard/heatmap')
  @ApiOperation({ summary: 'Get heatmap data' })
  async getHeatmapData(
    @MerchantId() merchantId: number,
    @Query('startDate') startDate?: string,
  ) {
    return this.dashboardService.getHeatmapData(
      merchantId,
      startDate ? new Date(startDate) : undefined,
    );
  }

  // ============================================================================
  // DRIVERS
  // ============================================================================

  @Get('drivers')
  @ApiOperation({ summary: 'Get drivers list' })
  async getDrivers(
    @MerchantId() merchantId: number,
    @Query('status') status?: 'active' | 'inactive' | 'pending',
    @Query('search') search?: string,
    @Query('page') page?: string,
    @Query('limit') limit?: string,
  ) {
    return this.adminService.getDrivers(merchantId, {
      status,
      search,
      page: page ? parseInt(page) : 1,
      limit: limit ? parseInt(limit) : 20,
    });
  }

  @Post('drivers/:id/approve')
  @ApiOperation({ summary: 'Approve driver' })
  async approveDriver(
    @Param('id', ParseIntPipe) driverId: number,
    @CurrentUser() admin: any,
  ) {
    return this.adminService.approveDriver(driverId, admin.id);
  }

  @Post('drivers/:id/reject')
  @ApiOperation({ summary: 'Reject driver' })
  async rejectDriver(
    @Param('id', ParseIntPipe) driverId: number,
    @CurrentUser() admin: any,
    @Body() body: { reason: string },
  ) {
    return this.adminService.rejectDriver(driverId, admin.id, body.reason);
  }

  @Post('drivers/:id/suspend')
  @ApiOperation({ summary: 'Suspend driver' })
  async suspendDriver(
    @Param('id', ParseIntPipe) driverId: number,
    @CurrentUser() admin: any,
    @Body() body: { reason: string },
  ) {
    return this.adminService.suspendDriver(driverId, admin.id, body.reason);
  }

  @Post('drivers/:id/reactivate')
  @ApiOperation({ summary: 'Reactivate driver' })
  async reactivateDriver(@Param('id', ParseIntPipe) driverId: number) {
    return this.adminService.reactivateDriver(driverId);
  }

  // ============================================================================
  // USERS
  // ============================================================================

  @Get('users')
  @ApiOperation({ summary: 'Get users list' })
  async getUsers(
    @MerchantId() merchantId: number,
    @Query('status') status?: 'active' | 'inactive',
    @Query('search') search?: string,
    @Query('page') page?: string,
    @Query('limit') limit?: string,
  ) {
    return this.adminService.getUsers(merchantId, {
      status,
      search,
      page: page ? parseInt(page) : 1,
      limit: limit ? parseInt(limit) : 20,
    });
  }

  @Post('users/:id/block')
  @ApiOperation({ summary: 'Block user' })
  async blockUser(
    @Param('id', ParseIntPipe) userId: number,
    @CurrentUser() admin: any,
    @Body() body: { reason: string },
  ) {
    return this.adminService.blockUser(userId, admin.id, body.reason);
  }

  @Post('users/:id/unblock')
  @ApiOperation({ summary: 'Unblock user' })
  async unblockUser(@Param('id', ParseIntPipe) userId: number) {
    return this.adminService.unblockUser(userId);
  }

  // ============================================================================
  // BOOKINGS
  // ============================================================================

  @Get('bookings')
  @ApiOperation({ summary: 'Get bookings list' })
  async getBookings(
    @MerchantId() merchantId: number,
    @Query('status') status?: string,
    @Query('startDate') startDate?: string,
    @Query('endDate') endDate?: string,
    @Query('driverId') driverId?: string,
    @Query('userId') userId?: string,
    @Query('page') page?: string,
    @Query('limit') limit?: string,
  ) {
    return this.adminService.getBookings(merchantId, {
      status,
      startDate: startDate ? new Date(startDate) : undefined,
      endDate: endDate ? new Date(endDate) : undefined,
      driverId: driverId ? parseInt(driverId) : undefined,
      userId: userId ? parseInt(userId) : undefined,
      page: page ? parseInt(page) : 1,
      limit: limit ? parseInt(limit) : 20,
    });
  }

  @Post('bookings/:id/cancel')
  @ApiOperation({ summary: 'Cancel booking' })
  async cancelBooking(
    @Param('id', ParseIntPipe) bookingId: number,
    @CurrentUser() admin: any,
    @Body() body: { reason: string },
  ) {
    return this.adminService.cancelBooking(bookingId, admin.id, body.reason);
  }

  // ============================================================================
  // REPORTS
  // ============================================================================

  @Get('reports/:type')
  @ApiOperation({ summary: 'Get report data (JSON)' })
  async getReportData(
    @MerchantId() merchantId: number,
    @Param('type') type: 'bookings' | 'drivers' | 'users' | 'financial',
    @Query('startDate') startDate?: string,
    @Query('endDate') endDate?: string,
  ) {
    return this.reportsService.getReportData(merchantId, type, {
      startDate: startDate ? new Date(startDate) : undefined,
      endDate: endDate ? new Date(endDate) : undefined,
    });
  }

  @Get('reports/:type/export')
  @ApiOperation({ summary: 'Export report (Excel)' })
  async exportReport(
    @MerchantId() merchantId: number,
    @Param('type') type: 'bookings' | 'drivers' | 'users' | 'financial',
    @Query('startDate') startDate?: string,
    @Query('endDate') endDate?: string,
    @Res() res: Response,
  ) {
    const filters = {
      startDate: startDate ? new Date(startDate) : undefined,
      endDate: endDate ? new Date(endDate) : undefined,
    };

    let buffer: Buffer;
    let filename: string;

    switch (type) {
      case 'bookings':
        buffer = await this.reportsService.generateBookingsReport(merchantId, filters);
        filename = `reservations_${new Date().toISOString().split('T')[0]}.xlsx`;
        break;
      case 'drivers':
        buffer = await this.reportsService.generateDriversReport(merchantId, filters);
        filename = `chauffeurs_${new Date().toISOString().split('T')[0]}.xlsx`;
        break;
      case 'users':
        buffer = await this.reportsService.generateUsersReport(merchantId, filters);
        filename = `utilisateurs_${new Date().toISOString().split('T')[0]}.xlsx`;
        break;
      case 'financial':
        buffer = await this.reportsService.generateFinancialReport(merchantId, filters);
        filename = `rapport_financier_${new Date().toISOString().split('T')[0]}.xlsx`;
        break;
      default:
        throw new Error('Invalid report type');
    }

    res.set({
      'Content-Type': 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
      'Content-Disposition': `attachment; filename="${filename}"`,
    });

    res.send(buffer);
  }

  // ============================================================================
  // CONFIGURATION
  // ============================================================================

  @Get('settings')
  @ApiOperation({ summary: 'Get merchant settings' })
  async getMerchantSettings(@MerchantId() merchantId: number) {
    return this.adminService.getMerchantSettings(merchantId);
  }

  @Put('settings')
  @ApiOperation({ summary: 'Update merchant setting' })
  async updateMerchantSetting(
    @MerchantId() merchantId: number,
    @Body() body: { key: string; value: string },
  ) {
    return this.adminService.updateMerchantSetting(merchantId, body.key, body.value);
  }

  @Get('vehicle-types')
  @ApiOperation({ summary: 'Get vehicle types' })
  async getVehicleTypes(@MerchantId() merchantId: number) {
    return this.adminService.getVehicleTypes(merchantId);
  }

  @Put('vehicle-types/:id')
  @ApiOperation({ summary: 'Update vehicle type' })
  async updateVehicleType(
    @Param('id', ParseIntPipe) vehicleTypeId: number,
    @Body() body: any,
  ) {
    return this.adminService.updateVehicleType(vehicleTypeId, body);
  }

  // ============================================================================
  // ACTIVITY LOGS
  // ============================================================================

  @Get('activity-logs')
  @ApiOperation({ summary: 'Get activity logs' })
  async getActivityLogs(
    @MerchantId() merchantId: number,
    @Query('adminId') adminId?: string,
    @Query('action') action?: string,
    @Query('page') page?: string,
    @Query('limit') limit?: string,
  ) {
    return this.adminService.getActivityLogs(merchantId, {
      adminId: adminId ? parseInt(adminId) : undefined,
      action,
      page: page ? parseInt(page) : 1,
      limit: limit ? parseInt(limit) : 50,
    });
  }

  // ============================================================================
  // BROADCAST NOTIFICATIONS
  // ============================================================================

  @Post('broadcast')
  @ApiOperation({ summary: 'Send broadcast notification' })
  async sendBroadcast(
    @MerchantId() merchantId: number,
    @Body()
    body: {
      target: 'all' | 'users' | 'drivers';
      title: string;
      body: string;
      data?: any;
    },
  ) {
    return this.adminService.sendBroadcastNotification(
      merchantId,
      body.target,
      body.title,
      body.body,
      body.data,
    );
  }
}
