import {
  Injectable,
  NestInterceptor,
  ExecutionContext,
  CallHandler,
} from '@nestjs/common';
import { Observable } from 'rxjs';
import { map } from 'rxjs/operators';
import { ConfigService } from '@nestjs/config';

/**
 * Response Interceptor - Format API standard compatible avec Laravel
 *
 * Format de reponse:
 * {
 *   "version": "1.5",
 *   "result": "1",
 *   "message": "Success",
 *   "data": { ... }
 * }
 */
export interface ApiResponse<T> {
  version: string;
  result: '0' | '1';
  message: string;
  data?: T;
}

@Injectable()
export class ResponseInterceptor<T> implements NestInterceptor<T, ApiResponse<T>> {
  constructor(private configService: ConfigService) {}

  intercept(
    context: ExecutionContext,
    next: CallHandler,
  ): Observable<ApiResponse<T>> {
    const apiVersion = this.configService.get<string>('app.apiVersion', '1.5');

    return next.handle().pipe(
      map((response) => {
        // Si la reponse est deja formatee (par exemple erreur)
        if (response && response.version && response.result !== undefined) {
          return response;
        }

        // Format standard
        const formatted: ApiResponse<T> = {
          version: apiVersion,
          result: '1',
          message: response?.message || 'Success',
          ...(response?.data !== undefined
            ? { data: response.data }
            : response && !response.message
            ? { data: response }
            : {}),
        };

        return formatted;
      }),
    );
  }
}
