// =============================================================================
// MonkAPI - Custom Exceptions
// =============================================================================

import { HttpException, HttpStatus } from '@nestjs/common';
import { ERROR_CODES } from '../constants';

// =============================================================================
// BASE EXCEPTION
// =============================================================================

export class ApiException extends HttpException {
  public readonly errorCode: string;
  public readonly details?: Record<string, any>;

  constructor(
    message: string,
    statusCode: HttpStatus,
    errorCode: string,
    details?: Record<string, any>,
  ) {
    super(
      {
        success: false,
        message,
        errorCode,
        details,
      },
      statusCode,
    );
    this.errorCode = errorCode;
    this.details = details;
  }
}

// =============================================================================
// AUTH EXCEPTIONS
// =============================================================================

export class InvalidCredentialsException extends ApiException {
  constructor(message = 'Email/telephone ou mot de passe incorrect') {
    super(message, HttpStatus.UNAUTHORIZED, ERROR_CODES.AUTH_INVALID_CREDENTIALS);
  }
}

export class TokenExpiredException extends ApiException {
  constructor(message = 'Token expire') {
    super(message, HttpStatus.UNAUTHORIZED, ERROR_CODES.AUTH_TOKEN_EXPIRED);
  }
}

export class InvalidTokenException extends ApiException {
  constructor(message = 'Token invalide') {
    super(message, HttpStatus.UNAUTHORIZED, ERROR_CODES.AUTH_TOKEN_INVALID);
  }
}

export class InvalidOtpException extends ApiException {
  constructor(message = 'Code OTP invalide') {
    super(message, HttpStatus.UNAUTHORIZED, ERROR_CODES.AUTH_OTP_INVALID);
  }
}

export class OtpExpiredException extends ApiException {
  constructor(message = 'Code OTP expire') {
    super(message, HttpStatus.UNAUTHORIZED, ERROR_CODES.AUTH_OTP_EXPIRED);
  }
}

export class AccountSuspendedException extends ApiException {
  constructor(message = 'Compte suspendu') {
    super(message, HttpStatus.FORBIDDEN, ERROR_CODES.AUTH_ACCOUNT_SUSPENDED);
  }
}

export class AccountNotVerifiedException extends ApiException {
  constructor(message = 'Compte non verifie') {
    super(message, HttpStatus.FORBIDDEN, ERROR_CODES.AUTH_ACCOUNT_NOT_VERIFIED);
  }
}

// =============================================================================
// USER EXCEPTIONS
// =============================================================================

export class UserNotFoundException extends ApiException {
  constructor(message = 'Utilisateur non trouve') {
    super(message, HttpStatus.NOT_FOUND, ERROR_CODES.USER_NOT_FOUND);
  }
}

export class EmailExistsException extends ApiException {
  constructor(message = 'Cet email est deja utilise') {
    super(message, HttpStatus.BAD_REQUEST, ERROR_CODES.USER_EMAIL_EXISTS);
  }
}

export class PhoneExistsException extends ApiException {
  constructor(message = 'Ce numero de telephone est deja utilise') {
    super(message, HttpStatus.BAD_REQUEST, ERROR_CODES.USER_PHONE_EXISTS);
  }
}

// =============================================================================
// DRIVER EXCEPTIONS
// =============================================================================

export class DriverNotFoundException extends ApiException {
  constructor(message = 'Chauffeur non trouve') {
    super(message, HttpStatus.NOT_FOUND, ERROR_CODES.DRIVER_NOT_FOUND);
  }
}

export class DriverNotApprovedException extends ApiException {
  constructor(message = 'Chauffeur non approuve') {
    super(message, HttpStatus.FORBIDDEN, ERROR_CODES.DRIVER_NOT_APPROVED);
  }
}

export class DriverNotOnlineException extends ApiException {
  constructor(message = 'Chauffeur hors ligne') {
    super(message, HttpStatus.BAD_REQUEST, ERROR_CODES.DRIVER_NOT_ONLINE);
  }
}

export class DriverBusyException extends ApiException {
  constructor(message = 'Chauffeur occupe') {
    super(message, HttpStatus.BAD_REQUEST, ERROR_CODES.DRIVER_BUSY);
  }
}

// =============================================================================
// BOOKING EXCEPTIONS
// =============================================================================

export class BookingNotFoundException extends ApiException {
  constructor(message = 'Reservation non trouvee') {
    super(message, HttpStatus.NOT_FOUND, ERROR_CODES.BOOKING_NOT_FOUND);
  }
}

export class InvalidBookingStatusException extends ApiException {
  constructor(
    currentStatus: string,
    attemptedStatus: string,
    message?: string,
  ) {
    super(
      message || `Impossible de passer de "${currentStatus}" a "${attemptedStatus}"`,
      HttpStatus.BAD_REQUEST,
      ERROR_CODES.BOOKING_INVALID_STATUS,
      { currentStatus, attemptedStatus },
    );
  }
}

export class BookingAlreadyAssignedException extends ApiException {
  constructor(message = 'Cette reservation est deja assignee') {
    super(message, HttpStatus.BAD_REQUEST, ERROR_CODES.BOOKING_ALREADY_ASSIGNED);
  }
}

export class ZoneRestrictedException extends ApiException {
  constructor(
    zoneName: string,
    message?: string,
  ) {
    super(
      message || `Zone restreinte: ${zoneName}`,
      HttpStatus.BAD_REQUEST,
      ERROR_CODES.BOOKING_ZONE_RESTRICTED,
      { zoneName },
    );
  }
}

export class NoDriversAvailableException extends ApiException {
  constructor(message = 'Aucun chauffeur disponible pour le moment') {
    super(message, HttpStatus.SERVICE_UNAVAILABLE, ERROR_CODES.BOOKING_NO_DRIVERS);
  }
}

// =============================================================================
// PAYMENT EXCEPTIONS
// =============================================================================

export class PaymentFailedException extends ApiException {
  constructor(
    reason: string,
    details?: Record<string, any>,
  ) {
    super(
      `Paiement echoue: ${reason}`,
      HttpStatus.PAYMENT_REQUIRED,
      ERROR_CODES.PAYMENT_FAILED,
      details,
    );
  }
}

export class InsufficientBalanceException extends ApiException {
  constructor(
    required: number,
    available: number,
    message?: string,
  ) {
    super(
      message || 'Solde insuffisant',
      HttpStatus.BAD_REQUEST,
      ERROR_CODES.PAYMENT_INSUFFICIENT_BALANCE,
      { required, available },
    );
  }
}

export class InvalidPaymentMethodException extends ApiException {
  constructor(message = 'Methode de paiement invalide') {
    super(message, HttpStatus.BAD_REQUEST, ERROR_CODES.PAYMENT_METHOD_INVALID);
  }
}

// =============================================================================
// PROMO EXCEPTIONS
// =============================================================================

export class InvalidPromoCodeException extends ApiException {
  constructor(code: string, reason?: string) {
    super(
      reason || `Code promo "${code}" invalide`,
      HttpStatus.BAD_REQUEST,
      ERROR_CODES.PROMO_INVALID,
      { code },
    );
  }
}

export class PromoCodeExpiredException extends ApiException {
  constructor(code: string) {
    super(
      `Code promo "${code}" expire`,
      HttpStatus.BAD_REQUEST,
      ERROR_CODES.PROMO_EXPIRED,
      { code },
    );
  }
}

export class PromoUsageExceededException extends ApiException {
  constructor(code: string) {
    super(
      `Limite d'utilisation atteinte pour le code "${code}"`,
      HttpStatus.BAD_REQUEST,
      ERROR_CODES.PROMO_USAGE_EXCEEDED,
      { code },
    );
  }
}

export class PromoMinOrderException extends ApiException {
  constructor(code: string, minOrder: number) {
    super(
      `Commande minimum de ${minOrder} requise pour le code "${code}"`,
      HttpStatus.BAD_REQUEST,
      ERROR_CODES.PROMO_MIN_ORDER,
      { code, minOrder },
    );
  }
}

// =============================================================================
// GENERAL EXCEPTIONS
// =============================================================================

export class ValidationException extends ApiException {
  constructor(errors: Record<string, string[]>) {
    super(
      'Erreurs de validation',
      HttpStatus.BAD_REQUEST,
      ERROR_CODES.VALIDATION_ERROR,
      { errors },
    );
  }
}

export class ResourceNotFoundException extends ApiException {
  constructor(resource: string, id?: number | string) {
    super(
      id ? `${resource} #${id} non trouve` : `${resource} non trouve`,
      HttpStatus.NOT_FOUND,
      ERROR_CODES.NOT_FOUND,
      { resource, id },
    );
  }
}

export class ForbiddenException extends ApiException {
  constructor(message = 'Acces refuse') {
    super(message, HttpStatus.FORBIDDEN, ERROR_CODES.FORBIDDEN);
  }
}

export class RateLimitedException extends ApiException {
  constructor(retryAfter?: number) {
    super(
      'Trop de requetes. Veuillez reessayer plus tard.',
      HttpStatus.TOO_MANY_REQUESTS,
      ERROR_CODES.RATE_LIMITED,
      retryAfter ? { retryAfter } : undefined,
    );
  }
}

// =============================================================================
// HELPER FUNCTION
// =============================================================================

export function throwIf(condition: boolean, exception: ApiException): void {
  if (condition) {
    throw exception;
  }
}

export function throwIfNull<T>(
  value: T | null | undefined,
  exception: ApiException,
): asserts value is T {
  if (value === null || value === undefined) {
    throw exception;
  }
}
