// =============================================================================
// MonkAPI - Shared DTOs with Swagger Decorators
// =============================================================================

import { ApiProperty, ApiPropertyOptional } from '@nestjs/swagger';
import {
  IsString,
  IsEmail,
  IsNumber,
  IsOptional,
  IsBoolean,
  IsArray,
  IsEnum,
  IsDate,
  IsNotEmpty,
  MinLength,
  MaxLength,
  Min,
  Max,
  IsLatitude,
  IsLongitude,
  ValidateNested,
} from 'class-validator';
import { Type, Transform } from 'class-transformer';

// =============================================================================
// PAGINATION
// =============================================================================

export class PaginationDto {
  @ApiPropertyOptional({ description: 'Page number', default: 1, minimum: 1 })
  @IsOptional()
  @IsNumber()
  @Min(1)
  @Type(() => Number)
  page?: number = 1;

  @ApiPropertyOptional({ description: 'Items per page', default: 20, minimum: 1, maximum: 100 })
  @IsOptional()
  @IsNumber()
  @Min(1)
  @Max(100)
  @Type(() => Number)
  limit?: number = 20;

  @ApiPropertyOptional({ description: 'Sort field' })
  @IsOptional()
  @IsString()
  sortBy?: string;

  @ApiPropertyOptional({ description: 'Sort direction', enum: ['asc', 'desc'] })
  @IsOptional()
  @IsEnum(['asc', 'desc'])
  sortOrder?: 'asc' | 'desc' = 'desc';
}

export class PaginatedResponseDto<T> {
  @ApiProperty({ description: 'Data items' })
  data: T[];

  @ApiProperty({ description: 'Total items count' })
  total: number;

  @ApiProperty({ description: 'Current page' })
  page: number;

  @ApiProperty({ description: 'Items per page' })
  limit: number;

  @ApiProperty({ description: 'Total pages' })
  totalPages: number;

  @ApiProperty({ description: 'Has more pages' })
  hasMore: boolean;
}

// =============================================================================
// LOCATION
// =============================================================================

export class LocationDto {
  @ApiProperty({ description: 'Latitude', example: 6.1319 })
  @IsLatitude()
  latitude: number;

  @ApiProperty({ description: 'Longitude', example: 1.2228 })
  @IsLongitude()
  longitude: number;

  @ApiPropertyOptional({ description: 'Address', example: '123 Rue du Commerce, Lome' })
  @IsOptional()
  @IsString()
  address?: string;

  @ApiPropertyOptional({ description: 'Place name', example: 'Hotel 2 Fevrier' })
  @IsOptional()
  @IsString()
  name?: string;
}

// =============================================================================
// AUTH DTOs
// =============================================================================

export class LoginDto {
  @ApiProperty({ description: 'Email or phone number', example: 'user@example.com' })
  @IsNotEmpty()
  @IsString()
  identifier: string;

  @ApiProperty({ description: 'Password', example: 'password123', minLength: 6 })
  @IsNotEmpty()
  @IsString()
  @MinLength(6)
  password: string;

  @ApiPropertyOptional({ description: 'Device type', enum: [1, 2, 3], example: 1 })
  @IsOptional()
  @IsNumber()
  deviceType?: number;

  @ApiPropertyOptional({ description: 'FCM token for push notifications' })
  @IsOptional()
  @IsString()
  fcmToken?: string;
}

export class SignupDto {
  @ApiProperty({ description: 'First name', example: 'John' })
  @IsNotEmpty()
  @IsString()
  @MaxLength(255)
  firstName: string;

  @ApiProperty({ description: 'Last name', example: 'Doe' })
  @IsNotEmpty()
  @IsString()
  @MaxLength(255)
  lastName: string;

  @ApiProperty({ description: 'Email address', example: 'john.doe@example.com' })
  @IsNotEmpty()
  @IsEmail()
  email: string;

  @ApiProperty({ description: 'Phone number', example: '+22890123456' })
  @IsNotEmpty()
  @IsString()
  phone: string;

  @ApiProperty({ description: 'Password', minLength: 6 })
  @IsNotEmpty()
  @IsString()
  @MinLength(6)
  password: string;

  @ApiPropertyOptional({ description: 'Referral code' })
  @IsOptional()
  @IsString()
  referralCode?: string;

  @ApiPropertyOptional({ description: 'Country code', example: 'TG' })
  @IsOptional()
  @IsString()
  countryCode?: string;
}

export class OtpDto {
  @ApiProperty({ description: 'Phone number or email' })
  @IsNotEmpty()
  @IsString()
  identifier: string;

  @ApiProperty({ description: 'OTP code', example: '123456' })
  @IsNotEmpty()
  @IsString()
  @MinLength(4)
  @MaxLength(6)
  otp: string;
}

export class SocialLoginDto {
  @ApiProperty({ description: 'Social provider', enum: ['google', 'facebook', 'apple'] })
  @IsNotEmpty()
  @IsEnum(['google', 'facebook', 'apple'])
  provider: 'google' | 'facebook' | 'apple';

  @ApiProperty({ description: 'Access token from social provider' })
  @IsNotEmpty()
  @IsString()
  accessToken: string;

  @ApiPropertyOptional({ description: 'ID token (for Apple)' })
  @IsOptional()
  @IsString()
  idToken?: string;
}

export class TokenResponseDto {
  @ApiProperty({ description: 'Access token' })
  accessToken: string;

  @ApiProperty({ description: 'Refresh token' })
  refreshToken: string;

  @ApiProperty({ description: 'Token expiry in seconds' })
  expiresIn: number;

  @ApiProperty({ description: 'Token type', example: 'Bearer' })
  tokenType: string;
}

// =============================================================================
// BOOKING DTOs
// =============================================================================

export enum BookingStatus {
  PENDING = 'pending',
  SEARCHING = 'searching',
  ACCEPTED = 'accepted',
  ARRIVED = 'arrived',
  STARTED = 'started',
  COMPLETED = 'completed',
  CANCELLED = 'cancelled',
}

export class CreateBookingDto {
  @ApiProperty({ description: 'Vehicle type ID' })
  @IsNumber()
  vehicleTypeId: number;

  @ApiProperty({ description: 'Pickup location', type: LocationDto })
  @ValidateNested()
  @Type(() => LocationDto)
  pickup: LocationDto;

  @ApiProperty({ description: 'Dropoff location', type: LocationDto })
  @ValidateNested()
  @Type(() => LocationDto)
  dropoff: LocationDto;

  @ApiPropertyOptional({ description: 'Additional stops', type: [LocationDto] })
  @IsOptional()
  @IsArray()
  @ValidateNested({ each: true })
  @Type(() => LocationDto)
  stops?: LocationDto[];

  @ApiPropertyOptional({ description: 'Payment method', enum: ['cash', 'card', 'wallet', 'mobile_money'] })
  @IsOptional()
  @IsEnum(['cash', 'card', 'wallet', 'mobile_money'])
  paymentMethod?: string;

  @ApiPropertyOptional({ description: 'Promo code' })
  @IsOptional()
  @IsString()
  promoCode?: string;

  @ApiPropertyOptional({ description: 'Scheduled time (ISO format)' })
  @IsOptional()
  @IsDate()
  @Type(() => Date)
  scheduledAt?: Date;

  @ApiPropertyOptional({ description: 'Notes for driver' })
  @IsOptional()
  @IsString()
  @MaxLength(500)
  notes?: string;
}

export class EstimateRequestDto {
  @ApiProperty({ description: 'Pickup location', type: LocationDto })
  @ValidateNested()
  @Type(() => LocationDto)
  pickup: LocationDto;

  @ApiProperty({ description: 'Dropoff location', type: LocationDto })
  @ValidateNested()
  @Type(() => LocationDto)
  dropoff: LocationDto;

  @ApiPropertyOptional({ description: 'Additional stops', type: [LocationDto] })
  @IsOptional()
  @IsArray()
  @ValidateNested({ each: true })
  @Type(() => LocationDto)
  stops?: LocationDto[];

  @ApiPropertyOptional({ description: 'Promo code' })
  @IsOptional()
  @IsString()
  promoCode?: string;
}

export class EstimateResponseDto {
  @ApiProperty({ description: 'Vehicle type ID' })
  vehicleTypeId: number;

  @ApiProperty({ description: 'Vehicle type name' })
  vehicleTypeName: string;

  @ApiProperty({ description: 'Estimated fare' })
  estimatedFare: number;

  @ApiProperty({ description: 'Distance in km' })
  distance: number;

  @ApiProperty({ description: 'Duration in minutes' })
  duration: number;

  @ApiProperty({ description: 'Surge multiplier' })
  surgeMultiplier: number;

  @ApiPropertyOptional({ description: 'Discount amount' })
  discountAmount?: number;

  @ApiProperty({ description: 'Currency' })
  currency: string;
}

export class BookingResponseDto {
  @ApiProperty({ description: 'Booking ID' })
  id: number;

  @ApiProperty({ description: 'Booking number', example: 'BK-202601-12345' })
  bookingNumber: string;

  @ApiProperty({ description: 'Booking status', enum: BookingStatus })
  status: BookingStatus;

  @ApiProperty({ description: 'Pickup location' })
  pickup: LocationDto;

  @ApiProperty({ description: 'Dropoff location' })
  dropoff: LocationDto;

  @ApiPropertyOptional({ description: 'Driver info' })
  driver?: {
    id: number;
    name: string;
    phone: string;
    rating: number;
    profileImage: string;
    vehicle: {
      make: string;
      model: string;
      color: string;
      plateNumber: string;
    };
    location?: {
      latitude: number;
      longitude: number;
    };
  };

  @ApiProperty({ description: 'Estimated amount' })
  estimatedAmount: number;

  @ApiPropertyOptional({ description: 'Final amount' })
  finalAmount?: number;

  @ApiProperty({ description: 'Payment method' })
  paymentMethod: string;

  @ApiProperty({ description: 'Created at' })
  createdAt: Date;
}

// =============================================================================
// RATING DTOs
// =============================================================================

export class RateBookingDto {
  @ApiProperty({ description: 'Rating (1-5)', minimum: 1, maximum: 5 })
  @IsNumber()
  @Min(1)
  @Max(5)
  rating: number;

  @ApiPropertyOptional({ description: 'Review comment' })
  @IsOptional()
  @IsString()
  @MaxLength(1000)
  comment?: string;

  @ApiPropertyOptional({ description: 'Rating tags', example: ['clean_car', 'friendly_driver'] })
  @IsOptional()
  @IsArray()
  @IsString({ each: true })
  tags?: string[];

  @ApiPropertyOptional({ description: 'Tip amount' })
  @IsOptional()
  @IsNumber()
  @Min(0)
  tipAmount?: number;
}

// =============================================================================
// DELIVERY DTOs
// =============================================================================

export class CreateDeliveryDto {
  @ApiProperty({ description: 'Vehicle type ID' })
  @IsNumber()
  vehicleTypeId: number;

  @ApiProperty({ description: 'Pickup location', type: LocationDto })
  @ValidateNested()
  @Type(() => LocationDto)
  pickup: LocationDto;

  @ApiProperty({ description: 'Dropoff location', type: LocationDto })
  @ValidateNested()
  @Type(() => LocationDto)
  dropoff: LocationDto;

  @ApiProperty({ description: 'Sender name' })
  @IsString()
  @MaxLength(255)
  senderName: string;

  @ApiProperty({ description: 'Sender phone' })
  @IsString()
  senderPhone: string;

  @ApiProperty({ description: 'Receiver name' })
  @IsString()
  @MaxLength(255)
  receiverName: string;

  @ApiProperty({ description: 'Receiver phone' })
  @IsString()
  receiverPhone: string;

  @ApiPropertyOptional({ description: 'Package type', enum: ['document', 'parcel', 'food', 'fragile', 'other'] })
  @IsOptional()
  @IsEnum(['document', 'parcel', 'food', 'fragile', 'other'])
  packageType?: string;

  @ApiPropertyOptional({ description: 'Package description' })
  @IsOptional()
  @IsString()
  @MaxLength(500)
  packageDescription?: string;

  @ApiPropertyOptional({ description: 'Is fragile' })
  @IsOptional()
  @IsBoolean()
  fragile?: boolean;

  @ApiPropertyOptional({ description: 'Requires signature' })
  @IsOptional()
  @IsBoolean()
  signatureRequired?: boolean;

  @ApiPropertyOptional({ description: 'Cash on delivery amount' })
  @IsOptional()
  @IsNumber()
  @Min(0)
  codAmount?: number;

  @ApiPropertyOptional({ description: 'Special instructions' })
  @IsOptional()
  @IsString()
  @MaxLength(500)
  instructions?: string;

  @ApiPropertyOptional({ description: 'Scheduled pickup time' })
  @IsOptional()
  @IsDate()
  @Type(() => Date)
  scheduledAt?: Date;
}

// =============================================================================
// PAYMENT DTOs
// =============================================================================

export class AddPaymentMethodDto {
  @ApiProperty({ description: 'Payment type', enum: ['card', 'mobile_money'] })
  @IsEnum(['card', 'mobile_money'])
  type: 'card' | 'mobile_money';

  @ApiPropertyOptional({ description: 'Card token (from Stripe)' })
  @IsOptional()
  @IsString()
  cardToken?: string;

  @ApiPropertyOptional({ description: 'Phone number (for mobile money)' })
  @IsOptional()
  @IsString()
  phoneNumber?: string;

  @ApiPropertyOptional({ description: 'Mobile money provider', enum: ['mtn', 'orange', 'moov', 'flooz'] })
  @IsOptional()
  @IsEnum(['mtn', 'orange', 'moov', 'flooz'])
  provider?: string;

  @ApiPropertyOptional({ description: 'Set as default' })
  @IsOptional()
  @IsBoolean()
  setAsDefault?: boolean;
}

export class TopUpWalletDto {
  @ApiProperty({ description: 'Amount to add' })
  @IsNumber()
  @Min(100)
  amount: number;

  @ApiProperty({ description: 'Payment method ID' })
  @IsNumber()
  paymentMethodId: number;
}

// =============================================================================
// DRIVER DTOs
// =============================================================================

export class UpdateDriverLocationDto {
  @ApiProperty({ description: 'Latitude' })
  @IsLatitude()
  latitude: number;

  @ApiProperty({ description: 'Longitude' })
  @IsLongitude()
  longitude: number;

  @ApiPropertyOptional({ description: 'Heading in degrees (0-360)' })
  @IsOptional()
  @IsNumber()
  @Min(0)
  @Max(360)
  heading?: number;

  @ApiPropertyOptional({ description: 'Speed in km/h' })
  @IsOptional()
  @IsNumber()
  @Min(0)
  speed?: number;
}

export class DriverGoOnlineDto {
  @ApiProperty({ description: 'Vehicle ID to use' })
  @IsNumber()
  vehicleId: number;

  @ApiProperty({ description: 'Current location', type: LocationDto })
  @ValidateNested()
  @Type(() => LocationDto)
  location: LocationDto;
}

// =============================================================================
// SUPPORT DTOs
// =============================================================================

export class CreateTicketDto {
  @ApiProperty({ description: 'Category', enum: ['booking', 'payment', 'driver', 'app', 'other'] })
  @IsEnum(['booking', 'payment', 'driver', 'app', 'other'])
  category: string;

  @ApiProperty({ description: 'Subject' })
  @IsString()
  @MaxLength(255)
  subject: string;

  @ApiProperty({ description: 'Description' })
  @IsString()
  @MaxLength(2000)
  description: string;

  @ApiPropertyOptional({ description: 'Related booking ID' })
  @IsOptional()
  @IsNumber()
  bookingId?: number;

  @ApiPropertyOptional({ description: 'Attachments URLs' })
  @IsOptional()
  @IsArray()
  @IsString({ each: true })
  attachments?: string[];
}

// =============================================================================
// PROMO DTOs
// =============================================================================

export class ValidatePromoDto {
  @ApiProperty({ description: 'Promo code' })
  @IsString()
  @MaxLength(50)
  code: string;

  @ApiPropertyOptional({ description: 'Vehicle type ID' })
  @IsOptional()
  @IsNumber()
  vehicleTypeId?: number;

  @ApiPropertyOptional({ description: 'Order amount' })
  @IsOptional()
  @IsNumber()
  @Min(0)
  orderAmount?: number;
}

export class PromoValidationResponseDto {
  @ApiProperty({ description: 'Is valid' })
  valid: boolean;

  @ApiPropertyOptional({ description: 'Discount type' })
  discountType?: 'percentage' | 'fixed' | 'free_ride';

  @ApiPropertyOptional({ description: 'Discount value' })
  discountValue?: number;

  @ApiPropertyOptional({ description: 'Max discount amount' })
  maxDiscount?: number;

  @ApiPropertyOptional({ description: 'Error message if invalid' })
  error?: string;
}

// =============================================================================
// RESPONSE WRAPPERS
// =============================================================================

export class ApiResponseDto<T> {
  @ApiProperty({ description: 'Success status' })
  success: boolean;

  @ApiPropertyOptional({ description: 'Response message' })
  message?: string;

  @ApiPropertyOptional({ description: 'Response data' })
  data?: T;
}

export class ErrorResponseDto {
  @ApiProperty({ description: 'Success status', example: false })
  success: boolean;

  @ApiProperty({ description: 'Error message' })
  message: string;

  @ApiPropertyOptional({ description: 'Error code' })
  errorCode?: string;

  @ApiPropertyOptional({ description: 'Validation errors' })
  errors?: Record<string, string[]>;
}
