// =============================================================================
// MonkAPI - Constants
// =============================================================================

// =============================================================================
// BOOKING STATUSES
// =============================================================================

export const BOOKING_STATUS = {
  PENDING: 'pending',
  SEARCHING: 'searching',
  ACCEPTED: 'accepted',
  ARRIVED: 'arrived',
  STARTED: 'started',
  COMPLETED: 'completed',
  CANCELLED: 'cancelled',
} as const;

export type BookingStatus = (typeof BOOKING_STATUS)[keyof typeof BOOKING_STATUS];

export const BOOKING_STATUS_FLOW: Record<BookingStatus, BookingStatus[]> = {
  pending: ['searching', 'cancelled'],
  searching: ['accepted', 'cancelled'],
  accepted: ['arrived', 'cancelled'],
  arrived: ['started', 'cancelled'],
  started: ['completed', 'cancelled'],
  completed: [],
  cancelled: [],
};

export const ACTIVE_BOOKING_STATUSES: BookingStatus[] = [
  BOOKING_STATUS.PENDING,
  BOOKING_STATUS.SEARCHING,
  BOOKING_STATUS.ACCEPTED,
  BOOKING_STATUS.ARRIVED,
  BOOKING_STATUS.STARTED,
];

// =============================================================================
// PAYMENT
// =============================================================================

export const PAYMENT_STATUS = {
  PENDING: 0,
  PAID: 1,
  FAILED: 2,
  REFUNDED: 3,
} as const;

export const PAYMENT_METHOD = {
  CASH: 'cash',
  CARD: 'card',
  WALLET: 'wallet',
  MOBILE_MONEY: 'mobile_money',
  CORPORATE: 'corporate',
} as const;

export type PaymentMethod = (typeof PAYMENT_METHOD)[keyof typeof PAYMENT_METHOD];

// =============================================================================
// DRIVER STATUSES
// =============================================================================

export const DRIVER_STATUS = {
  PENDING: 0,
  APPROVED: 1,
  REJECTED: 2,
  SUSPENDED: 3,
} as const;

export const DRIVER_ONLINE = {
  ONLINE: 1,
  OFFLINE: 2,
} as const;

export const DRIVER_AVAILABILITY = {
  BUSY: 1,
  FREE: 2,
} as const;

// =============================================================================
// USER STATUSES
// =============================================================================

export const USER_STATUS = {
  INACTIVE: 0,
  ACTIVE: 1,
  SUSPENDED: 2,
  DELETED: 3,
} as const;

// =============================================================================
// DEVICE TYPES
// =============================================================================

export const DEVICE_TYPE = {
  ANDROID: 1,
  IOS: 2,
  WEB: 3,
} as const;

// =============================================================================
// DOCUMENT STATUSES
// =============================================================================

export const DOCUMENT_STATUS = {
  PENDING: 'pending',
  APPROVED: 'approved',
  REJECTED: 'rejected',
  EXPIRED: 'expired',
} as const;

export const DOCUMENT_TYPE = {
  LICENSE: 'license',
  ID_CARD: 'id_card',
  PROFILE_PHOTO: 'profile_photo',
  CRIMINAL_RECORD: 'criminal_record',
  INSURANCE: 'insurance',
  REGISTRATION: 'registration',
  INSPECTION: 'inspection',
} as const;

// =============================================================================
// ZONE TYPES
// =============================================================================

export const ZONE_TYPE = {
  SERVICE_AREA: 'service_area',
  SURGE_ZONE: 'surge_zone',
  RESTRICTED: 'restricted',
  AIRPORT: 'airport',
  EVENT: 'event',
  DELIVERY_ONLY: 'delivery_only',
  PICKUP_ONLY: 'pickup_only',
  DROPOFF_ONLY: 'dropoff_only',
} as const;

export type ZoneType = (typeof ZONE_TYPE)[keyof typeof ZONE_TYPE];

// =============================================================================
// PROMO CODE
// =============================================================================

export const PROMO_TYPE = {
  PERCENTAGE: 'percentage',
  FIXED: 'fixed',
  FREE_RIDE: 'free_ride',
} as const;

// =============================================================================
// SUPPORT TICKET
// =============================================================================

export const TICKET_STATUS = {
  OPEN: 'open',
  IN_PROGRESS: 'in_progress',
  WAITING_CUSTOMER: 'waiting_customer',
  RESOLVED: 'resolved',
  CLOSED: 'closed',
} as const;

export const TICKET_PRIORITY = {
  LOW: 'low',
  MEDIUM: 'medium',
  HIGH: 'high',
  URGENT: 'urgent',
} as const;

// =============================================================================
// DISPUTE
// =============================================================================

export const DISPUTE_TYPE = {
  FARE: 'fare',
  SERVICE: 'service',
  SAFETY: 'safety',
  PAYMENT: 'payment',
  DRIVER_BEHAVIOR: 'driver_behavior',
  USER_BEHAVIOR: 'user_behavior',
  DAMAGE: 'damage',
  LOST_ITEM: 'lost_item',
  OTHER: 'other',
} as const;

export const DISPUTE_STATUS = {
  OPEN: 'open',
  INVESTIGATING: 'investigating',
  PENDING_RESPONSE: 'pending_response',
  RESOLVED: 'resolved',
  CLOSED: 'closed',
  ESCALATED: 'escalated',
} as const;

export const DISPUTE_RESOLUTION = {
  REFUND_FULL: 'refund_full',
  REFUND_PARTIAL: 'refund_partial',
  CREDIT: 'credit',
  NO_ACTION: 'no_action',
  WARNING_DRIVER: 'warning_driver',
  WARNING_USER: 'warning_user',
  BAN_DRIVER: 'ban_driver',
  BAN_USER: 'ban_user',
} as const;

// =============================================================================
// NOTIFICATION
// =============================================================================

export const NOTIFICATION_TYPE = {
  PUSH: 'push',
  SMS: 'sms',
  EMAIL: 'email',
  IN_APP: 'in_app',
} as const;

export const NOTIFICATION_STATUS = {
  PENDING: 'pending',
  SENT: 'sent',
  DELIVERED: 'delivered',
  FAILED: 'failed',
  READ: 'read',
} as const;

// =============================================================================
// WALLET TRANSACTION
// =============================================================================

export const TRANSACTION_TYPE = {
  CREDIT: 'credit',
  DEBIT: 'debit',
} as const;

export const TRANSACTION_REFERENCE = {
  BOOKING: 'booking',
  TOPUP: 'topup',
  REFUND: 'refund',
  PROMO: 'promo',
  REFERRAL: 'referral',
  PAYOUT: 'payout',
  BONUS: 'bonus',
  PENALTY: 'penalty',
} as const;

// =============================================================================
// PARTNER TYPES
// =============================================================================

export const PARTNER_TYPE = {
  HOTEL: 'hotel',
  RESTAURANT: 'restaurant',
  AIRPORT: 'airport',
  MALL: 'mall',
  HOSPITAL: 'hospital',
  CORPORATE: 'corporate',
  EVENT_VENUE: 'event_venue',
  OTHER: 'other',
} as const;

// =============================================================================
// FLEET
// =============================================================================

export const FLEET_DRIVER_ROLE = {
  DRIVER: 'driver',
  SUPERVISOR: 'supervisor',
  MANAGER: 'manager',
} as const;

export const FLEET_VEHICLE_STATUS = {
  AVAILABLE: 'available',
  ASSIGNED: 'assigned',
  MAINTENANCE: 'maintenance',
} as const;

// =============================================================================
// DELIVERY
// =============================================================================

export const DELIVERY_STATUS = {
  PENDING: 'pending',
  PICKED_UP: 'picked_up',
  IN_TRANSIT: 'in_transit',
  DELIVERED: 'delivered',
  FAILED: 'failed',
  RETURNED: 'returned',
} as const;

export const PACKAGE_SIZE = {
  SMALL: 'small',
  MEDIUM: 'medium',
  LARGE: 'large',
  EXTRA_LARGE: 'extra_large',
} as const;

// =============================================================================
// REPORT
// =============================================================================

export const REPORT_TYPE = {
  BOOKINGS: 'bookings',
  EARNINGS: 'earnings',
  DRIVERS: 'drivers',
  USERS: 'users',
  PAYMENTS: 'payments',
  DELIVERIES: 'deliveries',
  RATINGS: 'ratings',
  SUPPORT: 'support',
  FINANCIAL: 'financial',
  OPERATIONAL: 'operational',
} as const;

export const REPORT_FORMAT = {
  JSON: 'json',
  CSV: 'csv',
  EXCEL: 'excel',
  PDF: 'pdf',
} as const;

export const REPORT_PERIOD = {
  TODAY: 'today',
  YESTERDAY: 'yesterday',
  WEEK: 'week',
  MONTH: 'month',
  QUARTER: 'quarter',
  YEAR: 'year',
  CUSTOM: 'custom',
} as const;

// =============================================================================
// ADMIN ROLES
// =============================================================================

export const ADMIN_ROLE = {
  SUPER_ADMIN: 'super_admin',
  ADMIN: 'admin',
  SUPPORT: 'support',
  FINANCE: 'finance',
  OPERATIONS: 'operations',
} as const;

// =============================================================================
// RATE LIMITS
// =============================================================================

export const RATE_LIMITS = {
  DEFAULT_TTL: 60000, // 1 minute
  DEFAULT_LIMIT: 100,
  AUTH_TTL: 60000,
  AUTH_LIMIT: 10,
  OTP_TTL: 60000,
  OTP_LIMIT: 3,
  API_KEY_LIMIT: 1000,
} as const;

// =============================================================================
// TIME CONSTANTS
// =============================================================================

export const TIME = {
  SECOND: 1000,
  MINUTE: 60 * 1000,
  HOUR: 60 * 60 * 1000,
  DAY: 24 * 60 * 60 * 1000,
  WEEK: 7 * 24 * 60 * 60 * 1000,
} as const;

// =============================================================================
// PAGINATION DEFAULTS
// =============================================================================

export const PAGINATION = {
  DEFAULT_PAGE: 1,
  DEFAULT_LIMIT: 20,
  MAX_LIMIT: 100,
} as const;

// =============================================================================
// DRIVER SEARCH
// =============================================================================

export const DRIVER_SEARCH = {
  DEFAULT_RADIUS: 5, // km
  MAX_RADIUS: 20, // km
  DEFAULT_TIMEOUT: 30, // seconds
  MAX_ATTEMPTS: 5,
  BROADCAST_INTERVAL: 5000, // ms
} as const;

// =============================================================================
// FILE UPLOAD
// =============================================================================

export const FILE_UPLOAD = {
  MAX_SIZE: 10 * 1024 * 1024, // 10MB
  ALLOWED_IMAGE_TYPES: ['image/jpeg', 'image/png', 'image/webp'],
  ALLOWED_DOCUMENT_TYPES: ['application/pdf', 'image/jpeg', 'image/png'],
  IMAGE_QUALITY: 80,
  THUMBNAIL_SIZE: 200,
} as const;

// =============================================================================
// CACHE KEYS
// =============================================================================

export const CACHE_KEYS = {
  USER: (id: number) => `user:${id}`,
  DRIVER: (id: number) => `driver:${id}`,
  BOOKING: (id: number) => `booking:${id}`,
  VEHICLE_TYPES: (merchantId: number) => `vehicle_types:${merchantId}`,
  SETTINGS: (merchantId: number, key?: string) =>
    key ? `settings:${merchantId}:${key}` : `settings:${merchantId}`,
  ZONES: (merchantId: number) => `zones:${merchantId}`,
  ONLINE_DRIVERS: (merchantId: number) => `online_drivers:${merchantId}`,
  DRIVER_LOCATION: (driverId: number) => `driver_location:${driverId}`,
  ACTIVE_BOOKING: (userId: number) => `active_booking:user:${userId}`,
  RATE_LIMIT: (key: string) => `rate_limit:${key}`,
} as const;

// =============================================================================
// QUEUE NAMES
// =============================================================================

export const QUEUES = {
  BOOKING_ASSIGNMENT: 'booking-assignment',
  NOTIFICATIONS: 'notifications',
  EMAILS: 'emails',
  SMS: 'sms',
  WEBHOOKS: 'webhooks',
  REPORTS: 'reports',
  PAYMENTS: 'payments',
} as const;

// =============================================================================
// WEBSOCKET EVENTS
// =============================================================================

export const WS_EVENTS = {
  // Booking events
  BOOKING_CREATED: 'booking:created',
  BOOKING_ASSIGNED: 'booking:assigned',
  BOOKING_ACCEPTED: 'booking:accepted',
  BOOKING_ARRIVED: 'booking:arrived',
  BOOKING_STARTED: 'booking:started',
  BOOKING_COMPLETED: 'booking:completed',
  BOOKING_CANCELLED: 'booking:cancelled',

  // Driver events
  DRIVER_LOCATION: 'driver:location',
  DRIVER_ONLINE: 'driver:online',
  DRIVER_OFFLINE: 'driver:offline',
  NEW_BOOKING_REQUEST: 'driver:new_booking',

  // Chat events
  CHAT_MESSAGE: 'chat:message',
  CHAT_TYPING: 'chat:typing',

  // Notification events
  NOTIFICATION: 'notification',
} as const;

// =============================================================================
// ERROR CODES
// =============================================================================

export const ERROR_CODES = {
  // Auth errors (1xxx)
  AUTH_INVALID_CREDENTIALS: 'E1001',
  AUTH_TOKEN_EXPIRED: 'E1002',
  AUTH_TOKEN_INVALID: 'E1003',
  AUTH_OTP_INVALID: 'E1004',
  AUTH_OTP_EXPIRED: 'E1005',
  AUTH_ACCOUNT_SUSPENDED: 'E1006',
  AUTH_ACCOUNT_NOT_VERIFIED: 'E1007',

  // User errors (2xxx)
  USER_NOT_FOUND: 'E2001',
  USER_EMAIL_EXISTS: 'E2002',
  USER_PHONE_EXISTS: 'E2003',

  // Driver errors (3xxx)
  DRIVER_NOT_FOUND: 'E3001',
  DRIVER_NOT_APPROVED: 'E3002',
  DRIVER_NOT_ONLINE: 'E3003',
  DRIVER_BUSY: 'E3004',

  // Booking errors (4xxx)
  BOOKING_NOT_FOUND: 'E4001',
  BOOKING_INVALID_STATUS: 'E4002',
  BOOKING_ALREADY_ASSIGNED: 'E4003',
  BOOKING_ZONE_RESTRICTED: 'E4004',
  BOOKING_NO_DRIVERS: 'E4005',

  // Payment errors (5xxx)
  PAYMENT_FAILED: 'E5001',
  PAYMENT_INSUFFICIENT_BALANCE: 'E5002',
  PAYMENT_METHOD_INVALID: 'E5003',

  // Promo errors (6xxx)
  PROMO_INVALID: 'E6001',
  PROMO_EXPIRED: 'E6002',
  PROMO_USAGE_EXCEEDED: 'E6003',
  PROMO_MIN_ORDER: 'E6004',

  // General errors (9xxx)
  VALIDATION_ERROR: 'E9001',
  NOT_FOUND: 'E9002',
  FORBIDDEN: 'E9003',
  RATE_LIMITED: 'E9004',
  SERVER_ERROR: 'E9999',
} as const;
