# Modules MonkAPI

MonkAPI est compose de 47 modules NestJS organises par domaine metier.

## Vue d'ensemble

```
src/modules/
├── auth/           # Authentification
├── user/           # Utilisateurs
├── driver/         # Chauffeurs
├── booking/        # Reservations
├── payment/        # Paiements
├── delivery/       # Livraisons
├── notification/   # Notifications
├── chat/           # Messagerie
├── rating/         # Notes & avis
├── promo/          # Codes promo
├── wallet/         # Portefeuille
├── vehicle/        # Vehicules
├── zone/           # Zones geo
├── surge/          # Tarification dynamique
├── handyman/       # Services a domicile
├── corporate/      # Comptes entreprise
├── franchise/      # Franchises
├── fleet/          # Flottes
├── referral/       # Parrainage
├── support/        # Support client
├── dispute/        # Litiges
├── report/         # Rapports
├── admin/          # Administration
├── health/         # Health checks
├── webhook/        # Webhooks
└── ...
```

---

## Auth Module

Gestion de l'authentification et autorisation.

### Fichiers

```
src/modules/auth/
├── auth.module.ts
├── auth.controller.ts
├── auth.service.ts
├── dto/
│   ├── user-signup.dto.ts
│   ├── user-login.dto.ts
│   ├── driver-signup.dto.ts
│   ├── driver-login.dto.ts
│   ├── otp.dto.ts
│   ├── forgot-password.dto.ts
│   ├── reset-password.dto.ts
│   └── refresh-token.dto.ts
└── strategies/
    ├── jwt.strategy.ts
    ├── jwt-refresh.strategy.ts
    ├── google.strategy.ts
    ├── facebook.strategy.ts
    └── apple.strategy.ts
```

### Endpoints

| Methode | Endpoint | Description |
|---------|----------|-------------|
| POST | `/auth/signup` | Inscription user |
| POST | `/auth/signup/driver` | Inscription driver |
| POST | `/auth/login` | Connexion email/password |
| POST | `/auth/login/driver` | Connexion driver |
| POST | `/auth/login-otp` | Demande OTP |
| POST | `/auth/verify-otp` | Verifier OTP |
| POST | `/auth/social-sign` | Login social |
| POST | `/auth/forgot-password` | Mot de passe oublie |
| POST | `/auth/reset-password` | Reset password |
| POST | `/auth/refresh` | Rafraichir token |
| POST | `/auth/logout` | Deconnexion |

### Services

```typescript
@Injectable()
export class AuthService {
  async signup(dto: UserSignupDto): Promise<AuthResponse>
  async login(dto: UserLoginDto): Promise<AuthResponse>
  async loginDriver(dto: DriverLoginDto): Promise<AuthResponse>
  async sendOtp(phone: string): Promise<void>
  async verifyOtp(phone: string, otp: string): Promise<AuthResponse>
  async socialLogin(provider: string, token: string): Promise<AuthResponse>
  async forgotPassword(email: string): Promise<void>
  async resetPassword(token: string, password: string): Promise<void>
  async refreshToken(refreshToken: string): Promise<TokenResponse>
  async logout(userId: number): Promise<void>
}
```

### Guards

- `JwtAuthGuard` - Verifie le token JWT
- `JwtRefreshGuard` - Verifie le refresh token
- `DriverGuard` - Verifie que c'est un chauffeur

---

## User Module

Gestion des profils utilisateurs.

### Fichiers

```
src/modules/user/
├── user.module.ts
├── user.controller.ts
├── user.service.ts
└── dto/
    ├── update-profile.dto.ts
    └── update-device.dto.ts
```

### Endpoints

| Methode | Endpoint | Description |
|---------|----------|-------------|
| GET | `/user/profile` | Profil |
| PUT | `/user/profile` | Maj profil |
| POST | `/user/profile/image` | Upload avatar |
| PUT | `/user/device` | Maj device token |
| DELETE | `/user/account` | Supprimer compte |

### Services

```typescript
@Injectable()
export class UserService {
  async getProfile(userId: number): Promise<User>
  async updateProfile(userId: number, dto: UpdateProfileDto): Promise<User>
  async uploadImage(userId: number, file: File): Promise<string>
  async updateDevice(userId: number, dto: UpdateDeviceDto): Promise<void>
  async deleteAccount(userId: number): Promise<void>
}
```

---

## Driver Module

Gestion des chauffeurs.

### Fichiers

```
src/modules/driver/
├── driver.module.ts
├── driver.controller.ts
├── driver.service.ts
├── driver.gateway.ts          # WebSocket
├── dto/
│   ├── update-profile.dto.ts
│   ├── update-location.dto.ts
│   ├── update-status.dto.ts
│   └── upload-document.dto.ts
└── jobs/
    └── location-history.job.ts
```

### Endpoints

| Methode | Endpoint | Description |
|---------|----------|-------------|
| GET | `/driver/profile` | Profil |
| PUT | `/driver/profile` | Maj profil |
| PUT | `/driver/location` | Maj position |
| PUT | `/driver/status` | Online/Offline |
| GET | `/driver/earnings` | Gains |
| GET | `/driver/documents` | Documents |
| POST | `/driver/documents` | Upload document |
| PUT | `/driver/booking/:id/accept` | Accepter course |
| PUT | `/driver/booking/:id/arrived` | Arrive |
| PUT | `/driver/booking/:id/start` | Demarrer |
| PUT | `/driver/booking/:id/complete` | Terminer |
| PUT | `/driver/booking/:id/cancel` | Annuler |

### WebSocket Events

```typescript
@WebSocketGateway({ namespace: '/driver' })
export class DriverGateway {
  // Client -> Server
  @SubscribeMessage('location:update')
  handleLocationUpdate(client: Socket, payload: LocationDto)

  @SubscribeMessage('status:update')
  handleStatusUpdate(client: Socket, payload: { is_online: boolean })

  // Server -> Client
  emitNewBookingRequest(driverId: number, booking: Booking)
  emitBookingCancelled(driverId: number, bookingId: number)
}
```

---

## Booking Module

Gestion des reservations.

### Fichiers

```
src/modules/booking/
├── booking.module.ts
├── booking.controller.ts
├── booking.service.ts
├── booking.gateway.ts
├── dto/
│   ├── estimate.dto.ts
│   ├── create-booking.dto.ts
│   ├── cancel-booking.dto.ts
│   └── rate-booking.dto.ts
├── entities/
│   └── booking-status.enum.ts
└── jobs/
    ├── assign-driver.job.ts
    ├── auto-cancel.job.ts
    └── notify-nearby-drivers.job.ts
```

### Endpoints

| Methode | Endpoint | Description |
|---------|----------|-------------|
| POST | `/booking/estimate` | Estimation |
| POST | `/booking/create` | Creer |
| GET | `/booking/:id` | Details |
| PUT | `/booking/:id/cancel` | Annuler |
| POST | `/booking/:id/rate` | Noter |
| GET | `/booking/history` | Historique |
| GET | `/booking/active` | Courses actives |

### State Machine

```
pending -> searching -> accepted -> arrived -> started -> completed
    |          |           |          |          |
    +----------+-----------+----------+----------+-> cancelled
```

### Services

```typescript
@Injectable()
export class BookingService {
  async estimate(dto: EstimateDto): Promise<EstimateResponse>
  async create(userId: number, dto: CreateBookingDto): Promise<Booking>
  async getById(id: number): Promise<Booking>
  async cancel(id: number, dto: CancelBookingDto): Promise<Booking>
  async rate(id: number, dto: RateBookingDto): Promise<void>
  async getHistory(userId: number, pagination: PaginationDto): Promise<Booking[]>
  async getActive(userId: number): Promise<Booking | null>

  // Driver actions
  async accept(id: number, driverId: number): Promise<Booking>
  async markArrived(id: number, driverId: number): Promise<Booking>
  async start(id: number, driverId: number): Promise<Booking>
  async complete(id: number, dto: CompleteBookingDto): Promise<Booking>
}
```

---

## Payment Module

Gestion des paiements.

### Fichiers

```
src/modules/payment/
├── payment.module.ts
├── payment.controller.ts
├── payment.service.ts
├── dto/
│   ├── add-payment-method.dto.ts
│   └── charge.dto.ts
└── providers/
    ├── stripe.provider.ts
    ├── paypal.provider.ts
    ├── mobile-money.provider.ts
    └── ...
```

### Endpoints

| Methode | Endpoint | Description |
|---------|----------|-------------|
| GET | `/payment/methods` | Lister moyens |
| POST | `/payment/methods` | Ajouter moyen |
| DELETE | `/payment/methods/:id` | Supprimer |
| PUT | `/payment/methods/:id/default` | Definir defaut |
| POST | `/payment/charge` | Payer |
| POST | `/payment/webhook/stripe` | Webhook Stripe |

### Providers supportes

- Stripe
- PayPal
- Mobile Money (MTN, Orange, Moov)
- Flutterwave
- Paystack
- Et 45+ autres...

---

## Delivery Module

Gestion des livraisons.

### Fichiers

```
src/modules/delivery/
├── delivery.module.ts
├── delivery.controller.ts
├── delivery.service.ts
└── dto/
    ├── estimate-delivery.dto.ts
    ├── create-delivery.dto.ts
    └── update-delivery.dto.ts
```

### Endpoints

| Methode | Endpoint | Description |
|---------|----------|-------------|
| POST | `/delivery/estimate` | Estimer |
| POST | `/delivery/create` | Creer |
| GET | `/delivery/:id` | Details |
| GET | `/delivery/:id/track` | Suivre |
| PUT | `/delivery/:id/cancel` | Annuler |
| GET | `/delivery/history` | Historique |

---

## Notification Module

Gestion des notifications push.

### Fichiers

```
src/modules/notification/
├── notification.module.ts
├── notification.controller.ts
├── notification.service.ts
├── dto/
│   └── send-notification.dto.ts
└── providers/
    ├── fcm.provider.ts
    ├── onesignal.provider.ts
    └── apns.provider.ts
```

### Services

```typescript
@Injectable()
export class NotificationService {
  async send(userId: number, notification: NotificationDto): Promise<void>
  async sendToDriver(driverId: number, notification: NotificationDto): Promise<void>
  async sendBulk(userIds: number[], notification: NotificationDto): Promise<void>
  async getHistory(userId: number): Promise<Notification[]>
  async markAsRead(id: number): Promise<void>
}
```

---

## Wallet Module

Gestion des portefeuilles.

### Fichiers

```
src/modules/wallet/
├── wallet.module.ts
├── wallet.controller.ts
├── wallet.service.ts
└── dto/
    ├── topup.dto.ts
    └── withdraw.dto.ts
```

### Endpoints

| Methode | Endpoint | Description |
|---------|----------|-------------|
| GET | `/wallet` | Solde |
| GET | `/wallet/transactions` | Historique |
| POST | `/wallet/topup` | Recharger |
| POST | `/wallet/withdraw` | Retirer (driver) |

---

## Promo Module

Gestion des codes promotionnels.

### Fichiers

```
src/modules/promo/
├── promo.module.ts
├── promo.controller.ts
├── promo.service.ts
└── dto/
    └── validate-promo.dto.ts
```

### Endpoints

| Methode | Endpoint | Description |
|---------|----------|-------------|
| POST | `/promo/validate` | Valider code |
| GET | `/promo/available` | Codes disponibles |

---

## Zone Module

Gestion des zones geographiques.

### Fichiers

```
src/modules/zone/
├── zone.module.ts
├── zone.controller.ts
├── zone.service.ts
└── dto/
    └── check-zone.dto.ts
```

### Services

```typescript
@Injectable()
export class ZoneService {
  async getAll(merchantId: number): Promise<Zone[]>
  async checkPoint(lat: number, lng: number): Promise<ZoneCheckResult>
  async isInServiceArea(lat: number, lng: number): Promise<boolean>
  async getSurgeZones(): Promise<Zone[]>
}
```

---

## Surge Module

Tarification dynamique.

### Fichiers

```
src/modules/surge/
├── surge.module.ts
├── surge.service.ts
└── jobs/
    └── calculate-surge.job.ts
```

### Services

```typescript
@Injectable()
export class SurgeService {
  async getMultiplier(lat: number, lng: number): Promise<number>
  async calculateSurge(zoneId: number): Promise<number>
}
```

---

## Health Module

Health checks pour Kubernetes.

### Fichiers

```
src/modules/health/
├── health.module.ts
├── health.controller.ts
└── health.service.ts
```

### Endpoints

| Methode | Endpoint | Description |
|---------|----------|-------------|
| GET | `/health` | Check basique |
| GET | `/health/live` | Liveness |
| GET | `/health/ready` | Readiness |
| GET | `/health/metrics` | Prometheus |

---

## Chat Module

Messagerie in-app.

### Fichiers

```
src/modules/chat/
├── chat.module.ts
├── chat.controller.ts
├── chat.service.ts
└── chat.gateway.ts
```

### WebSocket Events

```typescript
@WebSocketGateway({ namespace: '/chat' })
export class ChatGateway {
  @SubscribeMessage('message:send')
  handleMessage(client: Socket, payload: SendMessageDto)

  @SubscribeMessage('typing:start')
  handleTyping(client: Socket, payload: { conversationId: string })
}
```

---

## Autres Modules

### Rating Module
Notes et avis.

### Referral Module
Programme de parrainage.

### Support Module
Tickets de support.

### Dispute Module
Gestion des litiges.

### Report Module
Rapports et analytics.

### Admin Module
Administration (CRUD).

### Webhook Module
Webhooks sortants.

### Corporate Module
Comptes entreprise.

### Franchise Module
Gestion des franchises.

### Fleet Module
Gestion des flottes.

### Handyman Module
Services a domicile.

### Vehicle Module
Types de vehicules.

---

## Creation d'un nouveau module

```bash
# Generer un module
nest g module modules/my-module
nest g controller modules/my-module
nest g service modules/my-module

# Structure recommandee
src/modules/my-module/
├── my-module.module.ts
├── my-module.controller.ts
├── my-module.service.ts
├── dto/
│   ├── create-my-module.dto.ts
│   └── update-my-module.dto.ts
├── entities/
│   └── my-module.entity.ts
└── guards/
    └── my-module.guard.ts
```

### Enregistrer le module

```typescript
// app.module.ts
@Module({
  imports: [
    // ...
    MyModuleModule,
  ],
})
export class AppModule {}
```
