# Guide de Deploiement MonkAPI

## Options de deploiement

| Option | Complexite | Scalabilite | Recommande pour |
|--------|------------|-------------|-----------------|
| Docker Compose | Faible | Faible | Dev, staging |
| Kubernetes | Haute | Haute | Production |
| Cloud Run | Moyenne | Moyenne | Production simple |
| AWS ECS | Moyenne | Haute | Production AWS |

---

## Docker Compose (Dev/Staging)

### Prerequis

- Docker >= 20.10
- Docker Compose >= 2.0

### Demarrage rapide

```bash
# Cloner le repo
git clone https://github.com/monkapi/api.git
cd monkapi-node

# Copier la config
cp .env.example .env

# Demarrer tous les services
docker-compose up -d

# Voir les logs
docker-compose logs -f api

# Arreter
docker-compose down
```

### Services inclus

| Service | Port | Description |
|---------|------|-------------|
| api | 3000 | MonkAPI |
| mysql | 3306 | Base de donnees |
| redis | 6379 | Cache/Queues |
| redis-commander | 8081 | UI Redis |

### docker-compose.yml

```yaml
version: '3.8'

services:
  api:
    build: .
    ports:
      - "3000:3000"
    environment:
      - NODE_ENV=development
      - DATABASE_URL=mysql://monkapi:password@mysql:3306/monkapi
      - REDIS_HOST=redis
    depends_on:
      - mysql
      - redis

  mysql:
    image: mysql:8
    environment:
      MYSQL_ROOT_PASSWORD: rootpassword
      MYSQL_DATABASE: monkapi
      MYSQL_USER: monkapi
      MYSQL_PASSWORD: password
    volumes:
      - mysql_data:/var/lib/mysql
    ports:
      - "3306:3306"

  redis:
    image: redis:7-alpine
    ports:
      - "6379:6379"
    volumes:
      - redis_data:/data

volumes:
  mysql_data:
  redis_data:
```

---

## Kubernetes (Production)

### Prerequis

- Cluster Kubernetes >= 1.25
- kubectl configure
- Helm >= 3.0 (optionnel)
- NGINX Ingress Controller
- cert-manager (pour TLS)

### Architecture

```
┌─────────────────────────────────────────────────────────────┐
│                        Internet                              │
└─────────────────────────────────────────────────────────────┘
                              │
                              ▼
┌─────────────────────────────────────────────────────────────┐
│                    NGINX Ingress                             │
│                  (Load Balancer)                             │
└─────────────────────────────────────────────────────────────┘
                              │
              ┌───────────────┼───────────────┐
              ▼               ▼               ▼
        ┌──────────┐   ┌──────────┐   ┌──────────┐
        │  Pod 1   │   │  Pod 2   │   │  Pod 3   │
        │ MonkAPI  │   │ MonkAPI  │   │ MonkAPI  │
        └──────────┘   └──────────┘   └──────────┘
              │               │               │
              └───────────────┼───────────────┘
                              │
        ┌─────────────────────┼─────────────────────┐
        │                     │                     │
        ▼                     ▼                     ▼
┌──────────────┐     ┌──────────────┐     ┌──────────────┐
│    MySQL     │     │    Redis     │     │      S3      │
│  (Managed)   │     │  (Managed)   │     │  (Storage)   │
└──────────────┘     └──────────────┘     └──────────────┘
```

### Deploiement etape par etape

#### 1. Creer le namespace

```bash
kubectl apply -f k8s/namespace.yaml
```

```yaml
# k8s/namespace.yaml
apiVersion: v1
kind: Namespace
metadata:
  name: monkapi
```

#### 2. Configurer les secrets

```bash
# Creer les secrets (ne pas commiter!)
kubectl create secret generic monkapi-secret \
  --namespace=monkapi \
  --from-literal=DATABASE_URL='mysql://user:pass@host:3306/db' \
  --from-literal=JWT_SECRET='your-jwt-secret' \
  --from-literal=STRIPE_SECRET_KEY='sk_live_xxx'
```

Ou via fichier:

```bash
kubectl apply -f k8s/secret.yaml
```

#### 3. Appliquer la ConfigMap

```bash
kubectl apply -f k8s/configmap.yaml
```

#### 4. Deployer Redis (si non manage)

```bash
kubectl apply -f k8s/redis.yaml
```

#### 5. Deployer l'application

```bash
kubectl apply -f k8s/deployment.yaml
kubectl apply -f k8s/service.yaml
kubectl apply -f k8s/hpa.yaml
```

#### 6. Verifier le deploiement

```bash
# Status des pods
kubectl get pods -n monkapi

# Logs
kubectl logs -f deployment/monkapi -n monkapi

# Decrire le deploiement
kubectl describe deployment monkapi -n monkapi
```

### Configuration TLS

#### Avec cert-manager (Let's Encrypt)

```bash
# Installer cert-manager
kubectl apply -f https://github.com/cert-manager/cert-manager/releases/download/v1.13.0/cert-manager.yaml

# Creer le ClusterIssuer
kubectl apply -f - <<EOF
apiVersion: cert-manager.io/v1
kind: ClusterIssuer
metadata:
  name: letsencrypt-prod
spec:
  acme:
    server: https://acme-v02.api.letsencrypt.org/directory
    email: admin@monkapi.com
    privateKeySecretRef:
      name: letsencrypt-prod
    solvers:
    - http01:
        ingress:
          class: nginx
EOF
```

### Scaling

#### Manuel

```bash
kubectl scale deployment monkapi --replicas=5 -n monkapi
```

#### Automatique (HPA)

Deja configure via `k8s/hpa.yaml`:
- Min: 3 replicas
- Max: 20 replicas
- Scale up a 70% CPU

### Rolling Update

```bash
# Mettre a jour l'image
kubectl set image deployment/monkapi \
  monkapi=monkapi/api:v1.2.0 \
  -n monkapi

# Suivre le rollout
kubectl rollout status deployment/monkapi -n monkapi

# Rollback si necessaire
kubectl rollout undo deployment/monkapi -n monkapi
```

---

## CI/CD avec GitHub Actions

### Workflow CI

Le workflow `.github/workflows/ci.yml` execute:

1. Lint & Type Check
2. Tests unitaires
3. Tests E2E
4. Build
5. Docker build & push
6. Security scan

### Workflow CD

Le workflow `.github/workflows/cd.yml` execute:

1. Build image Docker
2. Push vers GHCR
3. Deploy staging
4. Tests smoke
5. Deploy production (sur tag)
6. Create GitHub release

### Secrets GitHub requis

| Secret | Description |
|--------|-------------|
| `DOCKER_USERNAME` | Username Docker Hub |
| `DOCKER_PASSWORD` | Password Docker Hub |
| `KUBE_CONFIG_STAGING` | Kubeconfig staging |
| `KUBE_CONFIG_PRODUCTION` | Kubeconfig production |
| `DATABASE_URL` | URL base de donnees |
| `SLACK_WEBHOOK` | Webhook Slack |

### Creer un release

```bash
# Creer un tag
git tag v1.2.0
git push origin v1.2.0
```

Cela declenche automatiquement:
1. Build et push de l'image
2. Deploiement staging
3. Deploiement production
4. Creation du release GitHub

---

## Migration de base de donnees

### En developpement

```bash
# Creer une migration
pnpm prisma migrate dev --name add_new_feature

# Appliquer les migrations
pnpm prisma migrate dev
```

### En production

```bash
# Appliquer les migrations (sans interactif)
pnpm prisma migrate deploy
```

### Dans Kubernetes

```bash
# Executer un job de migration
kubectl apply -f - <<EOF
apiVersion: batch/v1
kind: Job
metadata:
  name: prisma-migrate
  namespace: monkapi
spec:
  template:
    spec:
      containers:
      - name: migrate
        image: monkapi/api:latest
        command: ["pnpm", "prisma", "migrate", "deploy"]
        envFrom:
        - secretRef:
            name: monkapi-secret
      restartPolicy: Never
  backoffLimit: 3
EOF
```

---

## Migration des donnees Laravel

### Prerequis

1. Acces a la base Laravel source
2. Base cible configuree
3. Schema Prisma genere

### Execution

```bash
# Configurer les URLs
export LARAVEL_DATABASE_URL="mysql://user:pass@laravel-host:3306/laravel_db"
export DATABASE_URL="mysql://user:pass@new-host:3306/monkapi"

# Lancer la migration
npx ts-node scripts/migrate-data.ts
```

### Tables migrees

1. merchants
2. users
3. drivers
4. vehicle_types
5. bookings
6. zones
7. promo_codes

---

## Monitoring

### Prometheus metrics

Endpoint: `GET /api/health/metrics`

```bash
# Scrape config Prometheus
- job_name: 'monkapi'
  kubernetes_sd_configs:
  - role: pod
  relabel_configs:
  - source_labels: [__meta_kubernetes_pod_annotation_prometheus_io_scrape]
    action: keep
    regex: true
```

### Health checks

| Endpoint | Usage |
|----------|-------|
| `/api/health` | Check general |
| `/api/health/live` | Kubernetes liveness |
| `/api/health/ready` | Kubernetes readiness |

### Alertes recommandees

| Metrique | Seuil | Severite |
|----------|-------|----------|
| CPU > 80% | 5 min | Warning |
| Memory > 85% | 5 min | Warning |
| Error rate > 1% | 2 min | Critical |
| Latency p99 > 2s | 5 min | Warning |
| Pod restarts > 3 | 10 min | Critical |

---

## Troubleshooting

### Pod en CrashLoopBackOff

```bash
# Voir les logs
kubectl logs <pod-name> -n monkapi --previous

# Causes communes:
# - DATABASE_URL incorrect
# - Secret manquant
# - Port deja utilise
```

### Connexion DB refusee

```bash
# Verifier le secret
kubectl get secret monkapi-secret -n monkapi -o yaml

# Tester la connexion
kubectl run mysql-client --rm -it --image=mysql:8 -- \
  mysql -h <host> -u <user> -p<password> <database>
```

### Ingress ne fonctionne pas

```bash
# Verifier l'ingress
kubectl describe ingress monkapi-ingress -n monkapi

# Verifier le service
kubectl get svc -n monkapi

# Verifier les endpoints
kubectl get endpoints -n monkapi
```

### Performance degradee

1. Verifier les metriques HPA:
   ```bash
   kubectl get hpa -n monkapi
   ```

2. Augmenter les ressources:
   ```yaml
   resources:
     requests:
       memory: "512Mi"
       cpu: "500m"
     limits:
       memory: "1Gi"
       cpu: "1000m"
   ```

3. Verifier les slow queries:
   ```bash
   kubectl logs deployment/monkapi -n monkapi | grep "slow query"
   ```
