# Schema Base de Donnees MonkAPI

## Vue d'ensemble

MonkAPI utilise Prisma ORM avec MySQL. Le schema comprend 60 models organises par domaine.

## Diagramme ER simplifie

```
┌─────────────┐     ┌─────────────┐     ┌─────────────┐
│  Merchant   │────<│    User     │────<│   Booking   │
└─────────────┘     └─────────────┘     └──────┬──────┘
       │                                        │
       │            ┌─────────────┐            │
       └───────────>│   Driver    │>───────────┘
                    └──────┬──────┘
                           │
                    ┌──────┴──────┐
                    │   Vehicle   │
                    └─────────────┘
```

---

## Models par domaine

### Core

| Model | Description | Relations principales |
|-------|-------------|----------------------|
| `Merchant` | Tenant principal | Users, Drivers, Bookings |
| `User` | Utilisateur client | Bookings, Wallet, Ratings |
| `Driver` | Chauffeur | Bookings, Vehicle, Documents |
| `Booking` | Reservation | User, Driver, Payment |

### Vehicules

| Model | Description |
|-------|-------------|
| `VehicleType` | Types de vehicules (Standard, Comfort...) |
| `Vehicle` | Vehicule d'un chauffeur |
| `DriverVehicle` | Association driver-vehicle |

### Business

| Model | Description |
|-------|-------------|
| `Fleet` | Flotte de vehicules |
| `Corporate` | Compte entreprise |
| `Franchise` | Franchise |
| `Partner` | Partenaire commercial |

### Finance

| Model | Description |
|-------|-------------|
| `Wallet` | Portefeuille utilisateur/driver |
| `WalletTransaction` | Transaction portefeuille |
| `Payment` | Paiement d'une course |
| `PaymentMethod` | Moyen de paiement |
| `PromoCode` | Code promotionnel |
| `PromoUsage` | Utilisation d'un promo |

### Services

| Model | Description |
|-------|-------------|
| `Delivery` | Livraison |
| `DeliveryStop` | Etape de livraison |
| `HandymanService` | Service a domicile |
| `HandymanBooking` | Reservation handyman |

### Communications

| Model | Description |
|-------|-------------|
| `Notification` | Notification push |
| `Chat` | Conversation |
| `ChatMessage` | Message |
| `SupportTicket` | Ticket support |

### Geographie

| Model | Description |
|-------|-------------|
| `Zone` | Zone geographique |
| `Country` | Pays |
| `Currency` | Devise |

### Systeme

| Model | Description |
|-------|-------------|
| `Setting` | Configuration |
| `AdminUser` | Administrateur |
| `ApiKey` | Cle API |
| `Webhook` | Webhook sortant |
| `AuditLog` | Log d'audit |

---

## Schema detaille

### Merchant

```prisma
model Merchant {
  id              Int       @id @default(autoincrement())
  merchant_name   String    @db.VarChar(255)
  merchant_email  String?   @db.VarChar(255)
  merchant_phone  String?   @db.VarChar(50)
  merchant_logo   String?   @db.VarChar(500)
  currency        String    @default("XOF") @db.VarChar(10)
  timezone        String    @default("Africa/Lome") @db.VarChar(50)
  status          Int       @default(1)
  settings        Json?
  created_at      DateTime? @default(now())
  updated_at      DateTime? @updatedAt

  // Relations
  users           User[]
  drivers         Driver[]
  bookings        Booking[]
  vehicleTypes    VehicleType[]
  zones           Zone[]
  promoCodes      PromoCode[]
  settings_rel    Setting[]

  @@map("merchants")
}
```

### User

```prisma
model User {
  id              Int       @id @default(autoincrement())
  merchant_id     Int
  first_name      String?   @db.VarChar(100)
  last_name       String?   @db.VarChar(100)
  email           String?   @db.VarChar(255)
  password        String?   @db.VarChar(255)
  phone           String?   @db.VarChar(50)
  country_code    String?   @db.VarChar(10)
  profile_image   String?   @db.VarChar(500)
  status          Int       @default(1)
  device_type     String?   @db.VarChar(20)
  device_token    String?   @db.VarChar(500)
  player_id       String?   @db.VarChar(255)
  fcm_token       String?   @db.VarChar(500)
  language_id     Int       @default(38)
  referral_code   String?   @db.VarChar(20)
  referred_by     Int?
  wallet_balance  Decimal   @default(0) @db.Decimal(12, 2)
  rating          Decimal   @default(5.0) @db.Decimal(3, 2)
  total_rides     Int       @default(0)
  created_at      DateTime? @default(now())
  updated_at      DateTime? @updatedAt

  // Relations
  merchant        Merchant  @relation(fields: [merchant_id], references: [id])
  bookings        Booking[]
  walletTransactions WalletTransaction[]
  paymentMethods  PaymentMethod[]
  ratings         Rating[]
  notifications   Notification[]

  @@index([merchant_id])
  @@index([email])
  @@index([phone])
  @@index([referral_code])
  @@map("users")
}
```

### Driver

```prisma
model Driver {
  id              Int       @id @default(autoincrement())
  merchant_id     Int
  first_name      String?   @db.VarChar(100)
  last_name       String?   @db.VarChar(100)
  email           String?   @db.VarChar(255)
  password        String?   @db.VarChar(255)
  phone           String?   @db.VarChar(50)
  country_code    String?   @db.VarChar(10)
  profile_image   String?   @db.VarChar(500)
  driver_status   Int       @default(0)  // 0=pending, 1=approved, 2=rejected
  is_online       Int       @default(2)  // 1=online, 2=offline
  free_busy       Int       @default(2)  // 1=busy, 2=free
  latitude        Decimal?  @db.Decimal(10, 8)
  longitude       Decimal?  @db.Decimal(11, 8)
  heading         Float?
  device_type     String?   @db.VarChar(20)
  device_token    String?   @db.VarChar(500)
  player_id       String?   @db.VarChar(255)
  fcm_token       String?   @db.VarChar(500)
  language_id     Int       @default(38)
  referral_code   String?   @db.VarChar(20)
  wallet_balance  Decimal   @default(0) @db.Decimal(12, 2)
  rating          Decimal   @default(5.0) @db.Decimal(3, 2)
  total_rides     Int       @default(0)
  total_earnings  Decimal   @default(0) @db.Decimal(12, 2)
  commission_rate Decimal   @default(20) @db.Decimal(5, 2)
  vehicle_type_id Int?
  fleet_id        Int?
  created_at      DateTime? @default(now())
  updated_at      DateTime? @updatedAt

  // Relations
  merchant        Merchant  @relation(fields: [merchant_id], references: [id])
  vehicleType     VehicleType? @relation(fields: [vehicle_type_id], references: [id])
  fleet           Fleet?    @relation(fields: [fleet_id], references: [id])
  bookings        Booking[]
  documents       DriverDocument[]
  earnings        DriverEarning[]
  vehicle         Vehicle?

  @@index([merchant_id])
  @@index([is_online, free_busy])
  @@index([latitude, longitude])
  @@map("drivers")
}
```

### Booking

```prisma
model Booking {
  id                  Int       @id @default(autoincrement())
  merchant_id         Int
  user_id             Int?
  driver_id           Int?
  vehicle_type_id     Int?
  booking_number      String?   @unique @db.VarChar(50)
  booking_type        String    @default("ride") @db.VarChar(20)
  booking_status      String    @default("pending") @db.VarChar(50)
  payment_status      Int       @default(0)
  payment_method      String?   @db.VarChar(50)

  // Pickup
  pickup_latitude     Decimal?  @db.Decimal(10, 8)
  pickup_longitude    Decimal?  @db.Decimal(11, 8)
  pickup_address      String?   @db.VarChar(500)

  // Drop
  drop_latitude       Decimal?  @db.Decimal(10, 8)
  drop_longitude      Decimal?  @db.Decimal(11, 8)
  drop_address        String?   @db.VarChar(500)

  // Pricing
  estimate_amount     Decimal?  @db.Decimal(12, 2)
  final_amount        Decimal?  @db.Decimal(12, 2)
  base_fare           Decimal?  @db.Decimal(12, 2)
  distance_fare       Decimal?  @db.Decimal(12, 2)
  time_fare           Decimal?  @db.Decimal(12, 2)
  surge_multiplier    Decimal   @default(1) @db.Decimal(3, 2)
  discount_amount     Decimal   @default(0) @db.Decimal(12, 2)
  tip_amount          Decimal   @default(0) @db.Decimal(12, 2)
  cancellation_fee    Decimal   @default(0) @db.Decimal(12, 2)

  // Trip details
  travel_distance     Decimal?  @db.Decimal(10, 2)
  travel_time         Int?
  promo_code_id       Int?
  notes               String?   @db.Text

  // Timestamps
  booking_time        DateTime?
  accepted_time       DateTime?
  arrived_time        DateTime?
  started_time        DateTime?
  completed_time      DateTime?
  cancelled_time      DateTime?
  cancelled_by        String?   @db.VarChar(20)
  cancellation_reason String?   @db.VarChar(500)
  created_at          DateTime? @default(now())
  updated_at          DateTime? @updatedAt

  // Relations
  merchant            Merchant  @relation(fields: [merchant_id], references: [id])
  user                User?     @relation(fields: [user_id], references: [id])
  driver              Driver?   @relation(fields: [driver_id], references: [id])
  vehicleType         VehicleType? @relation(fields: [vehicle_type_id], references: [id])
  promoCode           PromoCode? @relation(fields: [promo_code_id], references: [id])
  payment             Payment?
  rating              Rating?

  @@index([merchant_id])
  @@index([user_id])
  @@index([driver_id])
  @@index([booking_status])
  @@index([created_at])
  @@map("bookings")
}
```

### VehicleType

```prisma
model VehicleType {
  id              Int       @id @default(autoincrement())
  merchant_id     Int
  name            String    @db.VarChar(100)
  description     String?   @db.VarChar(500)
  icon            String?   @db.VarChar(500)
  image           String?   @db.VarChar(500)
  capacity        Int       @default(4)
  base_fare       Decimal   @default(0) @db.Decimal(12, 2)
  per_km_fare     Decimal   @default(0) @db.Decimal(12, 2)
  per_minute_fare Decimal   @default(0) @db.Decimal(12, 2)
  minimum_fare    Decimal   @default(0) @db.Decimal(12, 2)
  cancellation_fee Decimal  @default(0) @db.Decimal(12, 2)
  waiting_fee     Decimal   @default(0) @db.Decimal(12, 2)
  is_active       Boolean   @default(true)
  sort_order      Int       @default(0)
  created_at      DateTime? @default(now())
  updated_at      DateTime? @updatedAt

  // Relations
  merchant        Merchant  @relation(fields: [merchant_id], references: [id])
  drivers         Driver[]
  bookings        Booking[]

  @@index([merchant_id])
  @@map("vehicle_types")
}
```

### Zone

```prisma
model Zone {
  id              Int       @id @default(autoincrement())
  merchant_id     Int
  name            String    @db.VarChar(100)
  description     String?   @db.VarChar(500)
  type            String    @default("service_area") @db.VarChar(50)
  shape           String    @default("polygon") @db.VarChar(20)
  coordinates     Json?     // GeoJSON coordinates
  center_lat      Decimal?  @db.Decimal(10, 8)
  center_lng      Decimal?  @db.Decimal(11, 8)
  radius          Decimal?  @db.Decimal(10, 2)
  surge_multiplier Decimal  @default(1) @db.Decimal(3, 2)
  extra_fee       Decimal   @default(0) @db.Decimal(12, 2)
  is_active       Boolean   @default(true)
  created_at      DateTime? @default(now())
  updated_at      DateTime? @updatedAt

  // Relations
  merchant        Merchant  @relation(fields: [merchant_id], references: [id])

  @@index([merchant_id])
  @@index([type])
  @@map("zones")
}
```

### PromoCode

```prisma
model PromoCode {
  id              Int       @id @default(autoincrement())
  merchant_id     Int
  code            String    @db.VarChar(50)
  name            String?   @db.VarChar(100)
  description     String?   @db.VarChar(500)
  type            String    @default("percentage") @db.VarChar(20)
  discount_value  Decimal   @db.Decimal(12, 2)
  max_discount    Decimal?  @db.Decimal(12, 2)
  min_order_value Decimal?  @db.Decimal(12, 2)
  usage_limit     Int?
  usage_per_user  Int       @default(1)
  used_count      Int       @default(0)
  valid_from      DateTime?
  valid_until     DateTime?
  is_active       Boolean   @default(true)
  created_at      DateTime? @default(now())
  updated_at      DateTime? @updatedAt

  // Relations
  merchant        Merchant  @relation(fields: [merchant_id], references: [id])
  bookings        Booking[]
  usages          PromoUsage[]

  @@unique([merchant_id, code])
  @@index([code])
  @@map("promo_codes")
}
```

---

## Index et performance

### Index recommandes

```sql
-- Recherche de chauffeurs disponibles
CREATE INDEX idx_drivers_available ON drivers(merchant_id, is_online, free_busy, latitude, longitude);

-- Historique des courses
CREATE INDEX idx_bookings_history ON bookings(user_id, created_at DESC);
CREATE INDEX idx_bookings_driver ON bookings(driver_id, created_at DESC);

-- Recherche par status
CREATE INDEX idx_bookings_status ON bookings(merchant_id, booking_status);

-- Transactions wallet
CREATE INDEX idx_wallet_tx ON wallet_transactions(user_id, created_at DESC);
```

### Partitionnement (pour gros volumes)

```sql
-- Partitionner par mois pour les bookings
ALTER TABLE bookings PARTITION BY RANGE (YEAR(created_at) * 100 + MONTH(created_at)) (
  PARTITION p202401 VALUES LESS THAN (202402),
  PARTITION p202402 VALUES LESS THAN (202403),
  -- ...
);
```

---

## Migrations

### Creer une migration

```bash
pnpm prisma migrate dev --name add_new_field
```

### Appliquer en production

```bash
pnpm prisma migrate deploy
```

### Reset (dev uniquement)

```bash
pnpm prisma migrate reset
```

---

## Bonnes pratiques

1. **Toujours filtrer par merchant_id** pour le multi-tenant
2. **Utiliser les index** pour les requetes frequentes
3. **Eviter les N+1** avec `include` de Prisma
4. **Soft delete** plutot que hard delete
5. **Timestamps** sur toutes les tables

```typescript
// Bon - avec include
const booking = await prisma.booking.findUnique({
  where: { id },
  include: {
    user: true,
    driver: true,
    vehicleType: true,
  },
});

// Mauvais - N+1
const booking = await prisma.booking.findUnique({ where: { id } });
const user = await prisma.user.findUnique({ where: { id: booking.user_id } });
const driver = await prisma.driver.findUnique({ where: { id: booking.driver_id } });
```
